﻿# ---------------------------------------------------------------------- #
# Target DBMS:           MySQL 5                                         #
# Project name:          Northwind                                       #
# Author:                Valon Hoti                                      #
# Created on:            2010-07-07 20:00                                #
# ---------------------------------------------------------------------- #
DROP DATABASE IF EXISTS northwind;

CREATE DATABASE IF NOT EXISTS northwind;

USE northwind;

# ---------------------------------------------------------------------- #
# Tables                                                                 #
# ---------------------------------------------------------------------- #
# ---------------------------------------------------------------------- #
# Add table "Categories"                                                 #
# ---------------------------------------------------------------------- #

CREATE TABLE `Categories` (
    `CategoryID` INTEGER NOT NULL AUTO_INCREMENT,
    `CategoryName` VARCHAR(15) NOT NULL,
    `Description` MEDIUMTEXT,
    `Picture` LONGBLOB,
    CONSTRAINT `PK_Categories` PRIMARY KEY (`CategoryID`)
);

CREATE INDEX `CategoryName` ON `Categories` (`CategoryName`);

# ---------------------------------------------------------------------- #
# Add table "CustomerCustomerDemo"                                       #
# ---------------------------------------------------------------------- #

CREATE TABLE `CustomerCustomerDemo` (
    `CustomerID` VARCHAR(5) NOT NULL,
    `CustomerTypeID` VARCHAR(10) NOT NULL,
    CONSTRAINT `PK_CustomerCustomerDemo` PRIMARY KEY (`CustomerID`, `CustomerTypeID`)
);

# ---------------------------------------------------------------------- #
# Add table "CustomerDemographics"                                       #
# ---------------------------------------------------------------------- #

CREATE TABLE `CustomerDemographics` (
    `CustomerTypeID` VARCHAR(10) NOT NULL,
    `CustomerDesc` MEDIUMTEXT,
    CONSTRAINT `PK_CustomerDemographics` PRIMARY KEY (`CustomerTypeID`)
);

# ---------------------------------------------------------------------- #
# Add table "Customers"                                                  #
# ---------------------------------------------------------------------- #

CREATE TABLE `Customers` (
    `CustomerID` VARCHAR(5) NOT NULL,
    `CompanyName` VARCHAR(40) NOT NULL,
    `ContactName` VARCHAR(30),
    `ContactTitle` VARCHAR(30),
    `Address` VARCHAR(60),
    `City` VARCHAR(15),
    `Region` VARCHAR(15),
    `PostalCode` VARCHAR(10),
    `Country` VARCHAR(15),
    `Phone` VARCHAR(24),
    `Fax` VARCHAR(24),
    CONSTRAINT `PK_Customers` PRIMARY KEY (`CustomerID`)
);

CREATE INDEX `City` ON `Customers` (`City`);

CREATE INDEX `CompanyName` ON `Customers` (`CompanyName`);

CREATE INDEX `PostalCode` ON `Customers` (`PostalCode`);

CREATE INDEX `Region` ON `Customers` (`Region`);

# ---------------------------------------------------------------------- #
# Add table "Employees"                                                  #
# ---------------------------------------------------------------------- #

CREATE TABLE `Employees` (
    `EmployeeID` INTEGER NOT NULL AUTO_INCREMENT,
    `LastName` VARCHAR(20) NOT NULL,
    `FirstName` VARCHAR(10) NOT NULL,
    `Title` VARCHAR(30),
    `TitleOfCourtesy` VARCHAR(25),
    `BirthDate` DATETIME,
    `HireDate` DATETIME,
    `Address` VARCHAR(60),
    `City` VARCHAR(15),
    `Region` VARCHAR(15),
    `PostalCode` VARCHAR(10),
    `Country` VARCHAR(15),
    `HomePhone` VARCHAR(24),
    `Extension` VARCHAR(4),
    `Photo` LONGBLOB,
    `Notes` MEDIUMTEXT NOT NULL,
    `ReportsTo` INTEGER,
    `PhotoPath` VARCHAR(255),
     `Salary` FLOAT,
    CONSTRAINT `PK_Employees` PRIMARY KEY (`EmployeeID`)
);

CREATE INDEX `LastName` ON `Employees` (`LastName`);

CREATE INDEX `PostalCode` ON `Employees` (`PostalCode`);

# ---------------------------------------------------------------------- #
# Add table "EmployeeTerritories"                                        #
# ---------------------------------------------------------------------- #

CREATE TABLE `EmployeeTerritories` (
    `EmployeeID` INTEGER NOT NULL,
    `TerritoryID` VARCHAR(20) NOT NULL,
    CONSTRAINT `PK_EmployeeTerritories` PRIMARY KEY (`EmployeeID`, `TerritoryID`)
);

# ---------------------------------------------------------------------- #
# Add table "Order Details"                                              #
# ---------------------------------------------------------------------- #

CREATE TABLE `Order Details` (
    `OrderID` INTEGER NOT NULL,
    `ProductID` INTEGER NOT NULL,
    `UnitPrice` DECIMAL(10,4) NOT NULL DEFAULT 0,
    `Quantity` SMALLINT(2) NOT NULL DEFAULT 1,
    `Discount` REAL(8,0) NOT NULL DEFAULT 0,
    CONSTRAINT `PK_Order Details` PRIMARY KEY (`OrderID`, `ProductID`)
);

# ---------------------------------------------------------------------- #
# Add table "Orders"                                                     #
# ---------------------------------------------------------------------- #

CREATE TABLE `Orders` (
    `OrderID` INTEGER NOT NULL AUTO_INCREMENT,
    `CustomerID` VARCHAR(5),
    `EmployeeID` INTEGER,
    `OrderDate` DATETIME,
    `RequiredDate` DATETIME,
    `ShippedDate` DATETIME,
    `ShipVia` INTEGER,
    `Freight` DECIMAL(10,4) DEFAULT 0,
    `ShipName` VARCHAR(40),
    `ShipAddress` VARCHAR(60),
    `ShipCity` VARCHAR(15),
    `ShipRegion` VARCHAR(15),
    `ShipPostalCode` VARCHAR(10),
    `ShipCountry` VARCHAR(15),
    CONSTRAINT `PK_Orders` PRIMARY KEY (`OrderID`)
);

CREATE INDEX `OrderDate` ON `Orders` (`OrderDate`);

CREATE INDEX `ShippedDate` ON `Orders` (`ShippedDate`);

CREATE INDEX `ShipPostalCode` ON `Orders` (`ShipPostalCode`);

# ---------------------------------------------------------------------- #
# Add table "Products"                                                   #
# ---------------------------------------------------------------------- #

CREATE TABLE `Products` (
    `ProductID` INTEGER NOT NULL AUTO_INCREMENT,
    `ProductName` VARCHAR(40) NOT NULL,
    `SupplierID` INTEGER,
    `CategoryID` INTEGER,
    `QuantityPerUnit` VARCHAR(20),
    `UnitPrice` DECIMAL(10,4) DEFAULT 0,
    `UnitsInStock` SMALLINT(2) DEFAULT 0,
    `UnitsOnOrder` SMALLINT(2) DEFAULT 0,
    `ReorderLevel` SMALLINT(2) DEFAULT 0,
    `Discontinued` BIT NOT NULL DEFAULT 0,
    CONSTRAINT `PK_Products` PRIMARY KEY (`ProductID`)
);

CREATE INDEX `ProductName` ON `Products` (`ProductName`);

# ---------------------------------------------------------------------- #
# Add table "Region"                                                     #
# ---------------------------------------------------------------------- #

CREATE TABLE `Region` (
    `RegionID` INTEGER NOT NULL,
    `RegionDescription` VARCHAR(50) NOT NULL,
    CONSTRAINT `PK_Region` PRIMARY KEY (`RegionID`)
);

# ---------------------------------------------------------------------- #
# Add table "Shippers"                                                   #
# ---------------------------------------------------------------------- #

CREATE TABLE `Shippers` (
    `ShipperID` INTEGER NOT NULL AUTO_INCREMENT,
    `CompanyName` VARCHAR(40) NOT NULL,
    `Phone` VARCHAR(24),
    CONSTRAINT `PK_Shippers` PRIMARY KEY (`ShipperID`)
);

# ---------------------------------------------------------------------- #
# Add table "Suppliers"                                                  #
# ---------------------------------------------------------------------- #

CREATE TABLE `Suppliers` (
    `SupplierID` INTEGER NOT NULL AUTO_INCREMENT,
    `CompanyName` VARCHAR(40) NOT NULL,
    `ContactName` VARCHAR(30),
    `ContactTitle` VARCHAR(30),
    `Address` VARCHAR(60),
    `City` VARCHAR(15),
    `Region` VARCHAR(15),
    `PostalCode` VARCHAR(10),
    `Country` VARCHAR(15),
    `Phone` VARCHAR(24),
    `Fax` VARCHAR(24),
    `HomePage` MEDIUMTEXT,
    CONSTRAINT `PK_Suppliers` PRIMARY KEY (`SupplierID`)
);

CREATE INDEX `CompanyName` ON `Suppliers` (`CompanyName`);

CREATE INDEX `PostalCode` ON `Suppliers` (`PostalCode`);

# ---------------------------------------------------------------------- #
# Add table "Territories"                                                #
# ---------------------------------------------------------------------- #

CREATE TABLE `Territories` (
    `TerritoryID` VARCHAR(20) NOT NULL,
    `TerritoryDescription` VARCHAR(50) NOT NULL,
    `RegionID` INTEGER NOT NULL,
    CONSTRAINT `PK_Territories` PRIMARY KEY (`TerritoryID`)
);

# ---------------------------------------------------------------------- #
# Add info into "Categories"                                             #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Categories;
INSERT INTO Categories VALUES(null,'Beverages','Soft drinks, coffees, teas, beers, and ales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
INSERT INTO Categories VALUES(null,'Condiments','Sweet and savory sauces, relishes, spreads, and seasonings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
INSERT INTO Categories VALUES(null,'Confections','Desserts, candies, and sweet breads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
INSERT INTO Categories VALUES(null,'Dairy Products','Cheeses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
INSERT INTO Categories VALUES(null,'Grains/Cereals','Breads, crackers, pasta, and cereal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
INSERT INTO Categories VALUES(null,'Meat/Poultry','Prepared meats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
INSERT INTO Categories VALUES(null,'Produce','Dried fruit and bean curd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
INSERT INTO Categories VALUES(null,'Seafood','Seaweed and fish',X'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');

# ---------------------------------------------------------------------- #
# Add info into "Customers"                                              #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Customers;
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ALFKI', 'Alfreds Futterkiste', 'Maria Anders', 'Sales Representative', 'Obere Str. 57', 'Berlin', NULL, '12209', 'Germany', '030-0074321', '030-0076545');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ANATR', 'Ana Trujillo Emparedados y helados', 'Ana Trujillo', 'Owner', 'Avda. de la Constitucin 2222', 'Mxico D.F.', NULL, '05021', 'Mexico', '(5) 555-4729', '(5) 555-3745');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ANTON', 'Antonio Moreno Taquera', 'Antonio Moreno', 'Owner', 'Mataderos  2312', 'Mxico D.F.', NULL, '05023', 'Mexico', '(5) 555-3932', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('AROUT', 'Around the Horn', 'Thomas Hardy', 'Sales Representative', '120 Hanover Sq.', 'London', NULL, 'WA1 1DP', 'UK', '(171) 555-7788', '(171) 555-6750');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BERGS', 'Berglunds snabbkp', 'Christina Berglund', 'Order Administrator', 'Berguvsvgen  8', 'Lule', NULL, 'S-958 22', 'Sweden', '0921-12 34 65', '0921-12 34 67');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BLAUS', 'Blauer See Delikatessen', 'Hanna Moos', 'Sales Representative', 'Forsterstr. 57', 'Mannheim', NULL, '68306', 'Germany', '0621-08460', '0621-08924');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BLONP', 'Blondesddsl pre et fils', 'Frdrique Citeaux', 'Marketing Manager', '24, place Klber', 'Strasbourg', NULL, '67000', 'France', '88.60.15.31', '88.60.15.32');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BOLID', 'Blido Comidas preparadas', 'Martn Sommer', 'Owner', 'C/ Araquil, 67', 'Madrid', NULL, '28023', 'Spain', '(91) 555 22 82', '(91) 555 91 99');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BONAP', 'Bon app''', 'Laurence Lebihan', 'Owner', '12, rue des Bouchers', 'Marseille', NULL, '13008', 'France', '91.24.45.40', '91.24.45.41');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BOTTM', 'Bottom-Dollar Markets', 'Elizabeth Lincoln', 'Accounting Manager', '23 Tsawassen Blvd.', 'Tsawassen', 'BC', 'T2F 8M4', 'Canada', '(604) 555-4729', '(604) 555-3745');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('BSBEV', 'B''s Beverages', 'Victoria Ashworth', 'Sales Representative', 'Fauntleroy Circus', 'London', NULL, 'EC2 5NT', 'UK', '(171) 555-1212', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('CACTU', 'Cactus Comidas para llevar', 'Patricio Simpson', 'Sales Agent', 'Cerrito 333', 'Buenos Aires', NULL, '1010', 'Argentina', '(1) 135-5555', '(1) 135-4892');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('CENTC', 'Centro comercial Moctezuma', 'Francisco Chang', 'Marketing Manager', 'Sierras de Granada 9993', 'Mxico D.F.', NULL, '05022', 'Mexico', '(5) 555-3392', '(5) 555-7293');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('CHOPS', 'Chop-suey Chinese', 'Yang Wang', 'Owner', 'Hauptstr. 29', 'Bern', NULL, '3012', 'Switzerland', '0452-076545', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('COMMI', 'Comrcio Mineiro', 'Pedro Afonso', 'Sales Associate', 'Av. dos Lusadas, 23', 'Sao Paulo', 'SP', '05432-043', 'Brazil', '(11) 555-7647', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('CONSH', 'Consolidated Holdings', 'Elizabeth Brown', 'Sales Representative', 'Berkeley Gardens 12  Brewery', 'London', NULL, 'WX1 6LT', 'UK', '(171) 555-2282', '(171) 555-9199');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('DRACD', 'Drachenblut Delikatessen', 'Sven Ottlieb', 'Order Administrator', 'Walserweg 21', 'Aachen', NULL, '52066', 'Germany', '0241-039123', '0241-059428');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('DUMON', 'Du monde entier', 'Janine Labrune', 'Owner', '67, rue des Cinquante Otages', 'Nantes', NULL, '44000', 'France', '40.67.88.88', '40.67.89.89');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('EASTC', 'Eastern Connection', 'Ann Devon', 'Sales Agent', '35 King George', 'London', NULL, 'WX3 6FW', 'UK', '(171) 555-0297', '(171) 555-3373');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ERNSH', 'Ernst Handel', 'Roland Mendel', 'Sales Manager', 'Kirchgasse 6', 'Graz', NULL, '8010', 'Austria', '7675-3425', '7675-3426');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FAMIA', 'Familia Arquibaldo', 'Aria Cruz', 'Marketing Assistant', 'Rua Ors, 92', 'Sao Paulo', 'SP', '05442-030', 'Brazil', '(11) 555-9857', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FISSA', 'FISSA Fabrica Inter. Salchichas S.A.', 'Diego Roel', 'Accounting Manager', 'C/ Moralzarzal, 86', 'Madrid', NULL, '28034', 'Spain', '(91) 555 94 44', '(91) 555 55 93');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FOLIG', 'Folies gourmandes', 'Martine Ranc', 'Assistant Sales Agent', '184, chausse de Tournai', 'Lille', NULL, '59000', 'France', '20.16.10.16', '20.16.10.17');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FOLKO', 'Folk och f HB', 'Maria Larsson', 'Owner', 'kergatan 24', 'Brcke', NULL, 'S-844 67', 'Sweden', '0695-34 67 21', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FRANK', 'Frankenversand', 'Peter Franken', 'Marketing Manager', 'Berliner Platz 43', 'Mnchen', NULL, '80805', 'Germany', '089-0877310', '089-0877451');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FRANR', 'France restauration', 'Carine Schmitt', 'Marketing Manager', '54, rue Royale', 'Nantes', NULL, '44000', 'France', '40.32.21.21', '40.32.21.20');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FRANS', 'Franchi S.p.A.', 'Paolo Accorti', 'Sales Representative', 'Via Monte Bianco 34', 'Torino', NULL, '10100', 'Italy', '011-4988260', '011-4988261');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('FURIB', 'Furia Bacalhau e Frutos do Mar', 'Lino Rodriguez', 'Sales Manager', 'Jardim das rosas n. 32', 'Lisboa', NULL, '1675', 'Portugal', '(1) 354-2534', '(1) 354-2535');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('GALED', 'Galera del gastrnomo', 'Eduardo Saavedra', 'Marketing Manager', 'Rambla de Catalua, 23', 'Barcelona', NULL, '08022', 'Spain', '(93) 203 4560', '(93) 203 4561');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('GODOS', 'Godos Cocina Tpica', 'Jos Pedro Freyre', 'Sales Manager', 'C/ Romero, 33', 'Sevilla', NULL, '41101', 'Spain', '(95) 555 82 82', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('GOURL', 'Gourmet Lanchonetes', 'Andr Fonseca', 'Sales Associate', 'Av. Brasil, 442', 'Campinas', 'SP', '04876-786', 'Brazil', '(11) 555-9482', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('GREAL', 'Great Lakes Food Market', 'Howard Snyder', 'Marketing Manager', '2732 Baker Blvd.', 'Eugene', 'OR', '97403', 'USA', '(503) 555-7555', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('GROSR', 'GROSELLA-Restaurante', 'Manuel Pereira', 'Owner', '5 Ave. Los Palos Grandes', 'Caracas', 'DF', '1081', 'Venezuela', '(2) 283-2951', '(2) 283-3397');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('HANAR', 'Hanari Carnes', 'Mario Pontes', 'Accounting Manager', 'Rua do Pao, 67', 'Rio de Janeiro', 'RJ', '05454-876', 'Brazil', '(21) 555-0091', '(21) 555-8765');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('HILAA', 'HILARION-Abastos', 'Carlos Hernndez', 'Sales Representative', 'Carrera 22 con Ave. Carlos Soublette #8-35', 'San Cristbal', 'Tchira', '5022', 'Venezuela', '(5) 555-1340', '(5) 555-1948');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('HUNGC', 'Hungry Coyote Import Store', 'Yoshi Latimer', 'Sales Representative', 'City Center Plaza 516 Main St.', 'Elgin', 'OR', '97827', 'USA', '(503) 555-6874', '(503) 555-2376');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('HUNGO', 'Hungry Owl All-Night Grocers', 'Patricia McKenna', 'Sales Associate', '8 Johnstown Road', 'Cork', 'Co. Cork', NULL, 'Ireland', '2967 542', '2967 3333');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ISLAT', 'Island Trading', 'Helen Bennett', 'Marketing Manager', 'Garden House Crowther Way', 'Cowes', 'Isle of Wight', 'PO31 7PJ', 'UK', '(198) 555-8888', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('KOENE', 'Kniglich Essen', 'Philip Cramer', 'Sales Associate', 'Maubelstr. 90', 'Brandenburg', NULL, '14776', 'Germany', '0555-09876', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LACOR', 'La corne d''abondance', 'Daniel Tonini', 'Sales Representative', '67, avenue de l''Europe', 'Versailles', NULL, '78000', 'France', '30.59.84.10', '30.59.85.11');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LAMAI', 'La maison d''Asie', 'Annette Roulet', 'Sales Manager', '1 rue Alsace-Lorraine', 'Toulouse', NULL, '31000', 'France', '61.77.61.10', '61.77.61.11');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LAUGB', 'Laughing Bacchus Wine Cellars', 'Yoshi Tannamuri', 'Marketing Assistant', '1900 Oak St.', 'Vancouver', 'BC', 'V3F 2K1', 'Canada', '(604) 555-3392', '(604) 555-7293');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LAZYK', 'Lazy K Kountry Store', 'John Steel', 'Marketing Manager', '12 Orchestra Terrace', 'Walla Walla', 'WA', '99362', 'USA', '(509) 555-7969', '(509) 555-6221');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LEHMS', 'Lehmanns Marktstand', 'Renate Messner', 'Sales Representative', 'Magazinweg 7', 'Frankfurt a.M.', NULL, '60528', 'Germany', '069-0245984', '069-0245874');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LETSS', 'Let''s Stop N Shop', 'Jaime Yorres', 'Owner', '87 Polk St. Suite 5', 'San Francisco', 'CA', '94117', 'USA', '(415) 555-5938', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LILAS', 'LILA-Supermercado', 'Carlos Gonzlez', 'Accounting Manager', 'Carrera 52 con Ave. Bolvar #65-98 Llano Largo', 'Barquisimeto', 'Lara', '3508', 'Venezuela', '(9) 331-6954', '(9) 331-7256');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LINOD', 'LINO-Delicateses', 'Felipe Izquierdo', 'Owner', 'Ave. 5 de Mayo Porlamar', 'I. de Margarita', 'Nueva Esparta', '4980', 'Venezuela', '(8) 34-56-12', '(8) 34-93-93');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('LONEP', 'Lonesome Pine Restaurant', 'Fran Wilson', 'Sales Manager', '89 Chiaroscuro Rd.', 'Portland', 'OR', '97219', 'USA', '(503) 555-9573', '(503) 555-9646');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('MAGAA', 'Magazzini Alimentari Riuniti', 'Giovanni Rovelli', 'Marketing Manager', 'Via Ludovico il Moro 22', 'Bergamo', NULL, '24100', 'Italy', '035-640230', '035-640231');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('MAISD', 'Maison Dewey', 'Catherine Dewey', 'Sales Agent', 'Rue Joseph-Bens 532', 'Bruxelles', NULL, 'B-1180', 'Belgium', '(02) 201 24 67', '(02) 201 24 68');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('MEREP', 'Mre Paillarde', 'Jean Fresnire', 'Marketing Assistant', '43 rue St. Laurent', 'Montral', 'Qubec', 'H1J 1C3', 'Canada', '(514) 555-8054', '(514) 555-8055');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('MORGK', 'Morgenstern Gesundkost', 'Alexander Feuer', 'Marketing Assistant', 'Heerstr. 22', 'Leipzig', NULL, '04179', 'Germany', '0342-023176', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('NORTS', 'North/South', 'Simon Crowther', 'Sales Associate', 'South House 300 Queensbridge', 'London', NULL, 'SW7 1RZ', 'UK', '(171) 555-7733', '(171) 555-2530');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('OCEAN', 'Ocano Atlntico Ltda.', 'Yvonne Moncada', 'Sales Agent', 'Ing. Gustavo Moncada 8585 Piso 20-A', 'Buenos Aires', NULL, '1010', 'Argentina', '(1) 135-5333', '(1) 135-5535');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('OLDWO', 'Old World Delicatessen', 'Rene Phillips', 'Sales Representative', '2743 Bering St.', 'Anchorage', 'AK', '99508', 'USA', '(907) 555-7584', '(907) 555-2880');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('OTTIK', 'Ottilies Kseladen', 'Henriette Pfalzheim', 'Owner', 'Mehrheimerstr. 369', 'Kln', NULL, '50739', 'Germany', '0221-0644327', '0221-0765721');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('PARIS', 'Paris spcialits', 'Marie Bertrand', 'Owner', '265, boulevard Charonne', 'Paris', NULL, '75012', 'France', '(1) 42.34.22.66', '(1) 42.34.22.77');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('PERIC', 'Pericles Comidas clsicas', 'Guillermo Fernndez', 'Sales Representative', 'Calle Dr. Jorge Cash 321', 'Mxico D.F.', NULL, '05033', 'Mexico', '(5) 552-3745', '(5) 545-3745');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('PICCO', 'Piccolo und mehr', 'Georg Pipps', 'Sales Manager', 'Geislweg 14', 'Salzburg', NULL, '5020', 'Austria', '6562-9722', '6562-9723');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('PRINI', 'Princesa Isabel Vinhos', 'Isabel de Castro', 'Sales Representative', 'Estrada da sade n. 58', 'Lisboa', NULL, '1756', 'Portugal', '(1) 356-5634', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('QUEDE', 'Que Delcia', 'Bernardo Batista', 'Accounting Manager', 'Rua da Panificadora, 12', 'Rio de Janeiro', 'RJ', '02389-673', 'Brazil', '(21) 555-4252', '(21) 555-4545');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('QUEEN', 'Queen Cozinha', 'Lcia Carvalho', 'Marketing Assistant', 'Alameda dos Canrios, 891', 'Sao Paulo', 'SP', '05487-020', 'Brazil', '(11) 555-1189', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('QUICK', 'QUICK-Stop', 'Horst Kloss', 'Accounting Manager', 'Taucherstrae 10', 'Cunewalde', NULL, '01307', 'Germany', '0372-035188', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('RANCH', 'Rancho grande', 'Sergio Gutirrez', 'Sales Representative', 'Av. del Libertador 900', 'Buenos Aires', NULL, '1010', 'Argentina', '(1) 123-5555', '(1) 123-5556');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('RATTC', 'Rattlesnake Canyon Grocery', 'Paula Wilson', 'Assistant Sales Representative', '2817 Milton Dr.', 'Albuquerque', 'NM', '87110', 'USA', '(505) 555-5939', '(505) 555-3620');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('REGGC', 'Reggiani Caseifici', 'Maurizio Moroni', 'Sales Associate', 'Strada Provinciale 124', 'Reggio Emilia', NULL, '42100', 'Italy', '0522-556721', '0522-556722');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('RICAR', 'Ricardo Adocicados', 'Janete Limeira', 'Assistant Sales Agent', 'Av. Copacabana, 267', 'Rio de Janeiro', 'RJ', '02389-890', 'Brazil', '(21) 555-3412', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('RICSU', 'Richter Supermarkt', 'Michael Holz', 'Sales Manager', 'Grenzacherweg 237', 'Genve', NULL, '1203', 'Switzerland', '0897-034214', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('ROMEY', 'Romero y tomillo', 'Alejandra Camino', 'Accounting Manager', 'Gran Va, 1', 'Madrid', NULL, '28001', 'Spain', '(91) 745 6200', '(91) 745 6210');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SANTG', 'Sant Gourmet', 'Jonas Bergulfsen', 'Owner', 'Erling Skakkes gate 78', 'Stavern', NULL, '4110', 'Norway', '07-98 92 35', '07-98 92 47');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SAVEA', 'Save-a-lot Markets', 'Jose Pavarotti', 'Sales Representative', '187 Suffolk Ln.', 'Boise', 'ID', '83720', 'USA', '(208) 555-8097', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SEVES', 'Seven Seas Imports', 'Hari Kumar', 'Sales Manager', '90 Wadhurst Rd.', 'London', NULL, 'OX15 4NB', 'UK', '(171) 555-1717', '(171) 555-5646');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SIMOB', 'Simons bistro', 'Jytte Petersen', 'Owner', 'Vinbltet 34', 'Kobenhavn', NULL, '1734', 'Denmark', '31 12 34 56', '31 13 35 57');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SPECD', 'Spcialits du monde', 'Dominique Perrier', 'Marketing Manager', '25, rue Lauriston', 'Paris', NULL, '75016', 'France', '(1) 47.55.60.10', '(1) 47.55.60.20');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SPLIR', 'Split Rail Beer & Ale', 'Art Braunschweiger', 'Sales Manager', 'P.O. Box 555', 'Lander', 'WY', '82520', 'USA', '(307) 555-4680', '(307) 555-6525');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('SUPRD', 'Suprmes dlices', 'Pascale Cartrain', 'Accounting Manager', 'Boulevard Tirou, 255', 'Charleroi', NULL, 'B-6000', 'Belgium', '(071) 23 67 22 20', '(071) 23 67 22 21');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('THEBI', 'The Big Cheese', 'Liz Nixon', 'Marketing Manager', '89 Jefferson Way Suite 2', 'Portland', 'OR', '97201', 'USA', '(503) 555-3612', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('THECR', 'The Cracker Box', 'Liu Wong', 'Marketing Assistant', '55 Grizzly Peak Rd.', 'Butte', 'MT', '59801', 'USA', '(406) 555-5834', '(406) 555-8083');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('TOMSP', 'Toms Spezialitten', 'Karin Josephs', 'Marketing Manager', 'Luisenstr. 48', 'Mnster', NULL, '44087', 'Germany', '0251-031259', '0251-035695');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('TORTU', 'Tortuga Restaurante', 'Miguel Angel Paolino', 'Owner', 'Avda. Azteca 123', 'Mxico D.F.', NULL, '05033', 'Mexico', '(5) 555-2933', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('TRADH', 'Tradio Hipermercados', 'Anabela Domingues', 'Sales Representative', 'Av. Ins de Castro, 414', 'Sao Paulo', 'SP', '05634-030', 'Brazil', '(11) 555-2167', '(11) 555-2168');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('TRAIH', 'Trail''s Head Gourmet Provisioners', 'Helvetius Nagy', 'Sales Associate', '722 DaVinci Blvd.', 'Kirkland', 'WA', '98034', 'USA', '(206) 555-8257', '(206) 555-2174');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('VAFFE', 'Vaffeljernet', 'Palle Ibsen', 'Sales Manager', 'Smagsloget 45', 'rhus', NULL, '8200', 'Denmark', '86 21 32 43', '86 22 33 44');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('Val2 ', 'IT', 'Val2', 'IT', NULL, NULL, NULL, NULL, NULL, NULL, NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('VALON', 'IT', 'Valon Hoti', 'IT', NULL, NULL, NULL, NULL, NULL, NULL, NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('VICTE', 'Victuailles en stock', 'Mary Saveley', 'Sales Agent', '2, rue du Commerce', 'Lyon', NULL, '69004', 'France', '78.32.54.86', '78.32.54.87');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('VINET', 'Vins et alcools Chevalier', 'Paul Henriot', 'Accounting Manager', '59 rue de l''Abbaye', 'Reims', NULL, '51100', 'France', '26.47.15.10', '26.47.15.11');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WANDK', 'Die Wandernde Kuh', 'Rita Mller', 'Sales Representative', 'Adenauerallee 900', 'Stuttgart', NULL, '70563', 'Germany', '0711-020361', '0711-035428');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WARTH', 'Wartian Herkku', 'Pirkko Koskitalo', 'Accounting Manager', 'Torikatu 38', 'Oulu', NULL, '90110', 'Finland', '981-443655', '981-443655');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WELLI', 'Wellington Importadora', 'Paula Parente', 'Sales Manager', 'Rua do Mercado, 12', 'Resende', 'SP', '08737-363', 'Brazil', '(14) 555-8122', NULL);
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WHITC', 'White Clover Markets', 'Karl Jablonski', 'Owner', '305 - 14th Ave. S. Suite 3B', 'Seattle', 'WA', '98128', 'USA', '(206) 555-4112', '(206) 555-4115');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WILMK', 'Wilman Kala', 'Matti Karttunen', 'Owner/Marketing Assistant', 'Keskuskatu 45', 'Helsinki', NULL, '21240', 'Finland', '90-224 8858', '90-224 8858');
INSERT INTO Customers (CustomerID, CompanyName, ContactName, ContactTitle, Address, City, Region, PostalCode, Country, Phone, Fax)
VALUES('WOLZA', 'Wolski  Zajazd', 'Zbyszek Piestrzeniewicz', 'Owner', 'ul. Filtrowa 68', 'Warszawa', NULL, '01-012', 'Poland', '(26) 642-7012', '(26) 642-7012');

# ---------------------------------------------------------------------- #
# Add info into "Employees"                                              #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Employees;
INSERT INTO Employees VALUES(null,'Davolio','Nancy','Sales Representative','Ms.','1948-12-08','1992-05-01','507 - 20th Ave. E.Apt. 2A','Seattle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ducation includes a BA in psychology from Colorado State University in 1970.  She also completed "The Art of the Cold Call."  Nancy is a member of Toastmasters International.',2
,'http://accweb/emmployees/davolio.bmp','2954.55');
INSERT INTO Employees VALUES(null,'Fuller','Andrew','Vice President, Sales','Dr.','1952-02-19','1992-08-14','908 W. Capital Way','Tacoma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ndrew received his BTS commercial in 1974 and a Ph.D. in international marketing from the University of Dallas in 1981.  He is fluent in French and Italian and reads German.  He joined the company as a sales representative, was promoted to sales manager in January 1992 and to vice president of sales in March 1993.  Andrew is a member of the Sales Management Roundtable, the Seattle Chamber of Commerce, and the Pacific Rim Importers Association.',NULL
,'http://accweb/emmployees/fuller.bmp','2254.49');
INSERT INTO Employees VALUES(null,'Leverling','Janet','Sales Representative','Ms.','1963-08-30','1992-04-01','722 Moss Bay Blvd.','Kirkland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anet has a BS degree in chemistry from Boston College (1984).  She has also completed a certificate program in food retailing management.  Janet was hired as a sales associate in 1991 and promoted to sales representative in February 1992.',2
,'http://accweb/emmployees/leverling.bmp','3119.15');
INSERT INTO Employees VALUES(null,'Peacock','Margaret','Sales Representative','Mrs.','1937-09-19','1993-05-03','4110 Old Redmond Rd.','Redmond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argaret holds a BA in English literature from Concordia College (1958) and an MA from the American Institute of Culinary Arts (1966).  She was assigned to the London office temporarily from July through November 1992.',2
,'http://accweb/emmployees/peacock.bmp','1861.08');
INSERT INTO Employees VALUES(null,'Buchanan','Steven','Sales Manager','Mr.','1955-03-04','1993-10-17','14 Garrett Hill','London',NULL,'SW1 8JR','UK','(71) 555-4848','3453',X'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','Steven Buchanan graduated from St. Andrews University, Scotland, with a BSC degree in 1976.  Upon joining the company as a sales representative in 1992, he spent 6 months in an orientation program at the Seattle office and then returned to his permanent post in London.  He was promoted to sales manager in March 1993.  Mr. Buchanan has completed the courses "Successful Telemarketing" and "International Sales Management."  He is fluent in French.',2
,'http://accweb/emmployees/buchanan.bmp','1744.21');
INSERT INTO Employees VALUES(null,'Suyama','Michael','Sales Representative','Mr.','1963-07-02','1993-10-17','Coventry House
Miner Rd.','London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ichael is a graduate of Sussex University (MA, economics, 1983) and the University of California at Los Angeles (MBA, marketing, 1986).  He has also taken the courses "Multi-Cultural Selling" and "Time Management for the Sales Professional."  He is fluent in Japanese and can read and write French, Portuguese, and Spanish.',5
,'http://accweb/emmployees/davolio.bmp','2004.07');
INSERT INTO Employees VALUES(null,'King','Robert','Sales Representative','Mr.','1960-05-29','1994-01-02','Edgeham Hollow
Winchester Way','London',NULL,'RG1 9SP','UK','(71) 555-5598','465',X'FFD8FFE000104A46494600010200006400640000FFEC00114475636B7900010004000000500000FFEE002641646F62650064C0000000010300150403060A0D00000D060000148800001EFD00002E78FFDB0084000202020202020202020203020202030403020203040504040404040506050505050505060607070807070609090A0A09090C0C0C0C0C0C0C0C0C0C0C0C0C0C0C01030303050405090606090D0B090B0D0F0E0E0E0E0F0F0C0C0C0C0C0F0F0C0C0C0C0C0C0F0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0CFFC200110800AA00A003011100021101031101FFC400E500000105000301000000000000000000000600040507080102030901000301010101000000000000000000000003040201050610000103030304010304030101000000000100020311040510201221311306223032144134150723333542241100010302030505050604030900000000010002031104213112415171221310206132058191422314A1B1D1627233C1E15282303415A2B24353732444740612000104010403010000000000000000001130405021010010603120704161130100020201030402030101010000000001001121314110516120718191A1B1F0C1D130E1F1FFDA000C03010002110311000001F1F95699AB5A4BD449F7B3CE7420415648ECF72D35270AEF788976603BC9F46E43079AFB74ACDCDE9CA635E10601F967CE679AD3D54977AA2E9D3850933F342288F5D434BD8BF7A30C40F350D589979F45136DE635A1A6378FD1C1D9BCC1FF00374183D3AAFD1C4AD9D4032ADE1F9AE9E8BD2359E80256D87723DBCD7EC4D7F4CAD748F2D27B6336E40EFA6FEAC535E82F1278941C532699B3AE9DA415E2DD46C1EA1F4B642AD802870DF39C73822646B598C16C349ACFD2F36E3F3F5BBEF9AE3F5518EBC479AB93A33D2CFA6FA82B84D21B0FAC47C647AF55FC8F01575B73912647C21F23E5E2B5B2377B9368D08D37ECCD8EFE7E8B61D8BAFD8573A170ACD15F58BD27DC2096F009DF5BA3B19CE3201AE7224CBA5E7C7AAADEFF003F5CBE6D95EC4F803E70D2959757AE73D3C31B044D3C45E8016B813964CA5E272BC431C8E018C6234C38C73CFAA09B61D6F5CBAF7D69F217898BCAB2CBBF6BA0BCD6AE6BC53DAD536E59E1219DD948AEB65804E9074510C8D9B46E15C9694C7BCEF92E2F493E2AE7AEFBCE841169A0363F47D300DEB58C33C93C3EFC4F730B258FAE71C569C23471E755DB6B9574BF423DB8A628C204081042A688087D015CB2B39AC803452C4414EDC6D579648FD0E715232B6E5972C73AF1D51BEBDEF24BA99F8C9CE84083CB3A8086F81C554745E93AD66EBBBCECD3E6FA3973B0F933B3DDE3F528AE4731626C1653BC7E93E79047C848542E88100EC3755B2FA945CB7DDD52091D3657F32CA9F92403C2ECF5CA71388C8CB1DB4BD9F2EF6F47CF40DD43868810201187D1C9907B1EED5DC15606E57E7F98A85B1B31961A2A72A989B0B276F7777D2F81CF441D567660810750AC7CFF004EB78BD3A875B32A781C8767DECC31A8FC3B9398EA78237FDD059574F2EDC2040C503E788380C98A6C1CBEA5E33B20B344352EAAE7C5258C55F44736AE9123A442BE9E7B9E53BDF102041522021772677CE8DC7C9A83D3FA0D2FA45E2454688AA8A233AAC85344CFDB4AF3EE37E5B2681BE64081020CD3CC449ABC7D145531D38BBCDF737229EF7B3C6EB95D25F50AB95CE531E279337074FA37EE798810204083373130B8DDD97A69C8ABC85E5FB9B2D74CC6D356AB61D9CD702EB9D4CCC576C3FE8B591E80BA44081020419534A20672CCAB145CEDCAB07A76FC3EC595DE5209650AE93A9865D5F5E688E6EFD58F4FC833A96810204083316D33ADCE4DE332D2B77E3B32D1FA45FE5FB607CC826D5E3CCF4E721F68B0F31D83C4DCEDE6B4B94FDBC408107CEEDA33FE183E173B06FA2F67F72CFCE7D31AA5E37BC4332761579F38444D3F62B3A20C68E166A1BD579DB84083E74391198D0B85D6FE516AD69CB33833C1F7EE6459DE894BAEF040A6D464DA74BDC2F7B24126BD1D63BF46FD79F9D883FFDA0008010100010502B5F888E45871FE3D99DF606591BBB4730E4AFADC2932E2492EAFE2318BC4CBD8D8D9AFF946C89CD67ACFB65EE19D88CE63B350681B4308ABF14DE30684868CDFB4DCDA89BD81F0A6DBDFDE29B15D65C2706BF1E68F81EC4C9F82E5CC7E6BA082D6E035DEB79216F7C3B202A71F6FE4B8B46708B4CA5C786DF2AD96E67C4E061B765CC5464AC08C6D70B8B38A971661CA7C67516CF899787CF1DA1E471775F8F2E3E6FC9B44C62C475B887FD7A654B9D2B31959E7B7F129612E13DB14D610646FC6686AA48118398BAC531D0430BADA7838193D2B206F314A362C34245D37EDD2FC379DAC679CCCE4DE2DE376D1578E249A87D14817643E4DCADA7036330919FD75F1099335605BCA4D6F64315CC718ACA035A5955908CB0CBDEA9E54AE55519EB7CD6B995F04BFD6F7ACFE455BDC75F5A3C869248D89B7B0F99B1B45BC794CCF8145EDD6357E42D2FC4F0558450CC9FD51098AE5C38D1BF91FD798831E450F569187058E7DA47A6643CE34B43EDE22C0B239065B8BFC8E2E79B12CB67AAB8584B703C7779C8224CCE5ABCC57504E8355D0E2EF0791DE9111661D706ECBD9590DA59CAF9E2965157626DA539FC35C1C87AF5ADDFE6E67F11968D8E1FC8B9BDB08E66CF8E9DECB4B095F6E5FC6EED269062AD99249611C1159EDC8DA8BEB2B38BC56F9106074F917C723A37DC9C6625A64F6E3C659DC209EC5914B1C960F92E1B04B1BACBC3C1E64BBBAB86C85982C7C98BC46D78E4C8BB5FC6D7B6E2DA16BAC6DE29E586386DE1F6D787DCDDC7D638E85BCCA8E22E16AF13BA3B9FC65E92C390CEE8E790ED97F2189BEC199B7C5C308BEC89C75A436CDBA90D95AE6247492DE465C2DBE463A30174B29B0608DD7519907A0610E2B0DA39B53B323F7E5C7E47B06032B6791BA96B1ABDC944EB6F61BB640F9328F78B69446EF234B20782F987821F4DC71BECA6F2401FCAD965565AC99F9171EAB15E4181BACABADBD82D645746270E30B535ECE50F289B8E98C93FAA6321FE1ACEC6CF1F1EB1B8BCEBFD8DEDEEC5DBFF5CC0E8F013C7E48E367176571F1DC32EB377E2DB233433C977666EDF69630DB9B9700DF5DC6CB7B75042CB78365BFB0357F3B6C1457E266DC5D8821F65B8BAF61CFE32D598DC735C1C2F5C227DE5E97C17B3554EF7720107517CA67FA2FAD9C5DAEDB5B60AD9FF9B9384AF7BBD7DAE1BD120FCBCE48FEAD7F895E06CE2EFF00F9A2BB8FC91BE95AD5715E9B88191CC6EBB97F1AD70D118540BDF62F2D87A64E2CEFDEF45E08BECC0B196FB250DDC4EBAA4534D473270D5E67C87FAF30FF008188DD3FF99D8F8BE51F45EDF9DC232E64F5DCFD964ADF3A248E0BBF2C5ECF33791B873CB9F305F37AA2B3883A4F5F9592E27770E86E6CF116BEC7FDA3733AB88AE5EFF56CC65E0BCF68B7FE3EEB1993E76594E5707C2D8D70E49CD0D0646B4E2AD5C24C8E6EEAD21F58F7BB970B7B982EE3D992F6FC8C4B24CCD65E4871327972986E381F51A09FD96DA33EB386BCFF000CDF24F894CFE0BF0AFEE15BD841665B212AFC831629D1C71C79CBCB097D67DCA1CC6CB8573FEBB4FDC64BFE0FAB7EF337FF00030DA5A7EFEE3F7051FBA359350FDACFB303FF00435FFFDA0008010200010502D234363DDA86AE2A8A8A8AABB26495D42A28C6CAE85CB9AE6B920E45B5D38D51D187A26841A9835768E722AAAAAAB92122E754CE89C8AB7AD134209BB1E88446EAA12752EE415BBB5614379DF1B910A1FBB4621A947576F0828BEED23D49A277608AA270446D2820A1EA506A8DBACBF6FE829A508D0E8188C4516D346E96FA701B24EC0FC4A6A2FD1D44C62A23545382669101C76C82A023AD149DD87A5573555226E91338B769D1DA040297BB557576910F9EF9138AA263688F40E434E49CEAE95503753B64EE9BA3D52ABC68E8504D67276A76BFAA7261451D3AE85395BB3E2D606EC1B1C8AA69FA209AFE28BC9D0F54D141B06802A5172D48D29B6266E6B537E454A7A3B5AA3B5BD4EE3D033497B3BBE95D28A8A9A42DDE53747F54E8CD08D69B0A88FC7785CB4054AC5540ECA555289A1732D4D7076DEDAD13511D1E35E284689D027201364D8ED023D99DCF691051774ED0A6EAE4DEDA7FFFDA00080103000105021BDADFA3DD114D876F1544235E35E35C116FD127604026B29A534A2E28C68B289C10D1DA1450D426040A1B8845AB8D0A78D48D8D410D06F9068FD5DB1BA03A0DE5393B4AEC68AA6773D50554DDE5393B4A6CB7FBFF00F5425502F8A08228C945E45CB428A7EE8855D20A39A895E147A00E7274CBBAE8A9440A79D1DBA27F07486A46AF29BD9E3AF15E25D932A9EA26A95DC9DB4777288AA22288D491F691A9405152A65E8CDF08AA89AB927390F913A52AB8AA53460570FA9DF0F61D8E8DEEF345CD34A3A7EB23B8B7E8449A57747E2A81C9CC5C5043B02A47752E27E83426F6AE8D2BC4B814E8EA800112BB23F43BAF192A9A34E9E445EAAABA3DDB89D62EF1E850434AA2747741B823A44A3EDA71D2A9C5040295DD7704746A649D41D6BA51534777FA14D627A08854D49A273EBA52BF4C1EAC774D0B93A544D76D360D874851537DBF4FFFDA0008010202063F02614CEA2C710CB4A9B30F4804C22522CC343014FED3F8AFFFDA0008010302063F02F4861A6359DAF7E9629FE6C7461C238863E271AEF8F897FFDA0008010101063F02037209A7C3BBF4769F3AF5F8380C74577AFAAB90D6BB325CEA2F96EA7B6A98C73C363AE340A48A291CE8A423AD5D38D32C900DE468C1795F28F888A0C7DC9ADB7E415C6B9946E22F94F69E67BB663B950B19716D33BE650531F0426B39DAE77FC580E0F61DC5BDC8C7884DE1DA49C866A564223B50DC0CAFE67FB005AE2ADCDEDC790E9AE908BEE0C83579AA7EDDB8A2D6B08AF99E4F35117D164B1690B3A21A28D5D1B81A1F71B73141960A94662310DF29E215BBE360D1311150E7CD8535786CEE45C53476E96BB4C92E0386D53B43F0D66AF3F82134B57CCFC89D816400ECA6C39AAE8470D34556ADB86D51EA3A648F01B8FE08301F9ADF2B86DF04D2F617C7260E8FC767DAA09E94123416D73ED604DED2D0681ACA7BD0E7D5CD575136992E1D851ED2C2334699D16939EA1A4F8AA169D3A6BC2BFCD089EED525A3B413E1DBABC10ED989C5C6B40AAF6D10F0EC777A8992347152326A8ABA9AF8AF54656BCED70F68ED27B8C85C2A27796EAF655728E5ED3DE08022A8ED6E45BBC2BCB4EA57AB6E1ED6BB3E53FCFB29555DFDBADD965EFC14AF70E6CDA80268D68C1690D01BFD5992831D3D1C33690B5452B5C1CB5F785551C36ABEF5270E58E06C4CFD4FC4FD83B2A2AB9BB6EBA62B2340730788702817E05F4D54F146399BD4747835F4FB698A93CBD169D2F9E5341ABFA5BBD3E96D6F76EAF99BCAEF61A02BAB625CC0F3A3A6E3E57EE3C54E4FC07141DB76A2284D36AC496F1546BB139767151D0639A2F779A599DF6507665DC9E5930631B52A07114606EB1FDD929295D40FF0085FDBB836F62D4592BB9DB8E634952C8191471E2D63606B8000E7B4E3C53F4B0C7035BAAE243E5D0DC71F1DC9DA6F3E9C5CB43C36588EDDE59A9359777ADFA0CE692D9C1CE207C2D07227C4278B3F4AB46DAB7F6DD70CEB3C8FCC64AE2A92FA75AB8EE8C748FF00B242FF00B195F69203FB52BB532BBB56047B56891BA5F19D2E6F051C8E2D82220E82EF33BF4B76FDCAD21EB089951D6BA98D0515BC76CF12C0D6F248DC43BC7BD716A5DA44CDA6A4D8BFE5B432A7F2E0BEA3CCD384AD38F020782AC32D43B753F82A9AD5CA68751A4ADA5D115A6926B43C5168C83700AF75335B666B7450D310A621DCD24658E0EC4D1C8CB210FD54D54E5CB821AF134A178DBBAA84B700BE76C6C2C8CF95C721ABD8A42F7173EB4515ADAB7AB777123628A2DE4E0AC6C667F526823F9CF196A389A7BFBCE6EF08F81C51E089F29FEA4D8FEA1DFDA8322F2ED5CB8A6C9BB070F0283A37693BD61B36A74927EC45FBD26E1F894F9460D929A5BB80D9EC5717741C87946F764BD3E67455E80966977021BA41F79ED03BADD39BDC07BD5657739C9BB715AE4F9113B207CDEE418D34D7E771CD3DE5D81C19ED41CF38A246CD8B0D9982BFCB87F125007067C3137068F6294641A6B4E2AED908AB46007B712A39E78F4DE5EF3BAB9867C23F8F6B7BB6DE12B495EA37173CD1FA6C35B78FF338D014EB66CBA2EDA2A2DDF8175376F58FB57D3F51BAAB565547ABCDA71684E1D302BE283B76610737169C8A68F15777018E735879C8196C15510922EADB31A5F39D9E00F13FE0124D00CCA99F632F5459CDD290FE66E2A6980FF391697A984558AEDB492D2E1B839AE1E2BE83D5AE9B25EB1CD65AF54734D5C33DEAB3C125B3711D46E2C25AA9D527C4AC66038A018F0E2772E93F6E21337028BEE226CBFEA753331E2A0B3200D574ACAD996D1ED6B053B95D9DC77A7DA3F4CB20F9AEDBC11BB93F73D426748E2770C0270CCEC437A12B0F4AE233AA395B81077A8BD36E2265CC10BF51764F77356878A9E48EC445566989BB01FEA4C38441834E96FDE9BA5BC4A2FF008F26F156F690FEE5C3B45770CDC7D8A1B78C52381818C1E0D14EED34E0B998420E6C64039554934868D8DA5C549D1E712CBA211C55958B32B689ACF6EDEC69AD11D2781CD48E2E692E3CBBD119F8F68DC325FEA176CA5E5D37E530E71C7F8BBBC4D3009CCA7CAB7CC209F1C674BAE3949F042E24C45BD5C38A0AA7CAECFC163EF4F60F8455BED5D40453B281536956D191AE284F5AE0ECD2DFC7BF268F39C963FBB39D5220A1DF5387B1491BB95DAE8878E4A8BA52F936109E192875762315472D562AA56E1B97D6CADA4FEA1CC3C231E5F7E7DF6D472B507F659FA45CDEB59772BA876B63D430EA1D8A4BC82CFEA2DFCFD481CD7E1C3351F351E33424AF15D407DAA8305848EF7AE635EC6572A85641A7F6A311B86ED3DFA2375EA170CB689B8973CFDC9F6BFF00CF33E9A2C8DFBC739FD2362FAAB97BA592E0EA74AEC493E2AD34752E6D5A34DCB1DF0B7782997F6234D95D62F60F85FB5715F9762F1EDF139051DCDC61A718E2FE2546EB795CCA3B0D2689917A9FCF88E1D61E61F8A12DBCAD9633F137BBD2F4DF4F0D79FF00C8979A9C02EBFA95CC9707E169F2B7804D696ED51CC1BCD01C4F82D07DCBD5A8CAB843AC7F69AA0DD5C47662B486389390A2C84209C3567EE5A89EB4FB5E7B1AD76D2BC026CB613988FC5B8F109B6978CFA7BEF848F249C37777D850575FA3B3D57FF524FF00751415B7EA4FE28F128F641C4A282B7FFA8CFBFB9FFFDA0008010103013F21F7F1FDC1C3965FE261E800471507979788812048ACEDA28F885950D8D3DD653FB942256FA9E731EAE94D9EADABE3EE007688E52B58A65F554305F75A816C71522218B75F5514ACD02D7E4B189804E63C9DFC5C14670AEE8AC9D5CEB117BE7EE9A570EAE5D0DA8FB6DBB4FAD7F7333269E185AEF9DCC5FEB0051F896A5808C1659D19EFCC2B5A55D90951D5A807305BA4EE4669B774FF004CDB692CED7DEBB330417A3577472372DAE48D9F67AA59AE43D21546B25F2F758ED321D10F10C2EB36089DBAB6EC5EAEB92256C0C9B4C5995B7DE1267E4B7FE8C12A04680F100AD62627BD08AA01C29B8356CC2F7FB96AC1B704B0616C6B4C5D6833FCB4F6839B1BC2C32A849180F0D5C8F242D80F1C1DDF3D37307741CC15EC7579350AC672CB08CE5B4B1BBA3BC0C1D409ABDF64BAF60DCB97BE2025CCBAD95E65778B8764820295C12E950FD1A3B9543ADD15BE104CDD916D591F9E9B8804643221A1F1D6AE349B8E215F700C6FE268D19269ADC200519FB25A24CC4288591942EF2A8F92A2E54155DC331FEA0A7754AFCE60FE936BBF7D1850E61A3B2BD084B34323847DC97A6C6135FF00B16DE30CE1731A8E5FDCB0AE39A108161E639CFB08083685205FEE9FEE3AF97282783D87D3E413F3F7562B413E4A87DB0F89993418FD4C1A513DA799517C4DE0C1F313DA10ABE58A8CD9A7B40A3A2C7F72D6FE23C6202D3B70F3DA26C69FC32D386905D5389F7980DC4F8A1F7E8554CA12D3D7E368211AF6980F251C9CEAFB41A85D925B5817A633B962232BA5346D9E236E20A9C7DEFC894EC72B182FC4F0EFDE5646E9F0971B567B304A813B0D74C09E0068F79A26D706B33B9B45358A5206888365608FEBA78FF500345756D6C55FC4BC02DDB6CFE908428538933FBA2D5B3801F14C18304470706CB72A67796CF835BA35EE712DB55E5C1DA29F510EC56B84557AB686A32E6AA30E67E7015BC80F045AD4E1347885C6199F23459E289DD81C973B63833245392BE2CD3BB5E51FD788C4F2D9AD62A16388A3B8B18CBEA5BD721C523FD4636DA20A5065F328CC4ED43226F6BFFC9874DC295F021844C76218144BED84D5E81DAD9C6EA7682587DB18D653A799DBEE4C14FCCE3937ADAD14606B88A64B2DEB763BEBBC484E822E2C46702EB9C7994206B7C500715C11A8553EFE18F3092D46C1D3F01A1EA70B6C1F301CFA3E131966C9EE7FF62DD11B435F3DA070177CAA67442DAED576AC26BA43BBA5E77F84BB0EC3B4ECDDDCFEA5BF9908F80EFA00FD5C126E7664A0FB09B196ACF82426C591955E161EBA617E9A41C0FCA540384DC9C0C736C17985EF0F66A15240E513BB08C680CB71F288EDBE65276CB1552AEC2C4B1B53A447D08FE652B938363C0E7CEE68C10E0233287F913B72F8A7F10553EAFF006EA48ECFA6E85E3E0116EECFC493C1A8538E32A6CF8BC4D3155A456AA1175AE3339AD0839829AF41B357365CF94410F46E1C8C4545BE12921C25345B303C6EC0DC0000014180EAEC3D0098769A020722A368FD4A6526A077F77137B30BDB6351507E3138A8174D31595B9AA672630C54CE702B8F78E57D0A31AE5021FD4BD64BD3DBBC644C86FC0C0F35181A76C267E62D646F247CBB7D0D5E98F4608C63C1E12F53BFC7F852D431FEF0B1C07F719C5F753F04A6524125CCB572305D6A2A16DB55BF110015EE3C976B370165BB7E66DF52F3BA07F651F38F8199EDE20E03F5E9B3F3C58D5B27599CA1C5FF00C9E33F80A81D622F7756A54B0E2B9065F72C7F0937702FF31DF80E1569947500600BBA98403C5A62B65A6B8985CFF13301B2172739EDB1F18EFEAC96DB2E0B1C9FD47CCB4014711B9A287696CD07D6B97AFA94C1689C5E4E18B0D9A0ECA8768155802A10E2B2F999D66A254781CC08CF28ECD87CA8F59D861AF94B1BDFB7FB1303C23537023BAD4D86043096ADD2FCA616B2A676F66BAF72797B8E63E521A65BEDE2330B3147E8E61F839FECA1C46AEF4DF2BFA7ADDFE6D3CF13CE2EABDE688F15826C82C9459AF8F9539206DF503060A0EC4C2663763F3C4603B7C223F609B82E07CA2B5AAEF0CE019BCA3E6092B39E428FC7AF1AB39B8734F04CD6872FB459CCB887DDF1EF73085816AB6553CC55CCB966915A4ED197F9185DB76B84594A6C7199745B4B3C2163170D99D4C5A1EB6F4395F880055783FC311E660A7FBD47DF1A97FF0034D77952FE138F4E33A5F70EC6251D1CE27E048E419E5A87825F7AD1D931A62885740C885BF114B404D87105319599AF03B4B662B92CFB4C17160B7FB7DCBACF89FB10D6A25355EFC4B509861E7BA19F0A61881D5D6C1FB3C7A7FC6F13F8FDE7E2FF00AE9FF0BDD34F7463F8BE19F909BFF1667E54DD9FC0F69FB5FA9FB9D203475FFFDA0008010203013F217A0D4D3D16E0E95D129194B0ACA473B83DF0FA80E8B45C5A4ACC3BE93A76741ADCC97018CB0BE89CBD72C468964C3A84915CA42B43CC313CF5D28860EBCC5C4A7A29E8BE862331126EEBC935EAE08EE3A9C43E96338A55A9FAFA0DCDA69D57137FF899BCC898FBFA133874ADD2C20B963D035E97583E1F50EE8BBB2885AE3A45BDCD204B3A46EA10617502BAAA72C11CC332F9B60E0CB9CC68F41AEE50F42C00DBD56E4D6A18B020B5055260465943A943DE6B2EDA98AE97E836540A86E6A64D4ABA2B8870620EA0476C0C54C23D07A1D4E3977A07D238E250E3A0730A4DBDFF00AEA7D3AA5E597E996A594041121994AA20575D8F4BB4198833681196BC4F6446B333CC388393CFEA689E8C9BF42CC18CD531461CC15CF5135986F44A6F4AF1B22B282E0455CC5874AF1292A3D352BCFD72F861288A11E8DBADEA2BF59C4D4218D517AD3AACA54ACF2F5E7042030EA813998F4B971C341FF0C2317299C90F41772EA5CB4263343C92098F4B177D3599C3B4A584209956E71119B7454CA31F6FF899B26F358F67A6D1E8D231E8D1D7FFDA0008010303013F2131F5599EB72FA5B2EE532B947EA0F5010844E9658A25D44ED2E8E8E48FA21EA65D0A03A2A32CDD13A91053D4E69D75946659154BF4574052A3D166BE8FAAE9FF0001E96F4D3FE08D12E65CCE27A2FA6504D7A10B7D55D1308518CAC52FA31E8F46FE8ABAD737F3113208E540CCCCB5053368BA4AB0B7A3DBD4A23A004CD4310ED67226C3F4859EF0794AF44805D4B5E7D553DB323338188BE83A5CA98F465AFB733B25194161EA7418F3D02F36A2C882A39A57699984A9D1BBFF00E1CA9C9026A239958EB12E125F7EB53300D1D4F4EC8307412B93087419145847BC7FC4D130443680DB5DE02C304E624E67845546591F5E26E7FE18AD0E0C2071D040DB4C796076662498E5D2E2B6FD4662D4329A04C0A8E2590C4B12F8DA3172EC7AC707417191702098C516E34E85CDBF5E4C7D16638352A3CA62BA8B3D0BA9DBFE43A8640448E08DE24CBA347FC0B9B874D4E7412E85225CAA830B0BD04AF485CD4B971639177430087C4583A0C58FAC2690DCD27CA7B9E823D4F47FFFDA000C0301000211031100001082D005F5A168340AFA1748F7E7A070A480541ED5A9CB02180D444354C5F8F080643F2272F905305BD92218B4F8D205C32FC9B2A60950A9836476E250A80062E6D6C756000027870EB71120C0031EFB31B01406800B79DBB5F58000004C998D74EC04001233B188CC00E04B1E43137900075B487889F1000010303EF7A980000BDEFE0B45080000C8B417B90E000004B7D051A3C260007ACB8F6CD146A009D2CD18D6EC2CC0FFDA0008010103013F10BB7B43F298079803E632B8301343D0899034C38C28A75049B13EBDB305DE82FB409D2135C5811053127DE90217565E1A5C5D46B52C6C4DA4F72237C08CE800394A14B08F75264D764654364876C66356BE7A11AD90F052853DDCA4580A1497BD464DF8BFA15F375DB6D10EDAD1C7E09570D53D73A6DC4082E327C25770A1FC435D2C199DA016EA21F721C5A2A0108D384550C70D9CB45A9020BECDDA2473A5596A1B58416A1E7579A344592D4B4A9D351B00A576EB3618B1E2F3C4D8586FDCA0AE6B6919B9B5347DBF31FEF8019AD5E077136748405EF446CD912853530B0571C0099AE21E09318B054BC050EFB31622870445867B0ECC887656D42D77D070A1CB043B0305EA06DEE3B7559788F68180EE587CCA9AC303636D644A3A428BBCF9673C72350A8D2A4D68D78D452D380C675932E93B31562E928B18FD4211B872C8775ECE2A56656030ADEAB35FA96E024D5B180CBF882A54922099A6AD34D7D2E5AB96A815AC572569E3B72BB80642A69D088B56042AAD02E9564E98C4D6255C5ADF662536AA8F526C94A06C4A64DF7861C2D188DD79C1CB6788336A96E5B2CABACD0131ABF208E71C7994D439B40FA1614AA301C94BB228F3867BE65192C51482C183C543A54593163670E983D25EDD2B784AC15003DCD5416C9F0DD3DFDA3711DAECD52A7668A4AF3AC11DE6CAF9BE8E9451149A0C192DC4E28A1675CFE238A60383783983B73C7CBD9A57046BD60AB67B7F9333C2829A5D3F1005E0518A46BEE177FDB6ED84AAB3B3B471833A9570120AF946E9B29E1978D7038C1587377A8759D1A51592B01397985BBD8C518C29D59AE94219E21C57497F30283A3F5E60139C715DC942E478AE654E17A7F6E16F86E0920362FF003CC7ABE3F12F5B2D85F0DF1DB105C316BF13CD1C46615E23D10C4D5DCA7CA0BF786380408236027D3338F2C3C06ACA4184784B82664D85142AB6D1ED5E7A326846D4EE5B3AC869DD868E8064EAD013DA7B813169A0C2E961D8E6618500857818DF7ED1EED26B8AB0769E511031CCAA83741DB454A8AB4988B2F7D989529B0C6D0A18F6857D45ED16D71ED5332CAF328505A4513B83F50D694306F42D76B2361023B6DCF64C42E76220D01365FC11D17D855E6FF70CAA9A86B8D1D52819EB2F2F647B25CD5EB14A99B5D2DA61E110D35550AC8B645D51996DDE24B8A425F16099780959B597A098AA4772F01BD88E4C534AD328E6B554CDC2CCE9B208E3225058F75C06E58839BF5D6356DF6958A8B2865F602528EAC4DFB410A957DA3E422B7DE80650D7022817AE325A410D438E31DDAD9EFD15CAFF00087D10EC15D402E9E468FC923F9422BDCB7EF63BC221402AAB6FF528CA6781AB2545A8B3371DECC44E445AC962DC6311C28E9235D94B72BE466E34159C2DD7B0E73F379B1762D8808C1B285AC051961D70A3A362DE82A0EB6B75864390F88E874B036C3E0B7840660B4D8144E8479B862D3DF3D56A2996B319626527B499497B001F9A96D291E0A9BAAF569DFA941B334D45AF785473B30011C6170061A35A5B4A5AC06546CB68AE084DE7D549DD1ABF798AE057758036EB7A864B2B2930F4676ABB30001DBB88003B109169119EDED079317DE0D5505A300BA415846A34B5D80275A6FA5FB9DC3392D62F91D589C3FC8374BC28A310149D43861218136D6EAE00500280C1440418CBB48C8E3BAE148B035E428E4E42FA59E84569BF59084ADB65AEC360B332F609F02387DE91286FC10DD94BD98A6516D00376850572C1BD787B5B965772BC60027B6BF1046EE8C9A4E9CBC9F300EE223E96EA82CF105D75430166B6A31F10D3AD74DD6825606D2F35445FB8855A1E06047E639A4E63096965AB45755032286E472C69BCF5D1356F1BB87A1CE822ED74F66F6436156F180C395A0ACC1FAE5A56B65B0271BB5E4847ACD066C4716E8854089C215186BF739B1C3822DE1AA0984105CDD36BF5150664C50E30889BE18C0B191EBEEEED191999AE51784AD2AD72B0040EF615003831A993913D6121DAD11A9F0F155C85D0D8E697AEB58BBFF35FE7A48C416F14AF3F241F1E18206B66AEF57177022806489771CC974DD54E482A5132FBD710E7118D9B086A6AAA98D4952B2D8BABD175DE65A42A40ABAA2E5A3515652BB3FB80843E253E2019350DEEE0742DA92E07076E33193A5A96973AD057398098E08A00C0075477DBAF43B043F405AABA0269F39A0068655617998A0321A550AE85AC7E23F546A8D46BA9EE73962896BED86C674A8DDFBCC9CD0057C25CB420E3B467AD42609AA460F12F8FDB49FEE9090295AAEB0996DE2D76ADA0BC62FEE50069D6CB41F75712DE8AC9309007CE2909FA8EF6E0E2D6BD00B50D10C5F6EAA05AD073082B4EE8D1C879964B5A339297C228584809152AC7C1BA94358E9AA530A96A4A8AB694A555A8DCC2F8B03662C2BB68C32F611EA50145960254F1A6D216ED3233072E1C8E5CBC83C105E579E5E50A3C5AC629544A5A860E0ABC12C45B9B2E1AACD65E918191E00B72DE182F0840CFB5840A2AAE96B868FED0F6BE7405624D4A96E9B9703735C01E28F477B42A387DD4C6900E68113B35B894F2CF541132631B771AF5C597016B5E0A2E145705691BCE7BC56987889616692A93A402DB5B547B7E214169949196D57629C87A934F65605529A608708AD414E7716ACC2361C03406289C99CFD607E6A181A146CC2DFB3330077DA6D9E133A5F053A7E2080D5C8D9716F0FCAEC496D1C3B2B894BD5EEA156DCD563FF633403A4B9D976916A1D0B261F8B95E71AC6F5EFEB34B5006D45633F132C0047617CBA182B6E804DC1312C006FDA3365CE380681C8A54A6D019B23C4B402D29C955A9718A6866068285A0E48606025542AD38DA6A5DD932D6E4D9CD05A8D356DA3774F2B32A4E839F754AB34AE3973CECCA315842ED3BC588767D61BD29BD63951C73117B95FD961C3E08124DE261EEE101780D006F968886DF14041A6CA2F1A4B9E1A1EC40D3E263B077D0D02BC1550CC66CC823769ED0A9736D27E0C44DB94965FDB18A76DEE28C4BF30B7282F3600FB66139861A0938C04F5949B9946DD11670401565B1560031DA5A29661B00ED678112252929A12B977850E681C0CE98142F172A0A6E3C0CE0D1062ADD5D9C1912FEE5957226AC190C77CCFBB8697E1022D82BCA6BED5AC0E6651EDBA011AE1EDE41E2B7B3E65F0DE28E0DBC4A2470B0FDD78BCE7B3023AE068FC85E12FD2AD3CB1522E91879615B0362078C45EC5C63D0C5154376CC2C2282D5E5ACE3CC2D99B5691D622584B8C6E3560167E65A218B807429E62DB80CB36DFC863B10B6E23D99F88902CA315AD84A5738FD4B0639C82E0F68F78E5843CBA655A2EE13CF011B75B8259050D26F3C7EE4AC875D54EE72B5BEF8EDE8E3F13741F99FD27F11DE77FB9FAE87BC9DBF4CD7D9D2CFE6FB4FE03BA7E53F6CFCC9FC8F13F83FA27E57ECCFE33B67E21D7FFFDA0008010203013F10C88953008687A0B2B1CFF8444E25DD111DCD0A815A8C5699936451A66C549DE1E08ECDEFDBBF50A46CD4C878828E86CB44C8DDC064E881A6E5388AB07CB35A50B8DB77266E40953175E485188D6E8FBE96901484750B9F7F6998BA2328912C415C5954485B99655918E760F0CBCC32C1E23E3A10973E8B932CC232971728977D06BA2E370F30EA768D588859EE7F6741A97FA1311F30AA61B941A7302511C4CB2EBA05330AA5CBCA3B7FC71D3952E382BA12C8C46542262578829E845E957046915BA61E344FEFFAE984194D0E8173ED164C5A3105438D8965DCA0BE8A8D906650665E5AEE0BF6E3FDE89510815D0B71B11FC901EE37512A39ECC55A15DAD7E230EE8C2A4B22EB01C4DC466531331CE029F8DFEFA010CD1D416EAA307E60998414693E658CFF003EF30AB8D57D4781F71B8866902A080B8526132FE7D4B095277F1701103B9CBCFABD9D813C12A228D41DF31D87387DBA12D230DCADB97199E35D9E3DA5E117C22AB44D8880F6DAFCFF00E7E7A05F45BF7103084312F463C0FE7EE08A221730FE114811192B41F4C7CD0AB0EEEE6FEB3D559A7A2E15FC62986DFC4B3332F98B65E1D40ADBB80448CE650FF4C434E897A4062C2053381FC777F5D6F4AF4FE411CB815298961C4C1054C266E3EA67A95905A8989F039F68000D1D5E0F48D87133FCCA15FC779903572DC744CDAB3ED0761515A928CF72C0EF5F87FEFE8961417F7F6E7D17BDE86A1DBB87692E8419D349CC588E889CD04B595EF0DCB3201A8C3CCD4323415FCF7F4BD09C826E71330E26C834992FA39659A54B8E4624E666E26095B445CAEBDBBFA9AAE32DEA015169EF0D2450A2A54CC40542596554DE6391A1CFB40AF57B94E66DDCD25C7DE26CEE7E630C7536A622E629E2118209A966F7FA7ADD8BD4E6984232CCE0AD3B31AEE583312E4033048D230CE399603263D758CCA1CC7AA31FB8D9882CCF7218F13C4BF1006E2E180981A8867760A86335F0F3F12F0FE393D2D19DC5ED01634BC369A94BBEA3DA7EA167001C12B2DA42B88965D3DE3A34BAFF005FEFA3486BFC9BFA097B13FF00A373FF0097FA8EF9F511DE6A4FC03F5D7FFFDA0008010303013F10A098C5CC7D0F93500D114E6263660D30ABC4744A3510287D453C750262437D42DA9EF2350732F0E250DC0AD4322C64A6591883B830D9053D3474AE2FAE6B8950800C05898E30D97435328464B63C4D2128747327A031505287489662318861A54BCA5D1C4BEF28A39E99F4428AEA2D210A13065B7348B16FAE51940798BDE69F7E824447A812CE26758D0255221D1E948B531532A6647A572F7513BBFCCC151DA1691122F4726A566523AE8A299C3A2BA3065D704A86CFCA53C0DD5F8D4000D1792FCCB463E2561798E43CFD4CB2805C25507661F3971D456C781D107D1B925FFEFE2098F69495FCDCB55C5F0FDC61BFB37FE4C716747FEC0191E57F483280A418444FAEFF00110DB82E997C8712C21689A05B32EB4F6F5602BB7EF107C9CFDCE6AC437489C6A0D17C38F2CC8AED8AF8C0B916B0E20382DCB1903C9B76AEDE6FE261425AD219F89A0C5C7B74AF41B1C244A792369839027B625750BEECE04ED148764B939439C2F298BBAC0A4F7F996C1C4218EE8EA64F4825D02E582F0748586D10286CDCA184AD70C58D315BFC081566FA115E5997BFD79FAEA2BD3F8B04F162A2873127BCA160C4C2595AC4721D59A97C3749748D2C1155B7A87A573F331BB906AD5FB3C900B163638F7255417F3D0593814E431521CC8CC95FDA316D5F468EAB501E07128F993B52D87A4723B3BC5EC4FA26A03FDC42DAA62199812AEB822B2E7D2C5D1AC0404062147B0AE86A43BDC76A994B5132896CB3B97EBA3D5C131FBC5836768948B08ADA4C92CCCA90A2D710CA643DBAF599DDF1D2892110C60C3AA4B14E6203728659D4CF305D7EDEB744578E88EE648E18E26488B5976A57D0C09B9ED334F7FF0081B52ADCD4B2C72A382E20A8275D15054CBACC4982629518DC554FA4E04D2139B528801954916E1EF33F64D6852D7A1A0844BA739EA136E8689A7737FEBAE95F986A1E809B7C477D7FFFD900','Robert King served in the Peace Corps and traveled extensively before completing his degree in English at the University of Michigan in 1992, the year he joined the company.  After completing a course entitled "Selling in Europe," he was transferred to the London office in March 1993.',5
,'http://accweb/emmployees/davolio.bmp','1991.55');
INSERT INTO Employees VALUES(null,'Callahan','Laura','Inside Sales Coordinator','Ms.','1958-01-09','1994-03-05','4726 - 11th Ave. N.E.','Seattle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aura received a BA in psychology from the University of Washington.  She has also completed a course in business French.  She reads and writes French.',2
,'http://accweb/emmployees/davolio.bmp','2100.50');
INSERT INTO Employees VALUES(null,'Dodsworth','Anne','Sales Representative','Ms.','1966-01-27','1994-11-15','7 Houndstooth Rd.','London',NULL,'WG2 7LT','UK','(71) 555-4444','452',X'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','Anne has a BA degree in English from St. Lawrence College.  She is fluent in French and German.',5
,'http://accweb/emmployees/davolio.bmp','2333.33');

# ---------------------------------------------------------------------- #
# Add info into "EmployeeTerritories"                                    #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE EmployeeTerritories;
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(1, '06897');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(1, '19713');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '01581');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '01730');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '01833');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '02116');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '02139');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '02184');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(2, '40222');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(3, '30346');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(3, '31406');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(3, '32859');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(3, '33607');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(4, '20852');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(4, '27403');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(4, '27511');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '02903');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '07960');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '08837');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '10019');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '10038');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '11747');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(5, '14450');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(6, '85014');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(6, '85251');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(6, '98004');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(6, '98052');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(6, '98104');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '60179');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '60601');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '80202');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '80909');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '90405');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '94025');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '94105');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '95008');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '95054');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(7, '95060');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(8, '19428');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(8, '44122');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(8, '45839');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(8, '53404');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '03049');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '03801');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '48075');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '48084');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '48304');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '55113');
INSERT INTO EmployeeTerritories (EmployeeID, TerritoryID)
VALUES(9, '55439');

# ---------------------------------------------------------------------- #
# Add info into "Orders"                                                 #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Orders;
INSERT INTO Orders  Values ('10248','VINET',5,'1996-07-04 00:00:00.000','1996-08-01 00:00:00.000','1996-07-16 00:00:00.000',3,32.38,'Vins et alcools Chevalier','59 rue de l-Abbaye','Reims',NULL,'51100','France');
INSERT INTO Orders  Values ('10249','TOMSP',6,'1996-07-05 00:00:00.000','1996-08-16 00:00:00.000','1996-07-10 00:00:00.000',1,11.61,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10250','HANAR',4,'1996-07-08 00:00:00.000','1996-08-05 00:00:00.000','1996-07-12 00:00:00.000',2,65.83,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10251','VICTE',3,'1996-07-08 00:00:00.000','1996-08-05 00:00:00.000','1996-07-15 00:00:00.000',1,41.34,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10252','SUPRD',4,'1996-07-09 00:00:00.000','1996-08-06 00:00:00.000','1996-07-11 00:00:00.000',2,51.3,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10253','HANAR',3,'1996-07-10 00:00:00.000','1996-07-24 00:00:00.000','1996-07-16 00:00:00.000',2,58.17,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10254','CHOPS',5,'1996-07-11 00:00:00.000','1996-08-08 00:00:00.000','1996-07-23 00:00:00.000',2,22.98,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10255','RICSU',9,'1996-07-12 00:00:00.000','1996-08-09 00:00:00.000','1996-07-15 00:00:00.000',3,148.33,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10256','WELLI',3,'1996-07-15 00:00:00.000','1996-08-12 00:00:00.000','1996-07-17 00:00:00.000',2,13.97,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10257','HILAA',4,'1996-07-16 00:00:00.000','1996-08-13 00:00:00.000','1996-07-22 00:00:00.000',3,81.91,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10258','ERNSH',1,'1996-07-17 00:00:00.000','1996-08-14 00:00:00.000','1996-07-23 00:00:00.000',1,140.51,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10259','CENTC',4,'1996-07-18 00:00:00.000','1996-08-15 00:00:00.000','1996-07-25 00:00:00.000',3,3.25,'Centro comercial Moctezuma','Sierras de Granada 9993','Mxico D.F.',NULL,'5022','Mexico');
INSERT INTO Orders  Values ('10260','OTTIK',4,'1996-07-19 00:00:00.000','1996-08-16 00:00:00.000','1996-07-29 00:00:00.000',1,55.09,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10261','QUEDE',4,'1996-07-19 00:00:00.000','1996-08-16 00:00:00.000','1996-07-30 00:00:00.000',2,3.05,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10262','RATTC',8,'1996-07-22 00:00:00.000','1996-08-19 00:00:00.000','1996-07-25 00:00:00.000',3,48.29,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10263','ERNSH',9,'1996-07-23 00:00:00.000','1996-08-20 00:00:00.000','1996-07-31 00:00:00.000',3,146.06,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10264','FOLKO',6,'1996-07-24 00:00:00.000','1996-08-21 00:00:00.000','1996-08-23 00:00:00.000',3,3.67,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10265','BLONP',2,'1996-07-25 00:00:00.000','1996-08-22 00:00:00.000','1996-08-12 00:00:00.000',1,55.28,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10266','WARTH',3,'1996-07-26 00:00:00.000','1996-09-06 00:00:00.000','1996-07-31 00:00:00.000',3,25.73,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10267','FRANK',4,'1996-07-29 00:00:00.000','1996-08-26 00:00:00.000','1996-08-06 00:00:00.000',1,208.58,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10268','GROSR',8,'1996-07-30 00:00:00.000','1996-08-27 00:00:00.000','1996-08-02 00:00:00.000',3,66.29,'GROSELLA-Restaurante','5 Ave. Los Palos Grandes','Caracas','DF','1081','Venezuela');
INSERT INTO Orders  Values ('10269','WHITC',5,'1996-07-31 00:00:00.000','1996-08-14 00:00:00.000','1996-08-09 00:00:00.000',1,4.56,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10270','WARTH',1,'1996-08-01 00:00:00.000','1996-08-29 00:00:00.000','1996-08-02 00:00:00.000',1,136.54,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10271','SPLIR',6,'1996-08-01 00:00:00.000','1996-08-29 00:00:00.000','1996-08-30 00:00:00.000',2,4.54,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10272','RATTC',6,'1996-08-02 00:00:00.000','1996-08-30 00:00:00.000','1996-08-06 00:00:00.000',2,98.03,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10273','QUICK',3,'1996-08-05 00:00:00.000','1996-09-02 00:00:00.000','1996-08-12 00:00:00.000',3,76.07,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10274','VINET',6,'1996-08-06 00:00:00.000','1996-09-03 00:00:00.000','1996-08-16 00:00:00.000',1,6.01,'Vins et alcools Chevalier','59 rue de l-Abbaye','Reims',NULL,'51100','France');
INSERT INTO Orders  Values ('10275','MAGAA',1,'1996-08-07 00:00:00.000','1996-09-04 00:00:00.000','1996-08-09 00:00:00.000',1,26.93,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10276','TORTU',8,'1996-08-08 00:00:00.000','1996-08-22 00:00:00.000','1996-08-14 00:00:00.000',3,13.84,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10277','MORGK',2,'1996-08-09 00:00:00.000','1996-09-06 00:00:00.000','1996-08-13 00:00:00.000',3,125.77,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'4179','Germany');
INSERT INTO Orders  Values ('10278','BERGS',8,'1996-08-12 00:00:00.000','1996-09-09 00:00:00.000','1996-08-16 00:00:00.000',2,92.69,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10279','LEHMS',8,'1996-08-13 00:00:00.000','1996-09-10 00:00:00.000','1996-08-16 00:00:00.000',2,25.83,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10280','BERGS',2,'1996-08-14 00:00:00.000','1996-09-11 00:00:00.000','1996-09-12 00:00:00.000',1,8.98,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10281','ROMEY',4,'1996-08-14 00:00:00.000','1996-08-28 00:00:00.000','1996-08-21 00:00:00.000',1,2.94,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT INTO Orders  Values ('10282','ROMEY',4,'1996-08-15 00:00:00.000','1996-09-12 00:00:00.000','1996-08-21 00:00:00.000',1,12.69,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT INTO Orders  Values ('10283','LILAS',3,'1996-08-16 00:00:00.000','1996-09-13 00:00:00.000','1996-08-23 00:00:00.000',3,84.81,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10284','LEHMS',4,'1996-08-19 00:00:00.000','1996-09-16 00:00:00.000','1996-08-27 00:00:00.000',1,76.56,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10285','QUICK',1,'1996-08-20 00:00:00.000','1996-09-17 00:00:00.000','1996-08-26 00:00:00.000',2,76.83,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10286','QUICK',8,'1996-08-21 00:00:00.000','1996-09-18 00:00:00.000','1996-08-30 00:00:00.000',3,229.24,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10287','RICAR',8,'1996-08-22 00:00:00.000','1996-09-19 00:00:00.000','1996-08-28 00:00:00.000',3,12.76,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10288','REGGC',4,'1996-08-23 00:00:00.000','1996-09-20 00:00:00.000','1996-09-03 00:00:00.000',1,7.45,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10289','BSBEV',7,'1996-08-26 00:00:00.000','1996-09-23 00:00:00.000','1996-08-28 00:00:00.000',3,22.77,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10290','COMMI',8,'1996-08-27 00:00:00.000','1996-09-24 00:00:00.000','1996-09-03 00:00:00.000',1,79.7,'Comrcio Mineiro','Av. dos Lusadas, 23','Sao Paulo','SP','05432-043','Brazil');
INSERT INTO Orders  Values ('10291','QUEDE',6,'1996-08-27 00:00:00.000','1996-09-24 00:00:00.000','1996-09-04 00:00:00.000',2,6.4,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10292','TRADH',1,'1996-08-28 00:00:00.000','1996-09-25 00:00:00.000','1996-09-02 00:00:00.000',2,1.35,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10293','TORTU',1,'1996-08-29 00:00:00.000','1996-09-26 00:00:00.000','1996-09-11 00:00:00.000',3,21.18,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10294','RATTC',4,'1996-08-30 00:00:00.000','1996-09-27 00:00:00.000','1996-09-05 00:00:00.000',2,147.26,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10295','VINET',2,'1996-09-02 00:00:00.000','1996-09-30 00:00:00.000','1996-09-10 00:00:00.000',2,1.15,'Vins et alcools Chevalier','59 rue de l-Abbaye','Reims',NULL,'51100','France');
INSERT INTO Orders  Values ('10296','LILAS',6,'1996-09-03 00:00:00.000','1996-10-01 00:00:00.000','1996-09-11 00:00:00.000',1,0.12,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10297','BLONP',5,'1996-09-04 00:00:00.000','1996-10-16 00:00:00.000','1996-09-10 00:00:00.000',2,5.74,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10298','HUNGO',6,'1996-09-05 00:00:00.000','1996-10-03 00:00:00.000','1996-09-11 00:00:00.000',2,168.22,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10299','RICAR',4,'1996-09-06 00:00:00.000','1996-10-04 00:00:00.000','1996-09-13 00:00:00.000',2,29.76,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10300','MAGAA',2,'1996-09-09 00:00:00.000','1996-10-07 00:00:00.000','1996-09-18 00:00:00.000',2,17.68,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10301','WANDK',8,'1996-09-09 00:00:00.000','1996-10-07 00:00:00.000','1996-09-17 00:00:00.000',2,45.08,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10302','SUPRD',4,'1996-09-10 00:00:00.000','1996-10-08 00:00:00.000','1996-10-09 00:00:00.000',2,6.27,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10303','GODOS',7,'1996-09-11 00:00:00.000','1996-10-09 00:00:00.000','1996-09-18 00:00:00.000',2,107.83,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10304','TORTU',1,'1996-09-12 00:00:00.000','1996-10-10 00:00:00.000','1996-09-17 00:00:00.000',2,63.79,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10305','OLDWO',8,'1996-09-13 00:00:00.000','1996-10-11 00:00:00.000','1996-10-09 00:00:00.000',3,257.62,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10306','ROMEY',1,'1996-09-16 00:00:00.000','1996-10-14 00:00:00.000','1996-09-23 00:00:00.000',3,7.56,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT INTO Orders  Values ('10307','LONEP',2,'1996-09-17 00:00:00.000','1996-10-15 00:00:00.000','1996-09-25 00:00:00.000',2,0.56,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10308','ANATR',7,'1996-09-18 00:00:00.000','1996-10-16 00:00:00.000','1996-09-24 00:00:00.000',3,1.61,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'5021','Mexico');
INSERT INTO Orders  Values ('10309','HUNGO',3,'1996-09-19 00:00:00.000','1996-10-17 00:00:00.000','1996-10-23 00:00:00.000',1,47.3,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10310','THEBI',8,'1996-09-20 00:00:00.000','1996-10-18 00:00:00.000','1996-09-27 00:00:00.000',2,17.52,'The Big Cheese','89 Jefferson Way Suite 2','Portland','OR','97201','USA');
INSERT INTO Orders  Values ('10311','DUMON',1,'1996-09-20 00:00:00.000','1996-10-04 00:00:00.000','1996-09-26 00:00:00.000',3,24.69,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10312','WANDK',2,'1996-09-23 00:00:00.000','1996-10-21 00:00:00.000','1996-10-03 00:00:00.000',2,40.26,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10313','QUICK',2,'1996-09-24 00:00:00.000','1996-10-22 00:00:00.000','1996-10-04 00:00:00.000',2,1.96,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10314','RATTC',1,'1996-09-25 00:00:00.000','1996-10-23 00:00:00.000','1996-10-04 00:00:00.000',2,74.16,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10315','ISLAT',4,'1996-09-26 00:00:00.000','1996-10-24 00:00:00.000','1996-10-03 00:00:00.000',2,41.76,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10316','RATTC',1,'1996-09-27 00:00:00.000','1996-10-25 00:00:00.000','1996-10-08 00:00:00.000',3,150.15,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10317','LONEP',6,'1996-09-30 00:00:00.000','1996-10-28 00:00:00.000','1996-10-10 00:00:00.000',1,12.69,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10318','ISLAT',8,'1996-10-01 00:00:00.000','1996-10-29 00:00:00.000','1996-10-04 00:00:00.000',2,4.73,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10319','TORTU',7,'1996-10-02 00:00:00.000','1996-10-30 00:00:00.000','1996-10-11 00:00:00.000',3,64.5,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10320','WARTH',5,'1996-10-03 00:00:00.000','1996-10-17 00:00:00.000','1996-10-18 00:00:00.000',3,34.57,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10321','ISLAT',3,'1996-10-03 00:00:00.000','1996-10-31 00:00:00.000','1996-10-11 00:00:00.000',2,3.43,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10322','PERIC',7,'1996-10-04 00:00:00.000','1996-11-01 00:00:00.000','1996-10-23 00:00:00.000',3,0.4,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10323','KOENE',4,'1996-10-07 00:00:00.000','1996-11-04 00:00:00.000','1996-10-14 00:00:00.000',1,4.88,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10324','SAVEA',9,'1996-10-08 00:00:00.000','1996-11-05 00:00:00.000','1996-10-10 00:00:00.000',1,214.27,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10325','KOENE',1,'1996-10-09 00:00:00.000','1996-10-23 00:00:00.000','1996-10-14 00:00:00.000',3,64.86,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10326','BOLID',4,'1996-10-10 00:00:00.000','1996-11-07 00:00:00.000','1996-10-14 00:00:00.000',2,77.92,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT INTO Orders  Values ('10327','FOLKO',2,'1996-10-11 00:00:00.000','1996-11-08 00:00:00.000','1996-10-14 00:00:00.000',1,63.36,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10328','FURIB',4,'1996-10-14 00:00:00.000','1996-11-11 00:00:00.000','1996-10-17 00:00:00.000',3,87.03,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10329','SPLIR',4,'1996-10-15 00:00:00.000','1996-11-26 00:00:00.000','1996-10-23 00:00:00.000',2,191.67,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10330','LILAS',3,'1996-10-16 00:00:00.000','1996-11-13 00:00:00.000','1996-10-28 00:00:00.000',1,12.75,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10331','BONAP',9,'1996-10-16 00:00:00.000','1996-11-27 00:00:00.000','1996-10-21 00:00:00.000',1,10.19,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10332','MEREP',3,'1996-10-17 00:00:00.000','1996-11-28 00:00:00.000','1996-10-21 00:00:00.000',2,52.84,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10333','WARTH',5,'1996-10-18 00:00:00.000','1996-11-15 00:00:00.000','1996-10-25 00:00:00.000',3,0.59,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10334','VICTE',8,'1996-10-21 00:00:00.000','1996-11-18 00:00:00.000','1996-10-28 00:00:00.000',2,8.56,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10335','HUNGO',7,'1996-10-22 00:00:00.000','1996-11-19 00:00:00.000','1996-10-24 00:00:00.000',2,42.11,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10336','PRINI',7,'1996-10-23 00:00:00.000','1996-11-20 00:00:00.000','1996-10-25 00:00:00.000',2,15.51,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT INTO Orders  Values ('10337','FRANK',4,'1996-10-24 00:00:00.000','1996-11-21 00:00:00.000','1996-10-29 00:00:00.000',3,108.26,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10338','OLDWO',4,'1996-10-25 00:00:00.000','1996-11-22 00:00:00.000','1996-10-29 00:00:00.000',3,84.21,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10339','MEREP',2,'1996-10-28 00:00:00.000','1996-11-25 00:00:00.000','1996-11-04 00:00:00.000',2,15.66,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10340','BONAP',1,'1996-10-29 00:00:00.000','1996-11-26 00:00:00.000','1996-11-08 00:00:00.000',3,166.31,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10341','SIMOB',7,'1996-10-29 00:00:00.000','1996-11-26 00:00:00.000','1996-11-05 00:00:00.000',3,26.78,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10342','FRANK',4,'1996-10-30 00:00:00.000','1996-11-13 00:00:00.000','1996-11-04 00:00:00.000',2,54.83,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10343','LEHMS',4,'1996-10-31 00:00:00.000','1996-11-28 00:00:00.000','1996-11-06 00:00:00.000',1,110.37,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10344','WHITC',4,'1996-11-01 00:00:00.000','1996-11-29 00:00:00.000','1996-11-05 00:00:00.000',2,23.29,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10345','QUICK',2,'1996-11-04 00:00:00.000','1996-12-02 00:00:00.000','1996-11-11 00:00:00.000',2,249.06,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10346','RATTC',3,'1996-11-05 00:00:00.000','1996-12-17 00:00:00.000','1996-11-08 00:00:00.000',3,142.08,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10347','FAMIA',4,'1996-11-06 00:00:00.000','1996-12-04 00:00:00.000','1996-11-08 00:00:00.000',3,3.1,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10348','WANDK',4,'1996-11-07 00:00:00.000','1996-12-05 00:00:00.000','1996-11-15 00:00:00.000',2,0.78,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10349','SPLIR',7,'1996-11-08 00:00:00.000','1996-12-06 00:00:00.000','1996-11-15 00:00:00.000',1,8.63,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10350','LAMAI',6,'1996-11-11 00:00:00.000','1996-12-09 00:00:00.000','1996-12-03 00:00:00.000',2,64.19,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10351','ERNSH',1,'1996-11-11 00:00:00.000','1996-12-09 00:00:00.000','1996-11-20 00:00:00.000',1,162.33,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10352','FURIB',3,'1996-11-12 00:00:00.000','1996-11-26 00:00:00.000','1996-11-18 00:00:00.000',3,1.3,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10353','PICCO',7,'1996-11-13 00:00:00.000','1996-12-11 00:00:00.000','1996-11-25 00:00:00.000',3,360.63,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10354','PERIC',8,'1996-11-14 00:00:00.000','1996-12-12 00:00:00.000','1996-11-20 00:00:00.000',3,53.8,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10355','AROUT',6,'1996-11-15 00:00:00.000','1996-12-13 00:00:00.000','1996-11-20 00:00:00.000',1,41.95,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10356','WANDK',6,'1996-11-18 00:00:00.000','1996-12-16 00:00:00.000','1996-11-27 00:00:00.000',2,36.71,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10357','LILAS',1,'1996-11-19 00:00:00.000','1996-12-17 00:00:00.000','1996-12-02 00:00:00.000',3,34.88,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10358','LAMAI',5,'1996-11-20 00:00:00.000','1996-12-18 00:00:00.000','1996-11-27 00:00:00.000',1,19.64,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10359','SEVES',5,'1996-11-21 00:00:00.000','1996-12-19 00:00:00.000','1996-11-26 00:00:00.000',3,288.43,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10360','BLONP',4,'1996-11-22 00:00:00.000','1996-12-20 00:00:00.000','1996-12-02 00:00:00.000',3,131.7,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10361','QUICK',1,'1996-11-22 00:00:00.000','1996-12-20 00:00:00.000','1996-12-03 00:00:00.000',2,183.17,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10362','BONAP',3,'1996-11-25 00:00:00.000','1996-12-23 00:00:00.000','1996-11-28 00:00:00.000',1,96.04,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10363','DRACD',4,'1996-11-26 00:00:00.000','1996-12-24 00:00:00.000','1996-12-04 00:00:00.000',3,30.54,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('10364','EASTC',1,'1996-11-26 00:00:00.000','1997-01-07 00:00:00.000','1996-12-04 00:00:00.000',1,71.97,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('10365','ANTON',3,'1996-11-27 00:00:00.000','1996-12-25 00:00:00.000','1996-12-02 00:00:00.000',2,22,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10366','GALED',8,'1996-11-28 00:00:00.000','1997-01-09 00:00:00.000','1996-12-30 00:00:00.000',2,10.14,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT INTO Orders  Values ('10367','VAFFE',7,'1996-11-28 00:00:00.000','1996-12-26 00:00:00.000','1996-12-02 00:00:00.000',3,13.55,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10368','ERNSH',2,'1996-11-29 00:00:00.000','1996-12-27 00:00:00.000','1996-12-02 00:00:00.000',2,101.95,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10369','SPLIR',8,'1996-12-02 00:00:00.000','1996-12-30 00:00:00.000','1996-12-09 00:00:00.000',2,195.68,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10370','CHOPS',6,'1996-12-03 00:00:00.000','1996-12-31 00:00:00.000','1996-12-27 00:00:00.000',2,1.17,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10371','LAMAI',1,'1996-12-03 00:00:00.000','1996-12-31 00:00:00.000','1996-12-24 00:00:00.000',1,0.45,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10372','QUEEN',5,'1996-12-04 00:00:00.000','1997-01-01 00:00:00.000','1996-12-09 00:00:00.000',2,890.78,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10373','HUNGO',4,'1996-12-05 00:00:00.000','1997-01-02 00:00:00.000','1996-12-11 00:00:00.000',3,124.12,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10374','WOLZA',1,'1996-12-05 00:00:00.000','1997-01-02 00:00:00.000','1996-12-09 00:00:00.000',3,3.94,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10375','HUNGC',3,'1996-12-06 00:00:00.000','1997-01-03 00:00:00.000','1996-12-09 00:00:00.000',2,20.12,'Hungry Coyote Import Store','City Center Plaza 516 Main St.','Elgin','OR','97827','USA');
INSERT INTO Orders  Values ('10376','MEREP',1,'1996-12-09 00:00:00.000','1997-01-06 00:00:00.000','1996-12-13 00:00:00.000',2,20.39,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10377','SEVES',1,'1996-12-09 00:00:00.000','1997-01-06 00:00:00.000','1996-12-13 00:00:00.000',3,22.21,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10378','FOLKO',5,'1996-12-10 00:00:00.000','1997-01-07 00:00:00.000','1996-12-19 00:00:00.000',3,5.44,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10379','QUEDE',2,'1996-12-11 00:00:00.000','1997-01-08 00:00:00.000','1996-12-13 00:00:00.000',1,45.03,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10380','HUNGO',8,'1996-12-12 00:00:00.000','1997-01-09 00:00:00.000','1997-01-16 00:00:00.000',3,35.03,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10381','LILAS',3,'1996-12-12 00:00:00.000','1997-01-09 00:00:00.000','1996-12-13 00:00:00.000',3,7.99,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10382','ERNSH',4,'1996-12-13 00:00:00.000','1997-01-10 00:00:00.000','1996-12-16 00:00:00.000',1,94.77,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10383','AROUT',8,'1996-12-16 00:00:00.000','1997-01-13 00:00:00.000','1996-12-18 00:00:00.000',3,34.24,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10384','BERGS',3,'1996-12-16 00:00:00.000','1997-01-13 00:00:00.000','1996-12-20 00:00:00.000',3,168.64,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10385','SPLIR',1,'1996-12-17 00:00:00.000','1997-01-14 00:00:00.000','1996-12-23 00:00:00.000',2,30.96,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10386','FAMIA',9,'1996-12-18 00:00:00.000','1997-01-01 00:00:00.000','1996-12-25 00:00:00.000',3,13.99,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10387','SANTG',1,'1996-12-18 00:00:00.000','1997-01-15 00:00:00.000','1996-12-20 00:00:00.000',2,93.63,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('10388','SEVES',2,'1996-12-19 00:00:00.000','1997-01-16 00:00:00.000','1996-12-20 00:00:00.000',1,34.86,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10389','BOTTM',4,'1996-12-20 00:00:00.000','1997-01-17 00:00:00.000','1996-12-24 00:00:00.000',2,47.42,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10390','ERNSH',6,'1996-12-23 00:00:00.000','1997-01-20 00:00:00.000','1996-12-26 00:00:00.000',1,126.38,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10391','DRACD',3,'1996-12-23 00:00:00.000','1997-01-20 00:00:00.000','1996-12-31 00:00:00.000',3,5.45,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('10392','PICCO',2,'1996-12-24 00:00:00.000','1997-01-21 00:00:00.000','1997-01-01 00:00:00.000',3,122.46,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10393','SAVEA',1,'1996-12-25 00:00:00.000','1997-01-22 00:00:00.000','1997-01-03 00:00:00.000',3,126.56,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10394','HUNGC',1,'1996-12-25 00:00:00.000','1997-01-22 00:00:00.000','1997-01-03 00:00:00.000',3,30.34,'Hungry Coyote Import Store','City Center Plaza 516 Main St.','Elgin','OR','97827','USA');
INSERT INTO Orders  Values ('10395','HILAA',6,'1996-12-26 00:00:00.000','1997-01-23 00:00:00.000','1997-01-03 00:00:00.000',1,184.41,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10396','FRANK',1,'1996-12-27 00:00:00.000','1997-01-10 00:00:00.000','1997-01-06 00:00:00.000',3,135.35,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10397','PRINI',5,'1996-12-27 00:00:00.000','1997-01-24 00:00:00.000','1997-01-02 00:00:00.000',1,60.26,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT INTO Orders  Values ('10398','SAVEA',2,'1996-12-30 00:00:00.000','1997-01-27 00:00:00.000','1997-01-09 00:00:00.000',3,89.16,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10399','VAFFE',8,'1996-12-31 00:00:00.000','1997-01-14 00:00:00.000','1997-01-08 00:00:00.000',3,27.36,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10400','EASTC',1,'1997-01-01 00:00:00.000','1997-01-29 00:00:00.000','1997-01-16 00:00:00.000',3,83.93,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('10401','RATTC',1,'1997-01-01 00:00:00.000','1997-01-29 00:00:00.000','1997-01-10 00:00:00.000',1,12.51,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10402','ERNSH',8,'1997-01-02 00:00:00.000','1997-02-13 00:00:00.000','1997-01-10 00:00:00.000',2,67.88,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10403','ERNSH',4,'1997-01-03 00:00:00.000','1997-01-31 00:00:00.000','1997-01-09 00:00:00.000',3,73.79,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10404','MAGAA',2,'1997-01-03 00:00:00.000','1997-01-31 00:00:00.000','1997-01-08 00:00:00.000',1,155.97,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10405','LINOD',1,'1997-01-06 00:00:00.000','1997-02-03 00:00:00.000','1997-01-22 00:00:00.000',1,34.82,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10406','QUEEN',7,'1997-01-07 00:00:00.000','1997-02-18 00:00:00.000','1997-01-13 00:00:00.000',1,108.04,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10407','OTTIK',2,'1997-01-07 00:00:00.000','1997-02-04 00:00:00.000','1997-01-30 00:00:00.000',2,91.48,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10408','FOLIG',8,'1997-01-08 00:00:00.000','1997-02-05 00:00:00.000','1997-01-14 00:00:00.000',1,11.26,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT INTO Orders  Values ('10409','OCEAN',3,'1997-01-09 00:00:00.000','1997-02-06 00:00:00.000','1997-01-14 00:00:00.000',1,29.83,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585 Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10410','BOTTM',3,'1997-01-10 00:00:00.000','1997-02-07 00:00:00.000','1997-01-15 00:00:00.000',3,2.4,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10411','BOTTM',9,'1997-01-10 00:00:00.000','1997-02-07 00:00:00.000','1997-01-21 00:00:00.000',3,23.65,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10412','WARTH',8,'1997-01-13 00:00:00.000','1997-02-10 00:00:00.000','1997-01-15 00:00:00.000',2,3.77,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10413','LAMAI',3,'1997-01-14 00:00:00.000','1997-02-11 00:00:00.000','1997-01-16 00:00:00.000',2,95.66,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10414','FAMIA',2,'1997-01-14 00:00:00.000','1997-02-11 00:00:00.000','1997-01-17 00:00:00.000',3,21.48,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10415','HUNGC',3,'1997-01-15 00:00:00.000','1997-02-12 00:00:00.000','1997-01-24 00:00:00.000',1,0.2,'Hungry Coyote Import Store','City Center Plaza 516 Main St.','Elgin','OR','97827','USA');
INSERT INTO Orders  Values ('10416','WARTH',8,'1997-01-16 00:00:00.000','1997-02-13 00:00:00.000','1997-01-27 00:00:00.000',3,22.72,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10417','SIMOB',4,'1997-01-16 00:00:00.000','1997-02-13 00:00:00.000','1997-01-28 00:00:00.000',3,70.29,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10418','QUICK',4,'1997-01-17 00:00:00.000','1997-02-14 00:00:00.000','1997-01-24 00:00:00.000',1,17.55,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10419','RICSU',4,'1997-01-20 00:00:00.000','1997-02-17 00:00:00.000','1997-01-30 00:00:00.000',2,137.35,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10420','WELLI',3,'1997-01-21 00:00:00.000','1997-02-18 00:00:00.000','1997-01-27 00:00:00.000',1,44.12,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10421','QUEDE',8,'1997-01-21 00:00:00.000','1997-03-04 00:00:00.000','1997-01-27 00:00:00.000',1,99.23,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10422','FRANS',2,'1997-01-22 00:00:00.000','1997-02-19 00:00:00.000','1997-01-31 00:00:00.000',1,3.02,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('10423','GOURL',6,'1997-01-23 00:00:00.000','1997-02-06 00:00:00.000','1997-02-24 00:00:00.000',3,24.5,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10424','MEREP',7,'1997-01-23 00:00:00.000','1997-02-20 00:00:00.000','1997-01-27 00:00:00.000',2,370.61,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10425','LAMAI',6,'1997-01-24 00:00:00.000','1997-02-21 00:00:00.000','1997-02-14 00:00:00.000',2,7.93,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10426','GALED',4,'1997-01-27 00:00:00.000','1997-02-24 00:00:00.000','1997-02-06 00:00:00.000',1,18.69,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT INTO Orders  Values ('10427','PICCO',4,'1997-01-27 00:00:00.000','1997-02-24 00:00:00.000','1997-03-03 00:00:00.000',2,31.29,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10428','REGGC',7,'1997-01-28 00:00:00.000','1997-02-25 00:00:00.000','1997-02-04 00:00:00.000',1,11.09,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10429','HUNGO',3,'1997-01-29 00:00:00.000','1997-03-12 00:00:00.000','1997-02-07 00:00:00.000',2,56.63,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10430','ERNSH',4,'1997-01-30 00:00:00.000','1997-02-13 00:00:00.000','1997-02-03 00:00:00.000',1,458.78,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10431','BOTTM',4,'1997-01-30 00:00:00.000','1997-02-13 00:00:00.000','1997-02-07 00:00:00.000',2,44.17,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10432','SPLIR',3,'1997-01-31 00:00:00.000','1997-02-14 00:00:00.000','1997-02-07 00:00:00.000',2,4.34,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10433','PRINI',3,'1997-02-03 00:00:00.000','1997-03-03 00:00:00.000','1997-03-04 00:00:00.000',3,73.83,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT INTO Orders  Values ('10434','FOLKO',3,'1997-02-03 00:00:00.000','1997-03-03 00:00:00.000','1997-02-13 00:00:00.000',2,17.92,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10435','CONSH',8,'1997-02-04 00:00:00.000','1997-03-18 00:00:00.000','1997-02-07 00:00:00.000',2,9.21,'Consolidated Holdings','Berkeley Gardens 12 Brewery','London',NULL,'WX1 6LT','UK');
INSERT INTO Orders  Values ('10436','BLONP',3,'1997-02-05 00:00:00.000','1997-03-05 00:00:00.000','1997-02-11 00:00:00.000',2,156.66,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10437','WARTH',8,'1997-02-05 00:00:00.000','1997-03-05 00:00:00.000','1997-02-12 00:00:00.000',1,19.97,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10438','TOMSP',3,'1997-02-06 00:00:00.000','1997-03-06 00:00:00.000','1997-02-14 00:00:00.000',2,8.24,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10439','MEREP',6,'1997-02-07 00:00:00.000','1997-03-07 00:00:00.000','1997-02-10 00:00:00.000',3,4.07,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10440','SAVEA',4,'1997-02-10 00:00:00.000','1997-03-10 00:00:00.000','1997-02-28 00:00:00.000',2,86.53,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10441','OLDWO',3,'1997-02-10 00:00:00.000','1997-03-24 00:00:00.000','1997-03-14 00:00:00.000',2,73.02,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10442','ERNSH',3,'1997-02-11 00:00:00.000','1997-03-11 00:00:00.000','1997-02-18 00:00:00.000',2,47.94,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10443','REGGC',8,'1997-02-12 00:00:00.000','1997-03-12 00:00:00.000','1997-02-14 00:00:00.000',1,13.95,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10444','BERGS',3,'1997-02-12 00:00:00.000','1997-03-12 00:00:00.000','1997-02-21 00:00:00.000',3,3.5,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10445','BERGS',3,'1997-02-13 00:00:00.000','1997-03-13 00:00:00.000','1997-02-20 00:00:00.000',1,9.3,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10446','TOMSP',6,'1997-02-14 00:00:00.000','1997-03-14 00:00:00.000','1997-02-19 00:00:00.000',1,14.68,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10447','RICAR',4,'1997-02-14 00:00:00.000','1997-03-14 00:00:00.000','1997-03-07 00:00:00.000',2,68.66,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10448','RANCH',4,'1997-02-17 00:00:00.000','1997-03-17 00:00:00.000','1997-02-24 00:00:00.000',2,38.82,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10449','BLONP',3,'1997-02-18 00:00:00.000','1997-03-18 00:00:00.000','1997-02-27 00:00:00.000',2,53.3,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10450','VICTE',8,'1997-02-19 00:00:00.000','1997-03-19 00:00:00.000','1997-03-11 00:00:00.000',2,7.23,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10451','QUICK',4,'1997-02-19 00:00:00.000','1997-03-05 00:00:00.000','1997-03-12 00:00:00.000',3,189.09,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10452','SAVEA',8,'1997-02-20 00:00:00.000','1997-03-20 00:00:00.000','1997-02-26 00:00:00.000',1,140.26,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10453','AROUT',1,'1997-02-21 00:00:00.000','1997-03-21 00:00:00.000','1997-02-26 00:00:00.000',2,25.36,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10454','LAMAI',4,'1997-02-21 00:00:00.000','1997-03-21 00:00:00.000','1997-02-25 00:00:00.000',3,2.74,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10455','WARTH',8,'1997-02-24 00:00:00.000','1997-04-07 00:00:00.000','1997-03-03 00:00:00.000',2,180.45,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10456','KOENE',8,'1997-02-25 00:00:00.000','1997-04-08 00:00:00.000','1997-02-28 00:00:00.000',2,8.12,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10457','KOENE',2,'1997-02-25 00:00:00.000','1997-03-25 00:00:00.000','1997-03-03 00:00:00.000',1,11.57,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10458','SUPRD',7,'1997-02-26 00:00:00.000','1997-03-26 00:00:00.000','1997-03-04 00:00:00.000',3,147.06,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10459','VICTE',4,'1997-02-27 00:00:00.000','1997-03-27 00:00:00.000','1997-02-28 00:00:00.000',2,25.09,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10460','FOLKO',8,'1997-02-28 00:00:00.000','1997-03-28 00:00:00.000','1997-03-03 00:00:00.000',1,16.27,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10461','LILAS',1,'1997-02-28 00:00:00.000','1997-03-28 00:00:00.000','1997-03-05 00:00:00.000',3,148.61,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10462','CONSH',2,'1997-03-03 00:00:00.000','1997-03-31 00:00:00.000','1997-03-18 00:00:00.000',1,6.17,'Consolidated Holdings','Berkeley Gardens 12 Brewery','London',NULL,'WX1 6LT','UK');
INSERT INTO Orders  Values ('10463','SUPRD',5,'1997-03-04 00:00:00.000','1997-04-01 00:00:00.000','1997-03-06 00:00:00.000',3,14.78,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10464','FURIB',4,'1997-03-04 00:00:00.000','1997-04-01 00:00:00.000','1997-03-14 00:00:00.000',2,89,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10465','VAFFE',1,'1997-03-05 00:00:00.000','1997-04-02 00:00:00.000','1997-03-14 00:00:00.000',3,145.04,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10466','COMMI',4,'1997-03-06 00:00:00.000','1997-04-03 00:00:00.000','1997-03-13 00:00:00.000',1,11.93,'Comrcio Mineiro','Av. dos Lusadas, 23','Sao Paulo','SP','05432-043','Brazil');
INSERT INTO Orders  Values ('10467','MAGAA',8,'1997-03-06 00:00:00.000','1997-04-03 00:00:00.000','1997-03-11 00:00:00.000',2,4.93,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10468','KOENE',3,'1997-03-07 00:00:00.000','1997-04-04 00:00:00.000','1997-03-12 00:00:00.000',3,44.12,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10469','WHITC',1,'1997-03-10 00:00:00.000','1997-04-07 00:00:00.000','1997-03-14 00:00:00.000',1,60.18,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10470','BONAP',4,'1997-03-11 00:00:00.000','1997-04-08 00:00:00.000','1997-03-14 00:00:00.000',2,64.56,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10471','BSBEV',2,'1997-03-11 00:00:00.000','1997-04-08 00:00:00.000','1997-03-18 00:00:00.000',3,45.59,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10472','SEVES',8,'1997-03-12 00:00:00.000','1997-04-09 00:00:00.000','1997-03-19 00:00:00.000',1,4.2,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10473','ISLAT',1,'1997-03-13 00:00:00.000','1997-03-27 00:00:00.000','1997-03-21 00:00:00.000',3,16.37,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10474','PERIC',5,'1997-03-13 00:00:00.000','1997-04-10 00:00:00.000','1997-03-21 00:00:00.000',2,83.49,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10475','SUPRD',9,'1997-03-14 00:00:00.000','1997-04-11 00:00:00.000','1997-04-04 00:00:00.000',1,68.52,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10476','HILAA',8,'1997-03-17 00:00:00.000','1997-04-14 00:00:00.000','1997-03-24 00:00:00.000',3,4.41,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10477','PRINI',5,'1997-03-17 00:00:00.000','1997-04-14 00:00:00.000','1997-03-25 00:00:00.000',2,13.02,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT INTO Orders  Values ('10478','VICTE',2,'1997-03-18 00:00:00.000','1997-04-01 00:00:00.000','1997-03-26 00:00:00.000',3,4.81,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10479','RATTC',3,'1997-03-19 00:00:00.000','1997-04-16 00:00:00.000','1997-03-21 00:00:00.000',3,708.95,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10480','FOLIG',6,'1997-03-20 00:00:00.000','1997-04-17 00:00:00.000','1997-03-24 00:00:00.000',2,1.35,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT INTO Orders  Values ('10481','RICAR',8,'1997-03-20 00:00:00.000','1997-04-17 00:00:00.000','1997-03-25 00:00:00.000',2,64.33,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10482','LAZYK',1,'1997-03-21 00:00:00.000','1997-04-18 00:00:00.000','1997-04-10 00:00:00.000',3,7.48,'Lazy K Kountry Store','12 Orchestra Terrace','Walla Walla','WA','99362','USA');
INSERT INTO Orders  Values ('10483','WHITC',7,'1997-03-24 00:00:00.000','1997-04-21 00:00:00.000','1997-04-25 00:00:00.000',2,15.28,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10484','BSBEV',3,'1997-03-24 00:00:00.000','1997-04-21 00:00:00.000','1997-04-01 00:00:00.000',3,6.88,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10485','LINOD',4,'1997-03-25 00:00:00.000','1997-04-08 00:00:00.000','1997-03-31 00:00:00.000',2,64.45,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10486','HILAA',1,'1997-03-26 00:00:00.000','1997-04-23 00:00:00.000','1997-04-02 00:00:00.000',2,30.53,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10487','QUEEN',2,'1997-03-26 00:00:00.000','1997-04-23 00:00:00.000','1997-03-28 00:00:00.000',2,71.07,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10488','FRANK',8,'1997-03-27 00:00:00.000','1997-04-24 00:00:00.000','1997-04-02 00:00:00.000',2,4.93,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10489','PICCO',6,'1997-03-28 00:00:00.000','1997-04-25 00:00:00.000','1997-04-09 00:00:00.000',2,5.29,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10490','HILAA',7,'1997-03-31 00:00:00.000','1997-04-28 00:00:00.000','1997-04-03 00:00:00.000',2,210.19,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10491','FURIB',8,'1997-03-31 00:00:00.000','1997-04-28 00:00:00.000','1997-04-08 00:00:00.000',3,16.96,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10492','BOTTM',3,'1997-04-01 00:00:00.000','1997-04-29 00:00:00.000','1997-04-11 00:00:00.000',1,62.89,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10493','LAMAI',4,'1997-04-02 00:00:00.000','1997-04-30 00:00:00.000','1997-04-10 00:00:00.000',3,10.64,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10494','COMMI',4,'1997-04-02 00:00:00.000','1997-04-30 00:00:00.000','1997-04-09 00:00:00.000',2,65.99,'Comrcio Mineiro','Av. dos Lusadas, 23','Sao Paulo','SP','05432-043','Brazil');
INSERT INTO Orders  Values ('10495','LAUGB',3,'1997-04-03 00:00:00.000','1997-05-01 00:00:00.000','1997-04-11 00:00:00.000',3,4.65,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT INTO Orders  Values ('10496','TRADH',7,'1997-04-04 00:00:00.000','1997-05-02 00:00:00.000','1997-04-07 00:00:00.000',2,46.77,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10497','LEHMS',7,'1997-04-04 00:00:00.000','1997-05-02 00:00:00.000','1997-04-07 00:00:00.000',1,36.21,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10498','HILAA',8,'1997-04-07 00:00:00.000','1997-05-05 00:00:00.000','1997-04-11 00:00:00.000',2,29.75,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10499','LILAS',4,'1997-04-08 00:00:00.000','1997-05-06 00:00:00.000','1997-04-16 00:00:00.000',2,102.02,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10500','LAMAI',6,'1997-04-09 00:00:00.000','1997-05-07 00:00:00.000','1997-04-17 00:00:00.000',1,42.68,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10501','BLAUS',9,'1997-04-09 00:00:00.000','1997-05-07 00:00:00.000','1997-04-16 00:00:00.000',3,8.85,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10502','PERIC',2,'1997-04-10 00:00:00.000','1997-05-08 00:00:00.000','1997-04-29 00:00:00.000',1,69.32,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10503','HUNGO',6,'1997-04-11 00:00:00.000','1997-05-09 00:00:00.000','1997-04-16 00:00:00.000',2,16.74,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10504','WHITC',4,'1997-04-11 00:00:00.000','1997-05-09 00:00:00.000','1997-04-18 00:00:00.000',3,59.13,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10505','MEREP',3,'1997-04-14 00:00:00.000','1997-05-12 00:00:00.000','1997-04-21 00:00:00.000',3,7.13,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10506','KOENE',9,'1997-04-15 00:00:00.000','1997-05-13 00:00:00.000','1997-05-02 00:00:00.000',2,21.19,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10507','ANTON',7,'1997-04-15 00:00:00.000','1997-05-13 00:00:00.000','1997-04-22 00:00:00.000',1,47.45,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10508','OTTIK',1,'1997-04-16 00:00:00.000','1997-05-14 00:00:00.000','1997-05-13 00:00:00.000',2,4.99,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10509','BLAUS',4,'1997-04-17 00:00:00.000','1997-05-15 00:00:00.000','1997-04-29 00:00:00.000',1,0.15,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10510','SAVEA',6,'1997-04-18 00:00:00.000','1997-05-16 00:00:00.000','1997-04-28 00:00:00.000',3,367.63,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10511','BONAP',4,'1997-04-18 00:00:00.000','1997-05-16 00:00:00.000','1997-04-21 00:00:00.000',3,350.64,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10512','FAMIA',7,'1997-04-21 00:00:00.000','1997-05-19 00:00:00.000','1997-04-24 00:00:00.000',2,3.53,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10513','WANDK',7,'1997-04-22 00:00:00.000','1997-06-03 00:00:00.000','1997-04-28 00:00:00.000',1,105.65,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10514','ERNSH',3,'1997-04-22 00:00:00.000','1997-05-20 00:00:00.000','1997-05-16 00:00:00.000',2,789.95,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10515','QUICK',2,'1997-04-23 00:00:00.000','1997-05-07 00:00:00.000','1997-05-23 00:00:00.000',1,204.47,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10516','HUNGO',2,'1997-04-24 00:00:00.000','1997-05-22 00:00:00.000','1997-05-01 00:00:00.000',3,62.78,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10517','NORTS',3,'1997-04-24 00:00:00.000','1997-05-22 00:00:00.000','1997-04-29 00:00:00.000',3,32.07,'North/South','South House 300 Queensbridge','London',NULL,'SW7 1RZ','UK');
INSERT INTO Orders  Values ('10518','TORTU',4,'1997-04-25 00:00:00.000','1997-05-09 00:00:00.000','1997-05-05 00:00:00.000',2,218.15,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10519','CHOPS',6,'1997-04-28 00:00:00.000','1997-05-26 00:00:00.000','1997-05-01 00:00:00.000',3,91.76,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10520','SANTG',7,'1997-04-29 00:00:00.000','1997-05-27 00:00:00.000','1997-05-01 00:00:00.000',1,13.37,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('10521','CACTU',8,'1997-04-29 00:00:00.000','1997-05-27 00:00:00.000','1997-05-02 00:00:00.000',2,17.22,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10522','LEHMS',4,'1997-04-30 00:00:00.000','1997-05-28 00:00:00.000','1997-05-06 00:00:00.000',1,45.33,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10523','SEVES',7,'1997-05-01 00:00:00.000','1997-05-29 00:00:00.000','1997-05-30 00:00:00.000',2,77.63,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10524','BERGS',1,'1997-05-01 00:00:00.000','1997-05-29 00:00:00.000','1997-05-07 00:00:00.000',2,244.79,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10525','BONAP',1,'1997-05-02 00:00:00.000','1997-05-30 00:00:00.000','1997-05-23 00:00:00.000',2,11.06,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10526','WARTH',4,'1997-05-05 00:00:00.000','1997-06-02 00:00:00.000','1997-05-15 00:00:00.000',2,58.59,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10527','QUICK',7,'1997-05-05 00:00:00.000','1997-06-02 00:00:00.000','1997-05-07 00:00:00.000',1,41.9,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10528','GREAL',6,'1997-05-06 00:00:00.000','1997-05-20 00:00:00.000','1997-05-09 00:00:00.000',2,3.35,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10529','MAISD',5,'1997-05-07 00:00:00.000','1997-06-04 00:00:00.000','1997-05-09 00:00:00.000',2,66.69,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10530','PICCO',3,'1997-05-08 00:00:00.000','1997-06-05 00:00:00.000','1997-05-12 00:00:00.000',2,339.22,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10531','OCEAN',7,'1997-05-08 00:00:00.000','1997-06-05 00:00:00.000','1997-05-19 00:00:00.000',1,8.12,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585 Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10532','EASTC',7,'1997-05-09 00:00:00.000','1997-06-06 00:00:00.000','1997-05-12 00:00:00.000',3,74.46,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('10533','FOLKO',8,'1997-05-12 00:00:00.000','1997-06-09 00:00:00.000','1997-05-22 00:00:00.000',1,188.04,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10534','LEHMS',8,'1997-05-12 00:00:00.000','1997-06-09 00:00:00.000','1997-05-14 00:00:00.000',2,27.94,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10535','ANTON',4,'1997-05-13 00:00:00.000','1997-06-10 00:00:00.000','1997-05-21 00:00:00.000',1,15.64,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10536','LEHMS',3,'1997-05-14 00:00:00.000','1997-06-11 00:00:00.000','1997-06-06 00:00:00.000',2,58.88,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10537','RICSU',1,'1997-05-14 00:00:00.000','1997-05-28 00:00:00.000','1997-05-19 00:00:00.000',1,78.85,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10538','BSBEV',9,'1997-05-15 00:00:00.000','1997-06-12 00:00:00.000','1997-05-16 00:00:00.000',3,4.87,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10539','BSBEV',6,'1997-05-16 00:00:00.000','1997-06-13 00:00:00.000','1997-05-23 00:00:00.000',3,12.36,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10540','QUICK',3,'1997-05-19 00:00:00.000','1997-06-16 00:00:00.000','1997-06-13 00:00:00.000',3,1007.64,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10541','HANAR',2,'1997-05-19 00:00:00.000','1997-06-16 00:00:00.000','1997-05-29 00:00:00.000',1,68.65,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10542','KOENE',1,'1997-05-20 00:00:00.000','1997-06-17 00:00:00.000','1997-05-26 00:00:00.000',3,10.95,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10543','LILAS',8,'1997-05-21 00:00:00.000','1997-06-18 00:00:00.000','1997-05-23 00:00:00.000',2,48.17,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10544','LONEP',4,'1997-05-21 00:00:00.000','1997-06-18 00:00:00.000','1997-05-30 00:00:00.000',1,24.91,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10545','LAZYK',8,'1997-05-22 00:00:00.000','1997-06-19 00:00:00.000','1997-06-26 00:00:00.000',2,11.92,'Lazy K Kountry Store','12 Orchestra Terrace','Walla Walla','WA','99362','USA');
INSERT INTO Orders  Values ('10546','VICTE',1,'1997-05-23 00:00:00.000','1997-06-20 00:00:00.000','1997-05-27 00:00:00.000',3,194.72,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10547','SEVES',3,'1997-05-23 00:00:00.000','1997-06-20 00:00:00.000','1997-06-02 00:00:00.000',2,178.43,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10548','TOMSP',3,'1997-05-26 00:00:00.000','1997-06-23 00:00:00.000','1997-06-02 00:00:00.000',2,1.43,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10549','QUICK',5,'1997-05-27 00:00:00.000','1997-06-10 00:00:00.000','1997-05-30 00:00:00.000',1,171.24,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10550','GODOS',7,'1997-05-28 00:00:00.000','1997-06-25 00:00:00.000','1997-06-06 00:00:00.000',3,4.32,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10551','FURIB',4,'1997-05-28 00:00:00.000','1997-07-09 00:00:00.000','1997-06-06 00:00:00.000',3,72.95,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10552','HILAA',2,'1997-05-29 00:00:00.000','1997-06-26 00:00:00.000','1997-06-05 00:00:00.000',1,83.22,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10553','WARTH',2,'1997-05-30 00:00:00.000','1997-06-27 00:00:00.000','1997-06-03 00:00:00.000',2,149.49,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10554','OTTIK',4,'1997-05-30 00:00:00.000','1997-06-27 00:00:00.000','1997-06-05 00:00:00.000',3,120.97,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10555','SAVEA',6,'1997-06-02 00:00:00.000','1997-06-30 00:00:00.000','1997-06-04 00:00:00.000',3,252.49,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10556','SIMOB',2,'1997-06-03 00:00:00.000','1997-07-15 00:00:00.000','1997-06-13 00:00:00.000',1,9.8,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10557','LEHMS',9,'1997-06-03 00:00:00.000','1997-06-17 00:00:00.000','1997-06-06 00:00:00.000',2,96.72,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10558','AROUT',1,'1997-06-04 00:00:00.000','1997-07-02 00:00:00.000','1997-06-10 00:00:00.000',2,72.97,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10559','BLONP',6,'1997-06-05 00:00:00.000','1997-07-03 00:00:00.000','1997-06-13 00:00:00.000',1,8.05,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10560','FRANK',8,'1997-06-06 00:00:00.000','1997-07-04 00:00:00.000','1997-06-09 00:00:00.000',1,36.65,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10561','FOLKO',2,'1997-06-06 00:00:00.000','1997-07-04 00:00:00.000','1997-06-09 00:00:00.000',2,242.21,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10562','REGGC',1,'1997-06-09 00:00:00.000','1997-07-07 00:00:00.000','1997-06-12 00:00:00.000',1,22.95,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10563','RICAR',2,'1997-06-10 00:00:00.000','1997-07-22 00:00:00.000','1997-06-24 00:00:00.000',2,60.43,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10564','RATTC',4,'1997-06-10 00:00:00.000','1997-07-08 00:00:00.000','1997-06-16 00:00:00.000',3,13.75,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10565','MEREP',8,'1997-06-11 00:00:00.000','1997-07-09 00:00:00.000','1997-06-18 00:00:00.000',2,7.15,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10566','BLONP',9,'1997-06-12 00:00:00.000','1997-07-10 00:00:00.000','1997-06-18 00:00:00.000',1,88.4,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10567','HUNGO',1,'1997-06-12 00:00:00.000','1997-07-10 00:00:00.000','1997-06-17 00:00:00.000',1,33.97,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10568','GALED',3,'1997-06-13 00:00:00.000','1997-07-11 00:00:00.000','1997-07-09 00:00:00.000',3,6.54,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT INTO Orders  Values ('10569','RATTC',5,'1997-06-16 00:00:00.000','1997-07-14 00:00:00.000','1997-07-11 00:00:00.000',1,58.98,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10570','MEREP',3,'1997-06-17 00:00:00.000','1997-07-15 00:00:00.000','1997-06-19 00:00:00.000',3,188.99,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10571','ERNSH',8,'1997-06-17 00:00:00.000','1997-07-29 00:00:00.000','1997-07-04 00:00:00.000',3,26.06,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10572','BERGS',3,'1997-06-18 00:00:00.000','1997-07-16 00:00:00.000','1997-06-25 00:00:00.000',2,116.43,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10573','ANTON',7,'1997-06-19 00:00:00.000','1997-07-17 00:00:00.000','1997-06-20 00:00:00.000',3,84.84,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10574','TRAIH',4,'1997-06-19 00:00:00.000','1997-07-17 00:00:00.000','1997-06-30 00:00:00.000',2,37.6,'Trail-s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','USA');
INSERT INTO Orders  Values ('10575','MORGK',5,'1997-06-20 00:00:00.000','1997-07-04 00:00:00.000','1997-06-30 00:00:00.000',1,127.34,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'4179','Germany');
INSERT INTO Orders  Values ('10576','TORTU',3,'1997-06-23 00:00:00.000','1997-07-07 00:00:00.000','1997-06-30 00:00:00.000',3,18.56,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10577','TRAIH',9,'1997-06-23 00:00:00.000','1997-08-04 00:00:00.000','1997-06-30 00:00:00.000',2,25.41,'Trail-s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','USA');
INSERT INTO Orders  Values ('10578','BSBEV',4,'1997-06-24 00:00:00.000','1997-07-22 00:00:00.000','1997-07-25 00:00:00.000',3,29.6,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10579','LETSS',1,'1997-06-25 00:00:00.000','1997-07-23 00:00:00.000','1997-07-04 00:00:00.000',2,13.73,'Let-s Stop N Shop','87 Polk St. Suite 5','San Francisco','CA','94117','USA');
INSERT INTO Orders  Values ('10580','OTTIK',4,'1997-06-26 00:00:00.000','1997-07-24 00:00:00.000','1997-07-01 00:00:00.000',3,75.89,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10581','FAMIA',3,'1997-06-26 00:00:00.000','1997-07-24 00:00:00.000','1997-07-02 00:00:00.000',1,3.01,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10582','BLAUS',3,'1997-06-27 00:00:00.000','1997-07-25 00:00:00.000','1997-07-14 00:00:00.000',2,27.71,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10583','WARTH',2,'1997-06-30 00:00:00.000','1997-07-28 00:00:00.000','1997-07-04 00:00:00.000',2,7.28,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10584','BLONP',4,'1997-06-30 00:00:00.000','1997-07-28 00:00:00.000','1997-07-04 00:00:00.000',1,59.14,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10585','WELLI',7,'1997-07-01 00:00:00.000','1997-07-29 00:00:00.000','1997-07-10 00:00:00.000',1,13.41,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10586','REGGC',9,'1997-07-02 00:00:00.000','1997-07-30 00:00:00.000','1997-07-09 00:00:00.000',1,0.48,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10587','QUEDE',1,'1997-07-02 00:00:00.000','1997-07-30 00:00:00.000','1997-07-09 00:00:00.000',1,62.52,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10588','QUICK',2,'1997-07-03 00:00:00.000','1997-07-31 00:00:00.000','1997-07-10 00:00:00.000',3,194.67,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10589','GREAL',8,'1997-07-04 00:00:00.000','1997-08-01 00:00:00.000','1997-07-14 00:00:00.000',2,4.42,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10590','MEREP',4,'1997-07-07 00:00:00.000','1997-08-04 00:00:00.000','1997-07-14 00:00:00.000',3,44.77,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10591','VAFFE',1,'1997-07-07 00:00:00.000','1997-07-21 00:00:00.000','1997-07-16 00:00:00.000',1,55.92,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10592','LEHMS',3,'1997-07-08 00:00:00.000','1997-08-05 00:00:00.000','1997-07-16 00:00:00.000',1,32.1,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10593','LEHMS',7,'1997-07-09 00:00:00.000','1997-08-06 00:00:00.000','1997-08-13 00:00:00.000',2,174.2,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10594','OLDWO',3,'1997-07-09 00:00:00.000','1997-08-06 00:00:00.000','1997-07-16 00:00:00.000',2,5.24,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10595','ERNSH',2,'1997-07-10 00:00:00.000','1997-08-07 00:00:00.000','1997-07-14 00:00:00.000',1,96.78,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10596','WHITC',8,'1997-07-11 00:00:00.000','1997-08-08 00:00:00.000','1997-08-12 00:00:00.000',1,16.34,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10597','PICCO',7,'1997-07-11 00:00:00.000','1997-08-08 00:00:00.000','1997-07-18 00:00:00.000',3,35.12,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10598','RATTC',1,'1997-07-14 00:00:00.000','1997-08-11 00:00:00.000','1997-07-18 00:00:00.000',3,44.42,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10599','BSBEV',6,'1997-07-15 00:00:00.000','1997-08-26 00:00:00.000','1997-07-21 00:00:00.000',3,29.98,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10600','HUNGC',4,'1997-07-16 00:00:00.000','1997-08-13 00:00:00.000','1997-07-21 00:00:00.000',1,45.13,'Hungry Coyote Import Store','City Center Plaza 516 Main St.','Elgin','OR','97827','USA');
INSERT INTO Orders  Values ('10601','HILAA',7,'1997-07-16 00:00:00.000','1997-08-27 00:00:00.000','1997-07-22 00:00:00.000',1,58.3,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10602','VAFFE',8,'1997-07-17 00:00:00.000','1997-08-14 00:00:00.000','1997-07-22 00:00:00.000',2,2.92,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10603','SAVEA',8,'1997-07-18 00:00:00.000','1997-08-15 00:00:00.000','1997-08-08 00:00:00.000',2,48.77,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10604','FURIB',1,'1997-07-18 00:00:00.000','1997-08-15 00:00:00.000','1997-07-29 00:00:00.000',1,7.46,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10605','MEREP',1,'1997-07-21 00:00:00.000','1997-08-18 00:00:00.000','1997-07-29 00:00:00.000',2,379.13,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10606','TRADH',4,'1997-07-22 00:00:00.000','1997-08-19 00:00:00.000','1997-07-31 00:00:00.000',3,79.4,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10607','SAVEA',5,'1997-07-22 00:00:00.000','1997-08-19 00:00:00.000','1997-07-25 00:00:00.000',1,200.24,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10608','TOMSP',4,'1997-07-23 00:00:00.000','1997-08-20 00:00:00.000','1997-08-01 00:00:00.000',2,27.79,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10609','DUMON',7,'1997-07-24 00:00:00.000','1997-08-21 00:00:00.000','1997-07-30 00:00:00.000',2,1.85,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10610','LAMAI',8,'1997-07-25 00:00:00.000','1997-08-22 00:00:00.000','1997-08-06 00:00:00.000',1,26.78,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10611','WOLZA',6,'1997-07-25 00:00:00.000','1997-08-22 00:00:00.000','1997-08-01 00:00:00.000',2,80.65,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10612','SAVEA',1,'1997-07-28 00:00:00.000','1997-08-25 00:00:00.000','1997-08-01 00:00:00.000',2,544.08,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10613','HILAA',4,'1997-07-29 00:00:00.000','1997-08-26 00:00:00.000','1997-08-01 00:00:00.000',2,8.11,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10614','BLAUS',8,'1997-07-29 00:00:00.000','1997-08-26 00:00:00.000','1997-08-01 00:00:00.000',3,1.93,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10615','WILMK',2,'1997-07-30 00:00:00.000','1997-08-27 00:00:00.000','1997-08-06 00:00:00.000',3,0.75,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10616','GREAL',1,'1997-07-31 00:00:00.000','1997-08-28 00:00:00.000','1997-08-05 00:00:00.000',2,116.53,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10617','GREAL',4,'1997-07-31 00:00:00.000','1997-08-28 00:00:00.000','1997-08-04 00:00:00.000',2,18.53,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10618','MEREP',1,'1997-08-01 00:00:00.000','1997-09-12 00:00:00.000','1997-08-08 00:00:00.000',1,154.68,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10619','MEREP',3,'1997-08-04 00:00:00.000','1997-09-01 00:00:00.000','1997-08-07 00:00:00.000',3,91.05,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10620','LAUGB',2,'1997-08-05 00:00:00.000','1997-09-02 00:00:00.000','1997-08-14 00:00:00.000',3,0.94,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT INTO Orders  Values ('10621','ISLAT',4,'1997-08-05 00:00:00.000','1997-09-02 00:00:00.000','1997-08-11 00:00:00.000',2,23.73,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10622','RICAR',4,'1997-08-06 00:00:00.000','1997-09-03 00:00:00.000','1997-08-11 00:00:00.000',3,50.97,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10623','FRANK',8,'1997-08-07 00:00:00.000','1997-09-04 00:00:00.000','1997-08-12 00:00:00.000',2,97.18,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10624','THECR',4,'1997-08-07 00:00:00.000','1997-09-04 00:00:00.000','1997-08-19 00:00:00.000',2,94.8,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','USA');
INSERT INTO Orders  Values ('10625','ANATR',3,'1997-08-08 00:00:00.000','1997-09-05 00:00:00.000','1997-08-14 00:00:00.000',1,43.9,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'5021','Mexico');
INSERT INTO Orders  Values ('10626','BERGS',1,'1997-08-11 00:00:00.000','1997-09-08 00:00:00.000','1997-08-20 00:00:00.000',2,138.69,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10627','SAVEA',8,'1997-08-11 00:00:00.000','1997-09-22 00:00:00.000','1997-08-21 00:00:00.000',3,107.46,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10628','BLONP',4,'1997-08-12 00:00:00.000','1997-09-09 00:00:00.000','1997-08-20 00:00:00.000',3,30.36,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10629','GODOS',4,'1997-08-12 00:00:00.000','1997-09-09 00:00:00.000','1997-08-20 00:00:00.000',3,85.46,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10630','KOENE',1,'1997-08-13 00:00:00.000','1997-09-10 00:00:00.000','1997-08-19 00:00:00.000',2,32.35,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10631','LAMAI',8,'1997-08-14 00:00:00.000','1997-09-11 00:00:00.000','1997-08-15 00:00:00.000',1,0.87,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10632','WANDK',8,'1997-08-14 00:00:00.000','1997-09-11 00:00:00.000','1997-08-19 00:00:00.000',1,41.38,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10633','ERNSH',7,'1997-08-15 00:00:00.000','1997-09-12 00:00:00.000','1997-08-18 00:00:00.000',3,477.9,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10634','FOLIG',4,'1997-08-15 00:00:00.000','1997-09-12 00:00:00.000','1997-08-21 00:00:00.000',3,487.38,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT INTO Orders  Values ('10635','MAGAA',8,'1997-08-18 00:00:00.000','1997-09-15 00:00:00.000','1997-08-21 00:00:00.000',3,47.46,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10636','WARTH',4,'1997-08-19 00:00:00.000','1997-09-16 00:00:00.000','1997-08-26 00:00:00.000',1,1.15,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10637','QUEEN',6,'1997-08-19 00:00:00.000','1997-09-16 00:00:00.000','1997-08-26 00:00:00.000',1,201.29,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10638','LINOD',3,'1997-08-20 00:00:00.000','1997-09-17 00:00:00.000','1997-09-01 00:00:00.000',1,158.44,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10639','SANTG',7,'1997-08-20 00:00:00.000','1997-09-17 00:00:00.000','1997-08-27 00:00:00.000',3,38.64,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('10640','WANDK',4,'1997-08-21 00:00:00.000','1997-09-18 00:00:00.000','1997-08-28 00:00:00.000',1,23.55,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10641','HILAA',4,'1997-08-22 00:00:00.000','1997-09-19 00:00:00.000','1997-08-26 00:00:00.000',2,179.61,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10642','SIMOB',7,'1997-08-22 00:00:00.000','1997-09-19 00:00:00.000','1997-09-05 00:00:00.000',3,41.89,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10643','ALFKI',6,'1997-08-25 00:00:00.000','1997-09-22 00:00:00.000','1997-09-02 00:00:00.000',1,29.46,'Alfreds Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('10644','WELLI',3,'1997-08-25 00:00:00.000','1997-09-22 00:00:00.000','1997-09-01 00:00:00.000',2,0.14,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10645','HANAR',4,'1997-08-26 00:00:00.000','1997-09-23 00:00:00.000','1997-09-02 00:00:00.000',1,12.41,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10646','HUNGO',9,'1997-08-27 00:00:00.000','1997-10-08 00:00:00.000','1997-09-03 00:00:00.000',3,142.33,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10647','QUEDE',4,'1997-08-27 00:00:00.000','1997-09-10 00:00:00.000','1997-09-03 00:00:00.000',2,45.54,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10648','RICAR',5,'1997-08-28 00:00:00.000','1997-10-09 00:00:00.000','1997-09-09 00:00:00.000',2,14.25,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10649','MAISD',5,'1997-08-28 00:00:00.000','1997-09-25 00:00:00.000','1997-08-29 00:00:00.000',3,6.2,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10650','FAMIA',5,'1997-08-29 00:00:00.000','1997-09-26 00:00:00.000','1997-09-03 00:00:00.000',3,176.81,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10651','WANDK',8,'1997-09-01 00:00:00.000','1997-09-29 00:00:00.000','1997-09-11 00:00:00.000',2,20.6,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10652','GOURL',4,'1997-09-01 00:00:00.000','1997-09-29 00:00:00.000','1997-09-08 00:00:00.000',2,7.14,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10653','FRANK',1,'1997-09-02 00:00:00.000','1997-09-30 00:00:00.000','1997-09-19 00:00:00.000',1,93.25,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10654','BERGS',5,'1997-09-02 00:00:00.000','1997-09-30 00:00:00.000','1997-09-11 00:00:00.000',1,55.26,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10655','REGGC',1,'1997-09-03 00:00:00.000','1997-10-01 00:00:00.000','1997-09-11 00:00:00.000',2,4.41,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10656','GREAL',6,'1997-09-04 00:00:00.000','1997-10-02 00:00:00.000','1997-09-10 00:00:00.000',1,57.15,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10657','SAVEA',2,'1997-09-04 00:00:00.000','1997-10-02 00:00:00.000','1997-09-15 00:00:00.000',2,352.69,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10658','QUICK',4,'1997-09-05 00:00:00.000','1997-10-03 00:00:00.000','1997-09-08 00:00:00.000',1,364.15,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10659','QUEEN',7,'1997-09-05 00:00:00.000','1997-10-03 00:00:00.000','1997-09-10 00:00:00.000',2,105.81,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10660','HUNGC',8,'1997-09-08 00:00:00.000','1997-10-06 00:00:00.000','1997-10-15 00:00:00.000',1,111.29,'Hungry Coyote Import Store','City Center Plaza 516 Main St.','Elgin','OR','97827','USA');
INSERT INTO Orders  Values ('10661','HUNGO',7,'1997-09-09 00:00:00.000','1997-10-07 00:00:00.000','1997-09-15 00:00:00.000',3,17.55,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10662','LONEP',3,'1997-09-09 00:00:00.000','1997-10-07 00:00:00.000','1997-09-18 00:00:00.000',2,1.28,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10663','BONAP',2,'1997-09-10 00:00:00.000','1997-09-24 00:00:00.000','1997-10-03 00:00:00.000',2,113.15,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10664','FURIB',1,'1997-09-10 00:00:00.000','1997-10-08 00:00:00.000','1997-09-19 00:00:00.000',3,1.27,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10665','LONEP',1,'1997-09-11 00:00:00.000','1997-10-09 00:00:00.000','1997-09-17 00:00:00.000',2,26.31,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10666','RICSU',7,'1997-09-12 00:00:00.000','1997-10-10 00:00:00.000','1997-09-22 00:00:00.000',2,232.42,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10667','ERNSH',7,'1997-09-12 00:00:00.000','1997-10-10 00:00:00.000','1997-09-19 00:00:00.000',1,78.09,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10668','WANDK',1,'1997-09-15 00:00:00.000','1997-10-13 00:00:00.000','1997-09-23 00:00:00.000',2,47.22,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('10669','SIMOB',2,'1997-09-15 00:00:00.000','1997-10-13 00:00:00.000','1997-09-22 00:00:00.000',1,24.39,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10670','FRANK',4,'1997-09-16 00:00:00.000','1997-10-14 00:00:00.000','1997-09-18 00:00:00.000',1,203.48,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10671','FRANR',1,'1997-09-17 00:00:00.000','1997-10-15 00:00:00.000','1997-09-24 00:00:00.000',1,30.34,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10672','BERGS',9,'1997-09-17 00:00:00.000','1997-10-01 00:00:00.000','1997-09-26 00:00:00.000',2,95.75,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10673','WILMK',2,'1997-09-18 00:00:00.000','1997-10-16 00:00:00.000','1997-09-19 00:00:00.000',1,22.76,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10674','ISLAT',4,'1997-09-18 00:00:00.000','1997-10-16 00:00:00.000','1997-09-30 00:00:00.000',2,0.9,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10675','FRANK',5,'1997-09-19 00:00:00.000','1997-10-17 00:00:00.000','1997-09-23 00:00:00.000',2,31.85,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10676','TORTU',2,'1997-09-22 00:00:00.000','1997-10-20 00:00:00.000','1997-09-29 00:00:00.000',2,2.01,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10677','ANTON',1,'1997-09-22 00:00:00.000','1997-10-20 00:00:00.000','1997-09-26 00:00:00.000',3,4.03,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10678','SAVEA',7,'1997-09-23 00:00:00.000','1997-10-21 00:00:00.000','1997-10-16 00:00:00.000',3,388.98,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10679','BLONP',8,'1997-09-23 00:00:00.000','1997-10-21 00:00:00.000','1997-09-30 00:00:00.000',3,27.94,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10680','OLDWO',1,'1997-09-24 00:00:00.000','1997-10-22 00:00:00.000','1997-09-26 00:00:00.000',1,26.61,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10681','GREAL',3,'1997-09-25 00:00:00.000','1997-10-23 00:00:00.000','1997-09-30 00:00:00.000',3,76.13,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10682','ANTON',3,'1997-09-25 00:00:00.000','1997-10-23 00:00:00.000','1997-10-01 00:00:00.000',2,36.13,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10683','DUMON',2,'1997-09-26 00:00:00.000','1997-10-24 00:00:00.000','1997-10-01 00:00:00.000',1,4.4,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10684','OTTIK',3,'1997-09-26 00:00:00.000','1997-10-24 00:00:00.000','1997-09-30 00:00:00.000',1,145.63,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10685','GOURL',4,'1997-09-29 00:00:00.000','1997-10-13 00:00:00.000','1997-10-03 00:00:00.000',2,33.75,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10686','PICCO',2,'1997-09-30 00:00:00.000','1997-10-28 00:00:00.000','1997-10-08 00:00:00.000',1,96.5,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10687','HUNGO',9,'1997-09-30 00:00:00.000','1997-10-28 00:00:00.000','1997-10-30 00:00:00.000',2,296.43,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10688','VAFFE',4,'1997-10-01 00:00:00.000','1997-10-15 00:00:00.000','1997-10-07 00:00:00.000',2,299.09,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10689','BERGS',1,'1997-10-01 00:00:00.000','1997-10-29 00:00:00.000','1997-10-07 00:00:00.000',2,13.42,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10690','HANAR',1,'1997-10-02 00:00:00.000','1997-10-30 00:00:00.000','1997-10-03 00:00:00.000',1,15.8,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10691','QUICK',2,'1997-10-03 00:00:00.000','1997-11-14 00:00:00.000','1997-10-22 00:00:00.000',2,810.05,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10692','ALFKI',4,'1997-10-03 00:00:00.000','1997-10-31 00:00:00.000','1997-10-13 00:00:00.000',2,61.02,'Alfred-s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('10693','WHITC',3,'1997-10-06 00:00:00.000','1997-10-20 00:00:00.000','1997-10-10 00:00:00.000',3,139.34,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10694','QUICK',8,'1997-10-06 00:00:00.000','1997-11-03 00:00:00.000','1997-10-09 00:00:00.000',3,398.36,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10695','WILMK',7,'1997-10-07 00:00:00.000','1997-11-18 00:00:00.000','1997-10-14 00:00:00.000',1,16.72,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10696','WHITC',8,'1997-10-08 00:00:00.000','1997-11-19 00:00:00.000','1997-10-14 00:00:00.000',3,102.55,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10697','LINOD',3,'1997-10-08 00:00:00.000','1997-11-05 00:00:00.000','1997-10-14 00:00:00.000',1,45.52,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10698','ERNSH',4,'1997-10-09 00:00:00.000','1997-11-06 00:00:00.000','1997-10-17 00:00:00.000',1,272.47,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10699','MORGK',3,'1997-10-09 00:00:00.000','1997-11-06 00:00:00.000','1997-10-13 00:00:00.000',3,0.58,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'4179','Germany');
INSERT INTO Orders  Values ('10700','SAVEA',3,'1997-10-10 00:00:00.000','1997-11-07 00:00:00.000','1997-10-16 00:00:00.000',1,65.1,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10701','HUNGO',6,'1997-10-13 00:00:00.000','1997-10-27 00:00:00.000','1997-10-15 00:00:00.000',3,220.31,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10702','ALFKI',4,'1997-10-13 00:00:00.000','1997-11-24 00:00:00.000','1997-10-21 00:00:00.000',1,23.94,'Alfred-s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('10703','FOLKO',6,'1997-10-14 00:00:00.000','1997-11-11 00:00:00.000','1997-10-20 00:00:00.000',2,152.3,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10704','QUEEN',6,'1997-10-14 00:00:00.000','1997-11-11 00:00:00.000','1997-11-07 00:00:00.000',1,4.78,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10705','HILAA',9,'1997-10-15 00:00:00.000','1997-11-12 00:00:00.000','1997-11-18 00:00:00.000',2,3.52,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10706','OLDWO',8,'1997-10-16 00:00:00.000','1997-11-13 00:00:00.000','1997-10-21 00:00:00.000',3,135.63,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10707','AROUT',4,'1997-10-16 00:00:00.000','1997-10-30 00:00:00.000','1997-10-23 00:00:00.000',3,21.74,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10708','THEBI',6,'1997-10-17 00:00:00.000','1997-11-28 00:00:00.000','1997-11-05 00:00:00.000',2,2.96,'The Big Cheese','89 Jefferson Way Suite 2','Portland','OR','97201','USA');
INSERT INTO Orders  Values ('10709','GOURL',1,'1997-10-17 00:00:00.000','1997-11-14 00:00:00.000','1997-11-20 00:00:00.000',3,210.8,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10710','FRANS',1,'1997-10-20 00:00:00.000','1997-11-17 00:00:00.000','1997-10-23 00:00:00.000',1,4.98,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('10711','SAVEA',5,'1997-10-21 00:00:00.000','1997-12-02 00:00:00.000','1997-10-29 00:00:00.000',2,52.41,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10712','HUNGO',3,'1997-10-21 00:00:00.000','1997-11-18 00:00:00.000','1997-10-31 00:00:00.000',1,89.93,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10713','SAVEA',1,'1997-10-22 00:00:00.000','1997-11-19 00:00:00.000','1997-10-24 00:00:00.000',1,167.05,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10714','SAVEA',5,'1997-10-22 00:00:00.000','1997-11-19 00:00:00.000','1997-10-27 00:00:00.000',3,24.49,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10715','BONAP',3,'1997-10-23 00:00:00.000','1997-11-06 00:00:00.000','1997-10-29 00:00:00.000',1,63.2,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10716','RANCH',4,'1997-10-24 00:00:00.000','1997-11-21 00:00:00.000','1997-10-27 00:00:00.000',2,22.57,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10717','FRANK',1,'1997-10-24 00:00:00.000','1997-11-21 00:00:00.000','1997-10-29 00:00:00.000',2,59.25,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10718','KOENE',1,'1997-10-27 00:00:00.000','1997-11-24 00:00:00.000','1997-10-29 00:00:00.000',3,170.88,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10719','LETSS',8,'1997-10-27 00:00:00.000','1997-11-24 00:00:00.000','1997-11-05 00:00:00.000',2,51.44,'Let-s Stop N Shop','87 Polk St. Suite 5','San Francisco','CA','94117','USA');
INSERT INTO Orders  Values ('10720','QUEDE',8,'1997-10-28 00:00:00.000','1997-11-11 00:00:00.000','1997-11-05 00:00:00.000',2,9.53,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10721','QUICK',5,'1997-10-29 00:00:00.000','1997-11-26 00:00:00.000','1997-10-31 00:00:00.000',3,48.92,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10722','SAVEA',8,'1997-10-29 00:00:00.000','1997-12-10 00:00:00.000','1997-11-04 00:00:00.000',1,74.58,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10723','WHITC',3,'1997-10-30 00:00:00.000','1997-11-27 00:00:00.000','1997-11-25 00:00:00.000',1,21.72,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10724','MEREP',8,'1997-10-30 00:00:00.000','1997-12-11 00:00:00.000','1997-11-05 00:00:00.000',2,57.75,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT INTO Orders  Values ('10725','FAMIA',4,'1997-10-31 00:00:00.000','1997-11-28 00:00:00.000','1997-11-05 00:00:00.000',3,10.83,'Familia Arquibaldo','Rua Ors, 92','Sao Paulo','SP','05442-030','Brazil');
INSERT INTO Orders  Values ('10726','EASTC',4,'1997-11-03 00:00:00.000','1997-11-17 00:00:00.000','1997-12-05 00:00:00.000',1,16.56,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('10727','REGGC',2,'1997-11-03 00:00:00.000','1997-12-01 00:00:00.000','1997-12-05 00:00:00.000',1,89.9,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10728','QUEEN',4,'1997-11-04 00:00:00.000','1997-12-02 00:00:00.000','1997-11-11 00:00:00.000',2,58.33,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10729','LINOD',8,'1997-11-04 00:00:00.000','1997-12-16 00:00:00.000','1997-11-14 00:00:00.000',3,141.06,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10730','BONAP',5,'1997-11-05 00:00:00.000','1997-12-03 00:00:00.000','1997-11-14 00:00:00.000',1,20.12,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10731','CHOPS',7,'1997-11-06 00:00:00.000','1997-12-04 00:00:00.000','1997-11-14 00:00:00.000',1,96.65,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10732','BONAP',3,'1997-11-06 00:00:00.000','1997-12-04 00:00:00.000','1997-11-07 00:00:00.000',1,16.97,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10733','BERGS',1,'1997-11-07 00:00:00.000','1997-12-05 00:00:00.000','1997-11-10 00:00:00.000',3,110.11,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10734','GOURL',2,'1997-11-07 00:00:00.000','1997-12-05 00:00:00.000','1997-11-12 00:00:00.000',3,1.63,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10735','LETSS',6,'1997-11-10 00:00:00.000','1997-12-08 00:00:00.000','1997-11-21 00:00:00.000',2,45.97,'Let-s Stop N Shop','87 Polk St. Suite 5','San Francisco','CA','94117','USA');
INSERT INTO Orders  Values ('10736','HUNGO',9,'1997-11-11 00:00:00.000','1997-12-09 00:00:00.000','1997-11-21 00:00:00.000',2,44.1,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10737','VINET',2,'1997-11-11 00:00:00.000','1997-12-09 00:00:00.000','1997-11-18 00:00:00.000',2,7.79,'Vins et alcools Chevalier','59 rue de l-Abbaye','Reims',NULL,'51100','France');
INSERT INTO Orders  Values ('10738','SPECD',2,'1997-11-12 00:00:00.000','1997-12-10 00:00:00.000','1997-11-18 00:00:00.000',1,2.91,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT INTO Orders  Values ('10739','VINET',3,'1997-11-12 00:00:00.000','1997-12-10 00:00:00.000','1997-11-17 00:00:00.000',3,11.08,'Vins et alcools Chevalier','59 rue de l-Abbaye','Reims',NULL,'51100','France');
INSERT INTO Orders  Values ('10740','WHITC',4,'1997-11-13 00:00:00.000','1997-12-11 00:00:00.000','1997-11-25 00:00:00.000',2,81.88,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10741','AROUT',4,'1997-11-14 00:00:00.000','1997-11-28 00:00:00.000','1997-11-18 00:00:00.000',3,10.96,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10742','BOTTM',3,'1997-11-14 00:00:00.000','1997-12-12 00:00:00.000','1997-11-18 00:00:00.000',3,243.73,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10743','AROUT',1,'1997-11-17 00:00:00.000','1997-12-15 00:00:00.000','1997-11-21 00:00:00.000',2,23.72,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10744','VAFFE',6,'1997-11-17 00:00:00.000','1997-12-15 00:00:00.000','1997-11-24 00:00:00.000',1,69.19,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10745','QUICK',9,'1997-11-18 00:00:00.000','1997-12-16 00:00:00.000','1997-11-27 00:00:00.000',1,3.52,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10746','CHOPS',1,'1997-11-19 00:00:00.000','1997-12-17 00:00:00.000','1997-11-21 00:00:00.000',3,31.43,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10747','PICCO',6,'1997-11-19 00:00:00.000','1997-12-17 00:00:00.000','1997-11-26 00:00:00.000',1,117.33,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10748','SAVEA',3,'1997-11-20 00:00:00.000','1997-12-18 00:00:00.000','1997-11-28 00:00:00.000',1,232.55,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10749','ISLAT',4,'1997-11-20 00:00:00.000','1997-12-18 00:00:00.000','1997-12-19 00:00:00.000',2,61.53,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10750','WARTH',9,'1997-11-21 00:00:00.000','1997-12-19 00:00:00.000','1997-11-24 00:00:00.000',1,79.3,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10751','RICSU',3,'1997-11-24 00:00:00.000','1997-12-22 00:00:00.000','1997-12-03 00:00:00.000',3,130.79,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10752','NORTS',2,'1997-11-24 00:00:00.000','1997-12-22 00:00:00.000','1997-11-28 00:00:00.000',3,1.39,'North/South','South House 300 Queensbridge','London',NULL,'SW7 1RZ','UK');
INSERT INTO Orders  Values ('10753','FRANS',3,'1997-11-25 00:00:00.000','1997-12-23 00:00:00.000','1997-11-27 00:00:00.000',1,7.7,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('10754','MAGAA',6,'1997-11-25 00:00:00.000','1997-12-23 00:00:00.000','1997-11-27 00:00:00.000',3,2.38,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10755','BONAP',4,'1997-11-26 00:00:00.000','1997-12-24 00:00:00.000','1997-11-28 00:00:00.000',2,16.71,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10756','SPLIR',8,'1997-11-27 00:00:00.000','1997-12-25 00:00:00.000','1997-12-02 00:00:00.000',2,73.21,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10757','SAVEA',6,'1997-11-27 00:00:00.000','1997-12-25 00:00:00.000','1997-12-15 00:00:00.000',1,8.19,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10758','RICSU',3,'1997-11-28 00:00:00.000','1997-12-26 00:00:00.000','1997-12-04 00:00:00.000',3,138.17,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10759','ANATR',3,'1997-11-28 00:00:00.000','1997-12-26 00:00:00.000','1997-12-12 00:00:00.000',3,11.99,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'5021','Mexico');
INSERT INTO Orders  Values ('10760','MAISD',4,'1997-12-01 00:00:00.000','1997-12-29 00:00:00.000','1997-12-10 00:00:00.000',1,155.64,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10761','RATTC',5,'1997-12-02 00:00:00.000','1997-12-30 00:00:00.000','1997-12-08 00:00:00.000',2,18.66,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10762','FOLKO',3,'1997-12-02 00:00:00.000','1997-12-30 00:00:00.000','1997-12-09 00:00:00.000',1,328.74,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10763','FOLIG',3,'1997-12-03 00:00:00.000','1997-12-31 00:00:00.000','1997-12-08 00:00:00.000',3,37.35,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT INTO Orders  Values ('10764','ERNSH',6,'1997-12-03 00:00:00.000','1997-12-31 00:00:00.000','1997-12-08 00:00:00.000',3,145.45,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10765','QUICK',3,'1997-12-04 00:00:00.000','1998-01-01 00:00:00.000','1997-12-09 00:00:00.000',3,42.74,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10766','OTTIK',4,'1997-12-05 00:00:00.000','1998-01-02 00:00:00.000','1997-12-09 00:00:00.000',1,157.55,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10767','SUPRD',4,'1997-12-05 00:00:00.000','1998-01-02 00:00:00.000','1997-12-15 00:00:00.000',3,1.59,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10768','AROUT',3,'1997-12-08 00:00:00.000','1998-01-05 00:00:00.000','1997-12-15 00:00:00.000',2,146.32,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10769','VAFFE',3,'1997-12-08 00:00:00.000','1998-01-05 00:00:00.000','1997-12-12 00:00:00.000',1,65.06,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10770','HANAR',8,'1997-12-09 00:00:00.000','1998-01-06 00:00:00.000','1997-12-17 00:00:00.000',3,5.32,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10771','ERNSH',9,'1997-12-10 00:00:00.000','1998-01-07 00:00:00.000','1998-01-02 00:00:00.000',2,11.19,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10772','LEHMS',3,'1997-12-10 00:00:00.000','1998-01-07 00:00:00.000','1997-12-19 00:00:00.000',2,91.28,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10773','ERNSH',1,'1997-12-11 00:00:00.000','1998-01-08 00:00:00.000','1997-12-16 00:00:00.000',3,96.43,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10774','FOLKO',4,'1997-12-11 00:00:00.000','1997-12-25 00:00:00.000','1997-12-12 00:00:00.000',1,48.2,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10775','THECR',7,'1997-12-12 00:00:00.000','1998-01-09 00:00:00.000','1997-12-26 00:00:00.000',1,20.25,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','USA');
INSERT INTO Orders  Values ('10776','ERNSH',1,'1997-12-15 00:00:00.000','1998-01-12 00:00:00.000','1997-12-18 00:00:00.000',3,351.53,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10777','GOURL',7,'1997-12-15 00:00:00.000','1997-12-29 00:00:00.000','1998-01-21 00:00:00.000',2,3.01,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10778','BERGS',3,'1997-12-16 00:00:00.000','1998-01-13 00:00:00.000','1997-12-24 00:00:00.000',1,6.79,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10779','MORGK',3,'1997-12-16 00:00:00.000','1998-01-13 00:00:00.000','1998-01-14 00:00:00.000',2,58.13,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'4179','Germany');
INSERT INTO Orders  Values ('10780','LILAS',2,'1997-12-16 00:00:00.000','1997-12-30 00:00:00.000','1997-12-25 00:00:00.000',1,42.13,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10781','WARTH',2,'1997-12-17 00:00:00.000','1998-01-14 00:00:00.000','1997-12-19 00:00:00.000',3,73.16,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('10782','CACTU',9,'1997-12-17 00:00:00.000','1998-01-14 00:00:00.000','1997-12-22 00:00:00.000',3,1.1,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10783','HANAR',4,'1997-12-18 00:00:00.000','1998-01-15 00:00:00.000','1997-12-19 00:00:00.000',2,124.98,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10784','MAGAA',4,'1997-12-18 00:00:00.000','1998-01-15 00:00:00.000','1997-12-22 00:00:00.000',3,70.09,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10785','GROSR',1,'1997-12-18 00:00:00.000','1998-01-15 00:00:00.000','1997-12-24 00:00:00.000',3,1.51,'GROSELLA-Restaurante','5 Ave. Los Palos Grandes','Caracas','DF','1081','Venezuela');
INSERT INTO Orders  Values ('10786','QUEEN',8,'1997-12-19 00:00:00.000','1998-01-16 00:00:00.000','1997-12-23 00:00:00.000',1,110.87,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10787','LAMAI',2,'1997-12-19 00:00:00.000','1998-01-02 00:00:00.000','1997-12-26 00:00:00.000',1,249.93,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10788','QUICK',1,'1997-12-22 00:00:00.000','1998-01-19 00:00:00.000','1998-01-19 00:00:00.000',2,42.7,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10789','FOLIG',1,'1997-12-22 00:00:00.000','1998-01-19 00:00:00.000','1997-12-31 00:00:00.000',2,100.6,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT INTO Orders  Values ('10790','GOURL',6,'1997-12-22 00:00:00.000','1998-01-19 00:00:00.000','1997-12-26 00:00:00.000',1,28.23,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10791','FRANK',6,'1997-12-23 00:00:00.000','1998-01-20 00:00:00.000','1998-01-01 00:00:00.000',2,16.85,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10792','WOLZA',1,'1997-12-23 00:00:00.000','1998-01-20 00:00:00.000','1997-12-31 00:00:00.000',3,23.79,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10793','AROUT',3,'1997-12-24 00:00:00.000','1998-01-21 00:00:00.000','1998-01-08 00:00:00.000',3,4.52,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10794','QUEDE',6,'1997-12-24 00:00:00.000','1998-01-21 00:00:00.000','1998-01-02 00:00:00.000',1,21.49,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10795','ERNSH',8,'1997-12-24 00:00:00.000','1998-01-21 00:00:00.000','1998-01-20 00:00:00.000',2,126.66,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10796','HILAA',3,'1997-12-25 00:00:00.000','1998-01-22 00:00:00.000','1998-01-14 00:00:00.000',1,26.52,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10797','DRACD',7,'1997-12-25 00:00:00.000','1998-01-22 00:00:00.000','1998-01-05 00:00:00.000',2,33.35,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('10798','ISLAT',2,'1997-12-26 00:00:00.000','1998-01-23 00:00:00.000','1998-01-05 00:00:00.000',1,2.33,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10799','KOENE',9,'1997-12-26 00:00:00.000','1998-02-06 00:00:00.000','1998-01-05 00:00:00.000',3,30.76,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10800','SEVES',1,'1997-12-26 00:00:00.000','1998-01-23 00:00:00.000','1998-01-05 00:00:00.000',3,137.44,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10801','BOLID',4,'1997-12-29 00:00:00.000','1998-01-26 00:00:00.000','1997-12-31 00:00:00.000',2,97.09,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT INTO Orders  Values ('10802','SIMOB',4,'1997-12-29 00:00:00.000','1998-01-26 00:00:00.000','1998-01-02 00:00:00.000',2,257.26,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('10803','WELLI',4,'1997-12-30 00:00:00.000','1998-01-27 00:00:00.000','1998-01-06 00:00:00.000',1,55.23,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10804','SEVES',6,'1997-12-30 00:00:00.000','1998-01-27 00:00:00.000','1998-01-07 00:00:00.000',2,27.33,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10805','THEBI',2,'1997-12-30 00:00:00.000','1998-01-27 00:00:00.000','1998-01-09 00:00:00.000',3,237.34,'The Big Cheese','89 Jefferson Way Suite 2','Portland','OR','97201','USA');
INSERT INTO Orders  Values ('10806','VICTE',3,'1997-12-31 00:00:00.000','1998-01-28 00:00:00.000','1998-01-05 00:00:00.000',2,22.11,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10807','FRANS',4,'1997-12-31 00:00:00.000','1998-01-28 00:00:00.000','1998-01-30 00:00:00.000',1,1.36,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('10808','OLDWO',2,'1998-01-01 00:00:00.000','1998-01-29 00:00:00.000','1998-01-09 00:00:00.000',3,45.53,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10809','WELLI',7,'1998-01-01 00:00:00.000','1998-01-29 00:00:00.000','1998-01-07 00:00:00.000',1,4.87,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10810','LAUGB',2,'1998-01-01 00:00:00.000','1998-01-29 00:00:00.000','1998-01-07 00:00:00.000',3,4.33,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT INTO Orders  Values ('10811','LINOD',8,'1998-01-02 00:00:00.000','1998-01-30 00:00:00.000','1998-01-08 00:00:00.000',1,31.22,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10812','REGGC',5,'1998-01-02 00:00:00.000','1998-01-30 00:00:00.000','1998-01-12 00:00:00.000',1,59.78,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10813','RICAR',1,'1998-01-05 00:00:00.000','1998-02-02 00:00:00.000','1998-01-09 00:00:00.000',1,47.38,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10814','VICTE',3,'1998-01-05 00:00:00.000','1998-02-02 00:00:00.000','1998-01-14 00:00:00.000',3,130.94,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10815','SAVEA',2,'1998-01-05 00:00:00.000','1998-02-02 00:00:00.000','1998-01-14 00:00:00.000',3,14.62,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10816','GREAL',4,'1998-01-06 00:00:00.000','1998-02-03 00:00:00.000','1998-02-04 00:00:00.000',2,719.78,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10817','KOENE',3,'1998-01-06 00:00:00.000','1998-01-20 00:00:00.000','1998-01-13 00:00:00.000',2,306.07,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10818','MAGAA',7,'1998-01-07 00:00:00.000','1998-02-04 00:00:00.000','1998-01-12 00:00:00.000',3,65.48,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10819','CACTU',2,'1998-01-07 00:00:00.000','1998-02-04 00:00:00.000','1998-01-16 00:00:00.000',3,19.76,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10820','RATTC',3,'1998-01-07 00:00:00.000','1998-02-04 00:00:00.000','1998-01-13 00:00:00.000',2,37.52,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10821','SPLIR',1,'1998-01-08 00:00:00.000','1998-02-05 00:00:00.000','1998-01-15 00:00:00.000',1,36.68,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10822','TRAIH',6,'1998-01-08 00:00:00.000','1998-02-05 00:00:00.000','1998-01-16 00:00:00.000',3,7,'Trail-s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','USA');
INSERT INTO Orders  Values ('10823','LILAS',5,'1998-01-09 00:00:00.000','1998-02-06 00:00:00.000','1998-01-13 00:00:00.000',2,163.97,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10824','FOLKO',8,'1998-01-09 00:00:00.000','1998-02-06 00:00:00.000','1998-01-30 00:00:00.000',1,1.23,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10825','DRACD',1,'1998-01-09 00:00:00.000','1998-02-06 00:00:00.000','1998-01-14 00:00:00.000',1,79.25,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('10826','BLONP',6,'1998-01-12 00:00:00.000','1998-02-09 00:00:00.000','1998-02-06 00:00:00.000',1,7.09,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT INTO Orders  Values ('10827','BONAP',1,'1998-01-12 00:00:00.000','1998-01-26 00:00:00.000','1998-02-06 00:00:00.000',2,63.54,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10828','RANCH',9,'1998-01-13 00:00:00.000','1998-01-27 00:00:00.000','1998-02-04 00:00:00.000',1,90.85,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10829','ISLAT',9,'1998-01-13 00:00:00.000','1998-02-10 00:00:00.000','1998-01-23 00:00:00.000',1,154.72,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10830','TRADH',4,'1998-01-13 00:00:00.000','1998-02-24 00:00:00.000','1998-01-21 00:00:00.000',2,81.83,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10831','SANTG',3,'1998-01-14 00:00:00.000','1998-02-11 00:00:00.000','1998-01-23 00:00:00.000',2,72.19,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('10832','LAMAI',2,'1998-01-14 00:00:00.000','1998-02-11 00:00:00.000','1998-01-19 00:00:00.000',2,43.26,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10833','OTTIK',6,'1998-01-15 00:00:00.000','1998-02-12 00:00:00.000','1998-01-23 00:00:00.000',2,71.49,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('10834','TRADH',1,'1998-01-15 00:00:00.000','1998-02-12 00:00:00.000','1998-01-19 00:00:00.000',3,29.78,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10835','ALFKI',1,'1998-01-15 00:00:00.000','1998-02-12 00:00:00.000','1998-01-21 00:00:00.000',3,69.53,'Alfred-s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('10836','ERNSH',7,'1998-01-16 00:00:00.000','1998-02-13 00:00:00.000','1998-01-21 00:00:00.000',1,411.88,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10837','BERGS',9,'1998-01-16 00:00:00.000','1998-02-13 00:00:00.000','1998-01-23 00:00:00.000',3,13.32,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10838','LINOD',3,'1998-01-19 00:00:00.000','1998-02-16 00:00:00.000','1998-01-23 00:00:00.000',3,59.28,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10839','TRADH',3,'1998-01-19 00:00:00.000','1998-02-16 00:00:00.000','1998-01-22 00:00:00.000',3,35.43,'Tradiao Hipermercados','Av. Ins de Castro, 414','Sao Paulo','SP','05634-030','Brazil');
INSERT INTO Orders  Values ('10840','LINOD',4,'1998-01-19 00:00:00.000','1998-03-02 00:00:00.000','1998-02-16 00:00:00.000',2,2.71,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10841','SUPRD',5,'1998-01-20 00:00:00.000','1998-02-17 00:00:00.000','1998-01-29 00:00:00.000',2,424.3,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10842','TORTU',1,'1998-01-20 00:00:00.000','1998-02-17 00:00:00.000','1998-01-29 00:00:00.000',3,54.42,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10843','VICTE',4,'1998-01-21 00:00:00.000','1998-02-18 00:00:00.000','1998-01-26 00:00:00.000',2,9.26,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10844','PICCO',8,'1998-01-21 00:00:00.000','1998-02-18 00:00:00.000','1998-01-26 00:00:00.000',2,25.22,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('10845','QUICK',8,'1998-01-21 00:00:00.000','1998-02-04 00:00:00.000','1998-01-30 00:00:00.000',1,212.98,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10846','SUPRD',2,'1998-01-22 00:00:00.000','1998-03-05 00:00:00.000','1998-01-23 00:00:00.000',3,56.46,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10847','SAVEA',4,'1998-01-22 00:00:00.000','1998-02-05 00:00:00.000','1998-02-10 00:00:00.000',3,487.57,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10848','CONSH',7,'1998-01-23 00:00:00.000','1998-02-20 00:00:00.000','1998-01-29 00:00:00.000',2,38.24,'Consolidated Holdings','Berkeley Gardens 12 Brewery','London',NULL,'WX1 6LT','UK');
INSERT INTO Orders  Values ('10849','KOENE',9,'1998-01-23 00:00:00.000','1998-02-20 00:00:00.000','1998-01-30 00:00:00.000',2,0.56,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10850','VICTE',1,'1998-01-23 00:00:00.000','1998-03-06 00:00:00.000','1998-01-30 00:00:00.000',1,49.19,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT INTO Orders  Values ('10851','RICAR',5,'1998-01-26 00:00:00.000','1998-02-23 00:00:00.000','1998-02-02 00:00:00.000',1,160.55,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10852','RATTC',8,'1998-01-26 00:00:00.000','1998-02-09 00:00:00.000','1998-01-30 00:00:00.000',1,174.05,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10853','BLAUS',9,'1998-01-27 00:00:00.000','1998-02-24 00:00:00.000','1998-02-03 00:00:00.000',2,53.83,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10854','ERNSH',3,'1998-01-27 00:00:00.000','1998-02-24 00:00:00.000','1998-02-05 00:00:00.000',2,100.22,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10855','OLDWO',3,'1998-01-27 00:00:00.000','1998-02-24 00:00:00.000','1998-02-04 00:00:00.000',1,170.97,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10856','ANTON',3,'1998-01-28 00:00:00.000','1998-02-25 00:00:00.000','1998-02-10 00:00:00.000',2,58.43,'Antonio Moreno Taquera','Mataderos 2312','Mxico D.F.',NULL,'5023','Mexico');
INSERT INTO Orders  Values ('10857','BERGS',8,'1998-01-28 00:00:00.000','1998-02-25 00:00:00.000','1998-02-06 00:00:00.000',2,188.85,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10858','LACOR',2,'1998-01-29 00:00:00.000','1998-02-26 00:00:00.000','1998-02-03 00:00:00.000',1,52.51,'La corne d-abondance','67, avenue de l-Europe','Versailles',NULL,'78000','France');
INSERT INTO Orders  Values ('10859','FRANK',1,'1998-01-29 00:00:00.000','1998-02-26 00:00:00.000','1998-02-02 00:00:00.000',2,76.1,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10860','FRANR',3,'1998-01-29 00:00:00.000','1998-02-26 00:00:00.000','1998-02-04 00:00:00.000',3,19.26,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10861','WHITC',4,'1998-01-30 00:00:00.000','1998-02-27 00:00:00.000','1998-02-17 00:00:00.000',2,14.93,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10862','LEHMS',8,'1998-01-30 00:00:00.000','1998-03-13 00:00:00.000','1998-02-02 00:00:00.000',2,53.23,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10863','HILAA',4,'1998-02-02 00:00:00.000','1998-03-02 00:00:00.000','1998-02-17 00:00:00.000',2,30.26,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10864','AROUT',4,'1998-02-02 00:00:00.000','1998-03-02 00:00:00.000','1998-02-09 00:00:00.000',2,3.04,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10865','QUICK',2,'1998-02-02 00:00:00.000','1998-02-16 00:00:00.000','1998-02-12 00:00:00.000',1,348.14,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10866','BERGS',5,'1998-02-03 00:00:00.000','1998-03-03 00:00:00.000','1998-02-12 00:00:00.000',1,109.11,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10867','LONEP',6,'1998-02-03 00:00:00.000','1998-03-17 00:00:00.000','1998-02-11 00:00:00.000',1,1.93,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10868','QUEEN',7,'1998-02-04 00:00:00.000','1998-03-04 00:00:00.000','1998-02-23 00:00:00.000',2,191.27,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10869','SEVES',5,'1998-02-04 00:00:00.000','1998-03-04 00:00:00.000','1998-02-09 00:00:00.000',1,143.28,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','UK');
INSERT INTO Orders  Values ('10870','WOLZA',5,'1998-02-04 00:00:00.000','1998-03-04 00:00:00.000','1998-02-13 00:00:00.000',3,12.04,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10871','BONAP',9,'1998-02-05 00:00:00.000','1998-03-05 00:00:00.000','1998-02-10 00:00:00.000',2,112.27,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10872','GODOS',5,'1998-02-05 00:00:00.000','1998-03-05 00:00:00.000','1998-02-09 00:00:00.000',2,175.32,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10873','WILMK',4,'1998-02-06 00:00:00.000','1998-03-06 00:00:00.000','1998-02-09 00:00:00.000',1,0.82,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10874','GODOS',5,'1998-02-06 00:00:00.000','1998-03-06 00:00:00.000','1998-02-11 00:00:00.000',2,19.58,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10875','BERGS',4,'1998-02-06 00:00:00.000','1998-03-06 00:00:00.000','1998-03-03 00:00:00.000',2,32.37,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10876','BONAP',7,'1998-02-09 00:00:00.000','1998-03-09 00:00:00.000','1998-02-12 00:00:00.000',3,60.42,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10877','RICAR',1,'1998-02-09 00:00:00.000','1998-03-09 00:00:00.000','1998-02-19 00:00:00.000',1,38.06,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('10878','QUICK',4,'1998-02-10 00:00:00.000','1998-03-10 00:00:00.000','1998-02-12 00:00:00.000',1,46.69,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10879','WILMK',3,'1998-02-10 00:00:00.000','1998-03-10 00:00:00.000','1998-02-12 00:00:00.000',3,8.5,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10880','FOLKO',7,'1998-02-10 00:00:00.000','1998-03-24 00:00:00.000','1998-02-18 00:00:00.000',1,88.01,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10881','CACTU',4,'1998-02-11 00:00:00.000','1998-03-11 00:00:00.000','1998-02-18 00:00:00.000',1,2.84,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10882','SAVEA',4,'1998-02-11 00:00:00.000','1998-03-11 00:00:00.000','1998-02-20 00:00:00.000',3,23.1,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10883','LONEP',8,'1998-02-12 00:00:00.000','1998-03-12 00:00:00.000','1998-02-20 00:00:00.000',3,0.53,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('10884','LETSS',4,'1998-02-12 00:00:00.000','1998-03-12 00:00:00.000','1998-02-13 00:00:00.000',2,90.97,'Let-s Stop N Shop','87 Polk St. Suite 5','San Francisco','CA','94117','USA');
INSERT INTO Orders  Values ('10885','SUPRD',6,'1998-02-12 00:00:00.000','1998-03-12 00:00:00.000','1998-02-18 00:00:00.000',3,5.64,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10886','HANAR',1,'1998-02-13 00:00:00.000','1998-03-13 00:00:00.000','1998-03-02 00:00:00.000',1,4.99,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10887','GALED',8,'1998-02-13 00:00:00.000','1998-03-13 00:00:00.000','1998-02-16 00:00:00.000',3,1.25,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT INTO Orders  Values ('10888','GODOS',1,'1998-02-16 00:00:00.000','1998-03-16 00:00:00.000','1998-02-23 00:00:00.000',2,51.87,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10889','RATTC',9,'1998-02-16 00:00:00.000','1998-03-16 00:00:00.000','1998-02-23 00:00:00.000',3,280.61,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10890','DUMON',7,'1998-02-16 00:00:00.000','1998-03-16 00:00:00.000','1998-02-18 00:00:00.000',1,32.76,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10891','LEHMS',7,'1998-02-17 00:00:00.000','1998-03-17 00:00:00.000','1998-02-19 00:00:00.000',2,20.37,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10892','MAISD',4,'1998-02-17 00:00:00.000','1998-03-17 00:00:00.000','1998-02-19 00:00:00.000',2,120.27,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10893','KOENE',9,'1998-02-18 00:00:00.000','1998-03-18 00:00:00.000','1998-02-20 00:00:00.000',2,77.78,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('10894','SAVEA',1,'1998-02-18 00:00:00.000','1998-03-18 00:00:00.000','1998-02-20 00:00:00.000',1,116.13,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10895','ERNSH',3,'1998-02-18 00:00:00.000','1998-03-18 00:00:00.000','1998-02-23 00:00:00.000',1,162.75,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10896','MAISD',7,'1998-02-19 00:00:00.000','1998-03-19 00:00:00.000','1998-02-27 00:00:00.000',3,32.45,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10897','HUNGO',3,'1998-02-19 00:00:00.000','1998-03-19 00:00:00.000','1998-02-25 00:00:00.000',2,603.54,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10898','OCEAN',4,'1998-02-20 00:00:00.000','1998-03-20 00:00:00.000','1998-03-06 00:00:00.000',2,1.27,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585 Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10899','LILAS',5,'1998-02-20 00:00:00.000','1998-03-20 00:00:00.000','1998-02-26 00:00:00.000',3,1.21,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10900','WELLI',1,'1998-02-20 00:00:00.000','1998-03-20 00:00:00.000','1998-03-04 00:00:00.000',2,1.66,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10901','HILAA',4,'1998-02-23 00:00:00.000','1998-03-23 00:00:00.000','1998-02-26 00:00:00.000',1,62.09,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10902','FOLKO',1,'1998-02-23 00:00:00.000','1998-03-23 00:00:00.000','1998-03-03 00:00:00.000',1,44.15,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10903','HANAR',3,'1998-02-24 00:00:00.000','1998-03-24 00:00:00.000','1998-03-04 00:00:00.000',3,36.71,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10904','WHITC',3,'1998-02-24 00:00:00.000','1998-03-24 00:00:00.000','1998-02-27 00:00:00.000',3,162.95,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('10905','WELLI',9,'1998-02-24 00:00:00.000','1998-03-24 00:00:00.000','1998-03-06 00:00:00.000',2,13.72,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10906','WOLZA',4,'1998-02-25 00:00:00.000','1998-03-11 00:00:00.000','1998-03-03 00:00:00.000',3,26.29,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10907','SPECD',6,'1998-02-25 00:00:00.000','1998-03-25 00:00:00.000','1998-02-27 00:00:00.000',3,9.19,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT INTO Orders  Values ('10908','REGGC',4,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-06 00:00:00.000',2,32.96,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10909','SANTG',1,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-10 00:00:00.000',2,53.05,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('10910','WILMK',1,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-04 00:00:00.000',3,38.11,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('10911','GODOS',3,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-05 00:00:00.000',1,38.19,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10912','HUNGO',2,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-18 00:00:00.000',2,580.91,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10913','QUEEN',4,'1998-02-26 00:00:00.000','1998-03-26 00:00:00.000','1998-03-04 00:00:00.000',1,33.05,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10914','QUEEN',6,'1998-02-27 00:00:00.000','1998-03-27 00:00:00.000','1998-03-02 00:00:00.000',1,21.19,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10915','TORTU',2,'1998-02-27 00:00:00.000','1998-03-27 00:00:00.000','1998-03-02 00:00:00.000',2,3.51,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10916','RANCH',1,'1998-02-27 00:00:00.000','1998-03-27 00:00:00.000','1998-03-09 00:00:00.000',2,63.77,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10917','ROMEY',4,'1998-03-02 00:00:00.000','1998-03-30 00:00:00.000','1998-03-11 00:00:00.000',2,8.29,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT INTO Orders  Values ('10918','BOTTM',3,'1998-03-02 00:00:00.000','1998-03-30 00:00:00.000','1998-03-11 00:00:00.000',3,48.83,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10919','LINOD',2,'1998-03-02 00:00:00.000','1998-03-30 00:00:00.000','1998-03-04 00:00:00.000',2,19.8,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10920','AROUT',4,'1998-03-03 00:00:00.000','1998-03-31 00:00:00.000','1998-03-09 00:00:00.000',2,29.61,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10921','VAFFE',1,'1998-03-03 00:00:00.000','1998-04-14 00:00:00.000','1998-03-09 00:00:00.000',1,176.48,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10922','HANAR',5,'1998-03-03 00:00:00.000','1998-03-31 00:00:00.000','1998-03-05 00:00:00.000',3,62.74,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10923','LAMAI',7,'1998-03-03 00:00:00.000','1998-04-14 00:00:00.000','1998-03-13 00:00:00.000',3,68.26,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('10924','BERGS',3,'1998-03-04 00:00:00.000','1998-04-01 00:00:00.000','1998-04-08 00:00:00.000',2,151.52,'Berglunds snabbkp','Berguvsvgen 8','Lule',NULL,'S-958 22','Sweden');
INSERT INTO Orders  Values ('10925','HANAR',3,'1998-03-04 00:00:00.000','1998-04-01 00:00:00.000','1998-03-13 00:00:00.000',1,2.27,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10926','ANATR',4,'1998-03-04 00:00:00.000','1998-04-01 00:00:00.000','1998-03-11 00:00:00.000',3,39.92,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'5021','Mexico');
INSERT INTO Orders  Values ('10927','LACOR',4,'1998-03-05 00:00:00.000','1998-04-02 00:00:00.000','1998-04-08 00:00:00.000',1,19.79,'La corne d-abondance','67, avenue de l-Europe','Versailles',NULL,'78000','France');
INSERT INTO Orders  Values ('10928','GALED',1,'1998-03-05 00:00:00.000','1998-04-02 00:00:00.000','1998-03-18 00:00:00.000',1,1.36,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT INTO Orders  Values ('10929','FRANK',6,'1998-03-05 00:00:00.000','1998-04-02 00:00:00.000','1998-03-12 00:00:00.000',1,33.93,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('10930','SUPRD',4,'1998-03-06 00:00:00.000','1998-04-17 00:00:00.000','1998-03-18 00:00:00.000',3,15.55,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('10931','RICSU',4,'1998-03-06 00:00:00.000','1998-03-20 00:00:00.000','1998-03-19 00:00:00.000',2,13.6,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10932','BONAP',8,'1998-03-06 00:00:00.000','1998-04-03 00:00:00.000','1998-03-24 00:00:00.000',1,134.64,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10933','ISLAT',6,'1998-03-06 00:00:00.000','1998-04-03 00:00:00.000','1998-03-16 00:00:00.000',3,54.15,'Island Trading','Garden House Crowther Way','Cowes','Isle of Wight','PO31 7PJ','UK');
INSERT INTO Orders  Values ('10934','LEHMS',3,'1998-03-09 00:00:00.000','1998-04-06 00:00:00.000','1998-03-12 00:00:00.000',3,32.01,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('10935','WELLI',4,'1998-03-09 00:00:00.000','1998-04-06 00:00:00.000','1998-03-18 00:00:00.000',3,47.59,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT INTO Orders  Values ('10936','GREAL',3,'1998-03-09 00:00:00.000','1998-04-06 00:00:00.000','1998-03-18 00:00:00.000',2,33.68,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('10937','CACTU',7,'1998-03-10 00:00:00.000','1998-03-24 00:00:00.000','1998-03-13 00:00:00.000',3,31.51,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10938','QUICK',3,'1998-03-10 00:00:00.000','1998-04-07 00:00:00.000','1998-03-16 00:00:00.000',2,31.89,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10939','MAGAA',2,'1998-03-10 00:00:00.000','1998-04-07 00:00:00.000','1998-03-13 00:00:00.000',2,76.33,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10940','BONAP',8,'1998-03-11 00:00:00.000','1998-04-08 00:00:00.000','1998-03-23 00:00:00.000',3,19.77,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('10941','SAVEA',7,'1998-03-11 00:00:00.000','1998-04-08 00:00:00.000','1998-03-20 00:00:00.000',2,400.81,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10942','REGGC',9,'1998-03-11 00:00:00.000','1998-04-08 00:00:00.000','1998-03-18 00:00:00.000',3,17.95,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('10943','BSBEV',4,'1998-03-11 00:00:00.000','1998-04-08 00:00:00.000','1998-03-19 00:00:00.000',2,2.17,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10944','BOTTM',6,'1998-03-12 00:00:00.000','1998-03-26 00:00:00.000','1998-03-13 00:00:00.000',3,52.92,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10945','MORGK',4,'1998-03-12 00:00:00.000','1998-04-09 00:00:00.000','1998-03-18 00:00:00.000',1,10.22,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'4179','Germany');
INSERT INTO Orders  Values ('10946','VAFFE',1,'1998-03-12 00:00:00.000','1998-04-09 00:00:00.000','1998-03-19 00:00:00.000',2,27.2,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10947','BSBEV',3,'1998-03-13 00:00:00.000','1998-04-10 00:00:00.000','1998-03-16 00:00:00.000',2,3.26,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('10948','GODOS',3,'1998-03-13 00:00:00.000','1998-04-10 00:00:00.000','1998-03-19 00:00:00.000',3,23.39,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('10949','BOTTM',2,'1998-03-13 00:00:00.000','1998-04-10 00:00:00.000','1998-03-17 00:00:00.000',3,74.44,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10950','MAGAA',1,'1998-03-16 00:00:00.000','1998-04-13 00:00:00.000','1998-03-23 00:00:00.000',2,2.5,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT INTO Orders  Values ('10951','RICSU',9,'1998-03-16 00:00:00.000','1998-04-27 00:00:00.000','1998-04-07 00:00:00.000',2,30.85,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('10952','ALFKI',1,'1998-03-16 00:00:00.000','1998-04-27 00:00:00.000','1998-03-24 00:00:00.000',1,40.42,'Alfred-s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('10953','AROUT',9,'1998-03-16 00:00:00.000','1998-03-30 00:00:00.000','1998-03-25 00:00:00.000',2,23.72,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('10954','LINOD',5,'1998-03-17 00:00:00.000','1998-04-28 00:00:00.000','1998-03-20 00:00:00.000',1,27.91,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('10955','FOLKO',8,'1998-03-17 00:00:00.000','1998-04-14 00:00:00.000','1998-03-20 00:00:00.000',2,3.26,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10956','BLAUS',6,'1998-03-17 00:00:00.000','1998-04-28 00:00:00.000','1998-03-20 00:00:00.000',2,44.65,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('10957','HILAA',8,'1998-03-18 00:00:00.000','1998-04-15 00:00:00.000','1998-03-27 00:00:00.000',3,105.36,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10958','OCEAN',7,'1998-03-18 00:00:00.000','1998-04-15 00:00:00.000','1998-03-27 00:00:00.000',2,49.56,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585 Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10959','GOURL',6,'1998-03-18 00:00:00.000','1998-04-29 00:00:00.000','1998-03-23 00:00:00.000',2,4.98,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('10960','HILAA',3,'1998-03-19 00:00:00.000','1998-04-02 00:00:00.000','1998-04-08 00:00:00.000',1,2.08,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10961','QUEEN',8,'1998-03-19 00:00:00.000','1998-04-16 00:00:00.000','1998-03-30 00:00:00.000',1,104.47,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('10962','QUICK',8,'1998-03-19 00:00:00.000','1998-04-16 00:00:00.000','1998-03-23 00:00:00.000',2,275.79,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10963','FURIB',9,'1998-03-19 00:00:00.000','1998-04-16 00:00:00.000','1998-03-26 00:00:00.000',3,2.7,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT INTO Orders  Values ('10964','SPECD',3,'1998-03-20 00:00:00.000','1998-04-17 00:00:00.000','1998-03-24 00:00:00.000',2,87.38,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT INTO Orders  Values ('10965','OLDWO',6,'1998-03-20 00:00:00.000','1998-04-17 00:00:00.000','1998-03-30 00:00:00.000',3,144.38,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('10966','CHOPS',4,'1998-03-20 00:00:00.000','1998-04-17 00:00:00.000','1998-04-08 00:00:00.000',1,27.19,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('10967','TOMSP',2,'1998-03-23 00:00:00.000','1998-04-20 00:00:00.000','1998-04-02 00:00:00.000',2,62.22,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT INTO Orders  Values ('10968','ERNSH',1,'1998-03-23 00:00:00.000','1998-04-20 00:00:00.000','1998-04-01 00:00:00.000',3,74.6,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10969','COMMI',1,'1998-03-23 00:00:00.000','1998-04-20 00:00:00.000','1998-03-30 00:00:00.000',2,0.21,'Comrcio Mineiro','Av. dos Lusadas, 23','Sao Paulo','SP','05432-043','Brazil');
INSERT INTO Orders  Values ('10970','BOLID',9,'1998-03-24 00:00:00.000','1998-04-07 00:00:00.000','1998-04-24 00:00:00.000',1,16.16,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT INTO Orders  Values ('10971','FRANR',2,'1998-03-24 00:00:00.000','1998-04-21 00:00:00.000','1998-04-02 00:00:00.000',2,121.82,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT INTO Orders  Values ('10972','LACOR',4,'1998-03-24 00:00:00.000','1998-04-21 00:00:00.000','1998-03-26 00:00:00.000',2,0.02,'La corne d-abondance','67, avenue de l-Europe','Versailles',NULL,'78000','France');
INSERT INTO Orders  Values ('10973','LACOR',6,'1998-03-24 00:00:00.000','1998-04-21 00:00:00.000','1998-03-27 00:00:00.000',2,15.17,'La corne d-abondance','67, avenue de l-Europe','Versailles',NULL,'78000','France');
INSERT INTO Orders  Values ('10974','SPLIR',3,'1998-03-25 00:00:00.000','1998-04-08 00:00:00.000','1998-04-03 00:00:00.000',3,12.96,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','USA');
INSERT INTO Orders  Values ('10975','BOTTM',1,'1998-03-25 00:00:00.000','1998-04-22 00:00:00.000','1998-03-27 00:00:00.000',3,32.27,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10976','HILAA',1,'1998-03-25 00:00:00.000','1998-05-06 00:00:00.000','1998-04-03 00:00:00.000',1,37.97,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('10977','FOLKO',8,'1998-03-26 00:00:00.000','1998-04-23 00:00:00.000','1998-04-10 00:00:00.000',3,208.5,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10978','MAISD',9,'1998-03-26 00:00:00.000','1998-04-23 00:00:00.000','1998-04-23 00:00:00.000',2,32.82,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('10979','ERNSH',8,'1998-03-26 00:00:00.000','1998-04-23 00:00:00.000','1998-03-31 00:00:00.000',2,353.07,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10980','FOLKO',4,'1998-03-27 00:00:00.000','1998-05-08 00:00:00.000','1998-04-17 00:00:00.000',1,1.26,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10981','HANAR',1,'1998-03-27 00:00:00.000','1998-04-24 00:00:00.000','1998-04-02 00:00:00.000',2,193.37,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('10982','BOTTM',2,'1998-03-27 00:00:00.000','1998-04-24 00:00:00.000','1998-04-08 00:00:00.000',1,14.01,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('10983','SAVEA',2,'1998-03-27 00:00:00.000','1998-04-24 00:00:00.000','1998-04-06 00:00:00.000',2,657.54,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10984','SAVEA',1,'1998-03-30 00:00:00.000','1998-04-27 00:00:00.000','1998-04-03 00:00:00.000',3,211.22,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('10985','HUNGO',2,'1998-03-30 00:00:00.000','1998-04-27 00:00:00.000','1998-04-02 00:00:00.000',1,91.51,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('10986','OCEAN',8,'1998-03-30 00:00:00.000','1998-04-27 00:00:00.000','1998-04-21 00:00:00.000',2,217.86,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585 Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('10987','EASTC',8,'1998-03-31 00:00:00.000','1998-04-28 00:00:00.000','1998-04-06 00:00:00.000',1,185.48,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('10988','RATTC',3,'1998-03-31 00:00:00.000','1998-04-28 00:00:00.000','1998-04-10 00:00:00.000',2,61.14,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('10989','QUEDE',2,'1998-03-31 00:00:00.000','1998-04-28 00:00:00.000','1998-04-02 00:00:00.000',1,34.76,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT INTO Orders  Values ('10990','ERNSH',2,'1998-04-01 00:00:00.000','1998-05-13 00:00:00.000','1998-04-07 00:00:00.000',3,117.61,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('10991','QUICK',1,'1998-04-01 00:00:00.000','1998-04-29 00:00:00.000','1998-04-07 00:00:00.000',1,38.51,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10992','THEBI',1,'1998-04-01 00:00:00.000','1998-04-29 00:00:00.000','1998-04-03 00:00:00.000',3,4.27,'The Big Cheese','89 Jefferson Way Suite 2','Portland','OR','97201','USA');
INSERT INTO Orders  Values ('10993','FOLKO',7,'1998-04-01 00:00:00.000','1998-04-29 00:00:00.000','1998-04-10 00:00:00.000',3,8.81,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('10994','VAFFE',2,'1998-04-02 00:00:00.000','1998-04-16 00:00:00.000','1998-04-09 00:00:00.000',3,65.53,'Vaffeljernet','Smagsloget 45','rhus',NULL,'8200','Denmark');
INSERT INTO Orders  Values ('10995','PERIC',1,'1998-04-02 00:00:00.000','1998-04-30 00:00:00.000','1998-04-06 00:00:00.000',3,46,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('10996','QUICK',4,'1998-04-02 00:00:00.000','1998-04-30 00:00:00.000','1998-04-10 00:00:00.000',2,1.12,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('10997','LILAS',8,'1998-04-03 00:00:00.000','1998-05-15 00:00:00.000','1998-04-13 00:00:00.000',2,73.91,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('10998','WOLZA',8,'1998-04-03 00:00:00.000','1998-04-17 00:00:00.000','1998-04-17 00:00:00.000',2,20.31,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('10999','OTTIK',6,'1998-04-03 00:00:00.000','1998-05-01 00:00:00.000','1998-04-10 00:00:00.000',2,96.35,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('11000','RATTC',2,'1998-04-06 00:00:00.000','1998-05-04 00:00:00.000','1998-04-14 00:00:00.000',3,55.12,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');
INSERT INTO Orders  Values ('11001','FOLKO',2,'1998-04-06 00:00:00.000','1998-05-04 00:00:00.000','1998-04-14 00:00:00.000',2,197.3,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('11002','SAVEA',4,'1998-04-06 00:00:00.000','1998-05-04 00:00:00.000','1998-04-16 00:00:00.000',1,141.16,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('11003','THECR',3,'1998-04-06 00:00:00.000','1998-05-04 00:00:00.000','1998-04-08 00:00:00.000',3,14.91,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','USA');
INSERT INTO Orders  Values ('11004','MAISD',3,'1998-04-07 00:00:00.000','1998-05-05 00:00:00.000','1998-04-20 00:00:00.000',1,44.84,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT INTO Orders  Values ('11005','WILMK',2,'1998-04-07 00:00:00.000','1998-05-05 00:00:00.000','1998-04-10 00:00:00.000',1,0.75,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT INTO Orders  Values ('11006','GREAL',3,'1998-04-07 00:00:00.000','1998-05-05 00:00:00.000','1998-04-15 00:00:00.000',2,25.19,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('11007','PRINI',8,'1998-04-08 00:00:00.000','1998-05-06 00:00:00.000','1998-04-13 00:00:00.000',2,202.24,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT INTO Orders  Values ('11008','ERNSH',7,'1998-04-08 00:00:00.000','1998-05-06 00:00:00.000',NULL,3,79.46,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('11009','GODOS',2,'1998-04-08 00:00:00.000','1998-05-06 00:00:00.000','1998-04-10 00:00:00.000',1,59.11,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('11010','REGGC',2,'1998-04-09 00:00:00.000','1998-05-07 00:00:00.000','1998-04-21 00:00:00.000',2,28.71,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('11011','ALFKI',3,'1998-04-09 00:00:00.000','1998-05-07 00:00:00.000','1998-04-13 00:00:00.000',1,1.21,'Alfred-s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT INTO Orders  Values ('11012','FRANK',1,'1998-04-09 00:00:00.000','1998-04-23 00:00:00.000','1998-04-17 00:00:00.000',3,242.95,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT INTO Orders  Values ('11013','ROMEY',2,'1998-04-09 00:00:00.000','1998-05-07 00:00:00.000','1998-04-10 00:00:00.000',1,32.99,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT INTO Orders  Values ('11014','LINOD',2,'1998-04-10 00:00:00.000','1998-05-08 00:00:00.000','1998-04-15 00:00:00.000',3,23.6,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('11015','SANTG',2,'1998-04-10 00:00:00.000','1998-04-24 00:00:00.000','1998-04-20 00:00:00.000',2,4.62,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT INTO Orders  Values ('11016','AROUT',9,'1998-04-10 00:00:00.000','1998-05-08 00:00:00.000','1998-04-13 00:00:00.000',2,33.8,'Around the Horn','Brook Farm Stratford St. Mary','Colchester','Essex','CO7 6JX','UK');
INSERT INTO Orders  Values ('11017','ERNSH',9,'1998-04-13 00:00:00.000','1998-05-11 00:00:00.000','1998-04-20 00:00:00.000',2,754.26,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('11018','LONEP',4,'1998-04-13 00:00:00.000','1998-05-11 00:00:00.000','1998-04-16 00:00:00.000',2,11.65,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','USA');
INSERT INTO Orders  Values ('11019','RANCH',6,'1998-04-13 00:00:00.000','1998-05-11 00:00:00.000',NULL,3,3.17,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('11020','OTTIK',2,'1998-04-14 00:00:00.000','1998-05-12 00:00:00.000','1998-04-16 00:00:00.000',2,43.3,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT INTO Orders  Values ('11021','QUICK',3,'1998-04-14 00:00:00.000','1998-05-12 00:00:00.000','1998-04-21 00:00:00.000',1,297.18,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'1307','Germany');
INSERT INTO Orders  Values ('11022','HANAR',9,'1998-04-14 00:00:00.000','1998-05-12 00:00:00.000','1998-05-04 00:00:00.000',2,6.27,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('11023','BSBEV',1,'1998-04-14 00:00:00.000','1998-04-28 00:00:00.000','1998-04-24 00:00:00.000',2,123.83,'B-s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','UK');
INSERT INTO Orders  Values ('11024','EASTC',4,'1998-04-15 00:00:00.000','1998-05-13 00:00:00.000','1998-04-20 00:00:00.000',1,74.36,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('11025','WARTH',6,'1998-04-15 00:00:00.000','1998-05-13 00:00:00.000','1998-04-24 00:00:00.000',3,29.17,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT INTO Orders  Values ('11026','FRANS',4,'1998-04-15 00:00:00.000','1998-05-13 00:00:00.000','1998-04-28 00:00:00.000',1,47.09,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('11027','BOTTM',1,'1998-04-16 00:00:00.000','1998-05-14 00:00:00.000','1998-04-20 00:00:00.000',1,52.52,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('11028','KOENE',2,'1998-04-16 00:00:00.000','1998-05-14 00:00:00.000','1998-04-22 00:00:00.000',1,29.59,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT INTO Orders  Values ('11029','CHOPS',4,'1998-04-16 00:00:00.000','1998-05-14 00:00:00.000','1998-04-27 00:00:00.000',1,47.84,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('11030','SAVEA',7,'1998-04-17 00:00:00.000','1998-05-15 00:00:00.000','1998-04-27 00:00:00.000',2,830.75,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('11031','SAVEA',6,'1998-04-17 00:00:00.000','1998-05-15 00:00:00.000','1998-04-24 00:00:00.000',2,227.22,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('11032','WHITC',2,'1998-04-17 00:00:00.000','1998-05-15 00:00:00.000','1998-04-23 00:00:00.000',3,606.19,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('11033','RICSU',7,'1998-04-17 00:00:00.000','1998-05-15 00:00:00.000','1998-04-23 00:00:00.000',3,84.74,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('11034','OLDWO',8,'1998-04-20 00:00:00.000','1998-06-01 00:00:00.000','1998-04-27 00:00:00.000',1,40.32,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','USA');
INSERT INTO Orders  Values ('11035','SUPRD',2,'1998-04-20 00:00:00.000','1998-05-18 00:00:00.000','1998-04-24 00:00:00.000',2,0.17,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('11036','DRACD',8,'1998-04-20 00:00:00.000','1998-05-18 00:00:00.000','1998-04-22 00:00:00.000',3,149.47,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('11037','GODOS',7,'1998-04-21 00:00:00.000','1998-05-19 00:00:00.000','1998-04-27 00:00:00.000',1,3.2,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT INTO Orders  Values ('11038','SUPRD',1,'1998-04-21 00:00:00.000','1998-05-19 00:00:00.000','1998-04-30 00:00:00.000',2,29.59,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT INTO Orders  Values ('11039','LINOD',1,'1998-04-21 00:00:00.000','1998-05-19 00:00:00.000',NULL,2,65,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT INTO Orders  Values ('11040','GREAL',4,'1998-04-22 00:00:00.000','1998-05-20 00:00:00.000',NULL,3,18.84,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('11041','CHOPS',3,'1998-04-22 00:00:00.000','1998-05-20 00:00:00.000','1998-04-28 00:00:00.000',2,48.22,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT INTO Orders  Values ('11042','COMMI',2,'1998-04-22 00:00:00.000','1998-05-06 00:00:00.000','1998-05-01 00:00:00.000',1,29.99,'Comrcio Mineiro','Av. dos Lusadas, 23','Sao Paulo','SP','05432-043','Brazil');
INSERT INTO Orders  Values ('11043','SPECD',5,'1998-04-22 00:00:00.000','1998-05-20 00:00:00.000','1998-04-29 00:00:00.000',2,8.8,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT INTO Orders  Values ('11044','WOLZA',4,'1998-04-23 00:00:00.000','1998-05-21 00:00:00.000','1998-05-01 00:00:00.000',1,8.72,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT INTO Orders  Values ('11045','BOTTM',6,'1998-04-23 00:00:00.000','1998-05-21 00:00:00.000',NULL,2,70.58,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('11046','WANDK',8,'1998-04-23 00:00:00.000','1998-05-21 00:00:00.000','1998-04-24 00:00:00.000',2,71.64,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT INTO Orders  Values ('11047','EASTC',7,'1998-04-24 00:00:00.000','1998-05-22 00:00:00.000','1998-05-01 00:00:00.000',3,46.62,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('11048','BOTTM',7,'1998-04-24 00:00:00.000','1998-05-22 00:00:00.000','1998-04-30 00:00:00.000',3,24.12,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT INTO Orders  Values ('11049','GOURL',3,'1998-04-24 00:00:00.000','1998-05-22 00:00:00.000','1998-05-04 00:00:00.000',1,8.34,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT INTO Orders  Values ('11050','FOLKO',8,'1998-04-27 00:00:00.000','1998-05-25 00:00:00.000','1998-05-05 00:00:00.000',2,59.41,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT INTO Orders  Values ('11051','LAMAI',7,'1998-04-27 00:00:00.000','1998-05-25 00:00:00.000',NULL,3,2.79,'La maison d-Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT INTO Orders  Values ('11052','HANAR',3,'1998-04-27 00:00:00.000','1998-05-25 00:00:00.000','1998-05-01 00:00:00.000',1,67.26,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT INTO Orders  Values ('11053','PICCO',2,'1998-04-27 00:00:00.000','1998-05-25 00:00:00.000','1998-04-29 00:00:00.000',2,53.05,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT INTO Orders  Values ('11054','CACTU',8,'1998-04-28 00:00:00.000','1998-05-26 00:00:00.000',NULL,1,0.33,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT INTO Orders  Values ('11055','HILAA',7,'1998-04-28 00:00:00.000','1998-05-26 00:00:00.000','1998-05-05 00:00:00.000',2,120.92,'HILARION-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT INTO Orders  Values ('11056','EASTC',8,'1998-04-28 00:00:00.000','1998-05-12 00:00:00.000','1998-05-01 00:00:00.000',2,278.96,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','UK');
INSERT INTO Orders  Values ('11057','NORTS',3,'1998-04-29 00:00:00.000','1998-05-27 00:00:00.000','1998-05-01 00:00:00.000',3,4.13,'North/South','South House 300 Queensbridge','London',NULL,'SW7 1RZ','UK');
INSERT INTO Orders  Values ('11058','BLAUS',9,'1998-04-29 00:00:00.000','1998-05-27 00:00:00.000',NULL,3,31.14,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT INTO Orders  Values ('11059','RICAR',2,'1998-04-29 00:00:00.000','1998-06-10 00:00:00.000',NULL,2,85.8,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT INTO Orders  Values ('11060','FRANS',2,'1998-04-30 00:00:00.000','1998-05-28 00:00:00.000','1998-05-04 00:00:00.000',2,10.98,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT INTO Orders  Values ('11061','GREAL',4,'1998-04-30 00:00:00.000','1998-06-11 00:00:00.000',NULL,3,14.01,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','USA');
INSERT INTO Orders  Values ('11062','REGGC',4,'1998-04-30 00:00:00.000','1998-05-28 00:00:00.000',NULL,2,29.93,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT INTO Orders  Values ('11063','HUNGO',3,'1998-04-30 00:00:00.000','1998-05-28 00:00:00.000','1998-05-06 00:00:00.000',2,81.73,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT INTO Orders  Values ('11064','SAVEA',1,'1998-05-01 00:00:00.000','1998-05-29 00:00:00.000','1998-05-04 00:00:00.000',1,30.09,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','USA');
INSERT INTO Orders  Values ('11065','LILAS',8,'1998-05-01 00:00:00.000','1998-05-29 00:00:00.000',NULL,1,12.91,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('11066','WHITC',7,'1998-05-01 00:00:00.000','1998-05-29 00:00:00.000','1998-05-04 00:00:00.000',2,44.72,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','USA');
INSERT INTO Orders  Values ('11067','DRACD',1,'1998-05-04 00:00:00.000','1998-05-18 00:00:00.000','1998-05-06 00:00:00.000',2,7.98,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT INTO Orders  Values ('11068','QUEEN',8,'1998-05-04 00:00:00.000','1998-06-01 00:00:00.000',NULL,2,81.75,'Queen Cozinha','Alameda dos Canrios, 891','Sao Paulo','SP','05487-020','Brazil');
INSERT INTO Orders  Values ('11069','TORTU',1,'1998-05-04 00:00:00.000','1998-06-01 00:00:00.000','1998-05-06 00:00:00.000',2,15.67,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('11070','LEHMS',2,'1998-05-05 00:00:00.000','1998-06-02 00:00:00.000',NULL,1,136,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M.',NULL,'60528','Germany');
INSERT INTO Orders  Values ('11071','LILAS',1,'1998-05-05 00:00:00.000','1998-06-02 00:00:00.000',NULL,1,0.93,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT INTO Orders  Values ('11072','ERNSH',4,'1998-05-05 00:00:00.000','1998-06-02 00:00:00.000',NULL,2,258.64,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT INTO Orders  Values ('11073','PERIC',2,'1998-05-05 00:00:00.000','1998-06-02 00:00:00.000',NULL,2,24.95,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'5033','Mexico');
INSERT INTO Orders  Values ('11074','SIMOB',7,'1998-05-06 00:00:00.000','1998-06-03 00:00:00.000',NULL,2,18.44,'Simons bistro','Vinbltet 34','Kobenhavn',NULL,'1734','Denmark');
INSERT INTO Orders  Values ('11075','RICSU',8,'1998-05-06 00:00:00.000','1998-06-03 00:00:00.000',NULL,2,6.19,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT INTO Orders  Values ('11076','BONAP',4,'1998-05-06 00:00:00.000','1998-06-03 00:00:00.000',NULL,2,38.28,'Bon app-','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT INTO Orders  Values ('11077','RATTC',1,'1998-05-06 00:00:00.000','1998-06-03 00:00:00.000',NULL,2,8.53,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','USA');

# ---------------------------------------------------------------------- #
# Add info into "Order Details"                                          #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE `Order Details`;
INSERT INTO `Order Details` Values (10248, 11, 14, 12, 0.0);
INSERT INTO `Order Details` Values (10248, 42, 9.8, 10, 0.0);
INSERT INTO `Order Details` Values (10248, 72, 34.8, 5, 0.0);
INSERT INTO `Order Details` Values (10249, 14, 18.6, 9, 0.0);
INSERT INTO `Order Details` Values (10249, 51, 42.4, 40, 0.0);
INSERT INTO `Order Details` Values (10250, 41, 7.7, 10, 0.0);
INSERT INTO `Order Details` Values (10250, 51, 42.4, 35, 0.15);
INSERT INTO `Order Details` Values (10250, 65, 16.8, 15, 0.15);
INSERT INTO `Order Details` Values (10251, 22, 16.8, 6, 0.05);
INSERT INTO `Order Details` Values (10251, 57, 15.6, 15, 0.05);
INSERT INTO `Order Details` Values (10251, 65, 16.8, 20, 0.0);
INSERT INTO `Order Details` Values (10252, 20, 64.8, 40, 0.05);
INSERT INTO `Order Details` Values (10252, 33, 2, 25, 0.05);
INSERT INTO `Order Details` Values (10252, 60, 27.2, 40, 0.0);
INSERT INTO `Order Details` Values (10253, 31, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10253, 39, 14.4, 42, 0.0);
INSERT INTO `Order Details` Values (10253, 49, 16, 40, 0.0);
INSERT INTO `Order Details` Values (10254, 24, 3.6, 15, 0.15);
INSERT INTO `Order Details` Values (10254, 55, 19.2, 21, 0.15);
INSERT INTO `Order Details` Values (10254, 74, 8, 21, 0.0);
INSERT INTO `Order Details` Values (10255, 2, 15.2, 20, 0.0);
INSERT INTO `Order Details` Values (10255, 16, 13.9, 35, 0.0);
INSERT INTO `Order Details` Values (10255, 36, 15.2, 25, 0.0);
INSERT INTO `Order Details` Values (10255, 59, 44, 30, 0.0);
INSERT INTO `Order Details` Values (10256, 53, 26.2, 15, 0.0);
INSERT INTO `Order Details` Values (10256, 77, 10.4, 12, 0.0);
INSERT INTO `Order Details` Values (10257, 27, 35.1, 25, 0.0);
INSERT INTO `Order Details` Values (10257, 39, 14.4, 6, 0.0);
INSERT INTO `Order Details` Values (10257, 77, 10.4, 15, 0.0);
INSERT INTO `Order Details` Values (10258, 2, 15.2, 50, 0.2);
INSERT INTO `Order Details` Values (10258, 5, 17, 65, 0.2);
INSERT INTO `Order Details` Values (10258, 32, 25.6, 6, 0.2);
INSERT INTO `Order Details` Values (10259, 21, 8, 10, 0.0);
INSERT INTO `Order Details` Values (10259, 37, 20.8, 1, 0.0);
INSERT INTO `Order Details` Values (10260, 41, 7.7, 16, 0.25);
INSERT INTO `Order Details` Values (10260, 57, 15.6, 50, 0.0);
INSERT INTO `Order Details` Values (10260, 62, 39.4, 15, 0.25);
INSERT INTO `Order Details` Values (10260, 70, 12, 21, 0.25);
INSERT INTO `Order Details` Values (10261, 21, 8, 20, 0.0);
INSERT INTO `Order Details` Values (10261, 35, 14.4, 20, 0.0);
INSERT INTO `Order Details` Values (10262, 5, 17, 12, 0.2);
INSERT INTO `Order Details` Values (10262, 7, 24, 15, 0.0);
INSERT INTO `Order Details` Values (10262, 56, 30.4, 2, 0.0);
INSERT INTO `Order Details` Values (10263, 16, 13.9, 60, 0.25);
INSERT INTO `Order Details` Values (10263, 24, 3.6, 28, 0.0);
INSERT INTO `Order Details` Values (10263, 30, 20.7, 60, 0.25);
INSERT INTO `Order Details` Values (10263, 74, 8, 36, 0.25);
INSERT INTO `Order Details` Values (10264, 2, 15.2, 35, 0.0);
INSERT INTO `Order Details` Values (10264, 41, 7.7, 25, 0.15);
INSERT INTO `Order Details` Values (10265, 17, 31.2, 30, 0.0);
INSERT INTO `Order Details` Values (10265, 70, 12, 20, 0.0);
INSERT INTO `Order Details` Values (10266, 12, 30.4, 12, 0.05);
INSERT INTO `Order Details` Values (10267, 40, 14.7, 50, 0.0);
INSERT INTO `Order Details` Values (10267, 59, 44, 70, 0.15);
INSERT INTO `Order Details` Values (10267, 76, 14.4, 15, 0.15);
INSERT INTO `Order Details` Values (10268, 29, 99, 10, 0.0);
INSERT INTO `Order Details` Values (10268, 72, 27.8, 4, 0.0);
INSERT INTO `Order Details` Values (10269, 33, 2, 60, 0.05);
INSERT INTO `Order Details` Values (10269, 72, 27.8, 20, 0.05);
INSERT INTO `Order Details` Values (10270, 36, 15.2, 30, 0.0);
INSERT INTO `Order Details` Values (10270, 43, 36.8, 25, 0.0);
INSERT INTO `Order Details` Values (10271, 33, 2, 24, 0.0);
INSERT INTO `Order Details` Values (10272, 20, 64.8, 6, 0.0);
INSERT INTO `Order Details` Values (10272, 31, 10, 40, 0.0);
INSERT INTO `Order Details` Values (10272, 72, 27.8, 24, 0.0);
INSERT INTO `Order Details` Values (10273, 10, 24.8, 24, 0.05);
INSERT INTO `Order Details` Values (10273, 31, 10, 15, 0.05);
INSERT INTO `Order Details` Values (10273, 33, 2, 20, 0.0);
INSERT INTO `Order Details` Values (10273, 40, 14.7, 60, 0.05);
INSERT INTO `Order Details` Values (10273, 76, 14.4, 33, 0.05);
INSERT INTO `Order Details` Values (10274, 71, 17.2, 20, 0.0);
INSERT INTO `Order Details` Values (10274, 72, 27.8, 7, 0.0);
INSERT INTO `Order Details` Values (10275, 24, 3.6, 12, 0.05);
INSERT INTO `Order Details` Values (10275, 59, 44, 6, 0.05);
INSERT INTO `Order Details` Values (10276, 10, 24.8, 15, 0.0);
INSERT INTO `Order Details` Values (10276, 13, 4.8, 10, 0.0);
INSERT INTO `Order Details` Values (10277, 28, 36.4, 20, 0.0);
INSERT INTO `Order Details` Values (10277, 62, 39.4, 12, 0.0);
INSERT INTO `Order Details` Values (10278, 44, 15.5, 16, 0.0);
INSERT INTO `Order Details` Values (10278, 59, 44, 15, 0.0);
INSERT INTO `Order Details` Values (10278, 63, 35.1, 8, 0.0);
INSERT INTO `Order Details` Values (10278, 73, 12, 25, 0.0);
INSERT INTO `Order Details` Values (10279, 17, 31.2, 15, 0.25);
INSERT INTO `Order Details` Values (10280, 24, 3.6, 12, 0.0);
INSERT INTO `Order Details` Values (10280, 55, 19.2, 20, 0.0);
INSERT INTO `Order Details` Values (10280, 75, 6.2, 30, 0.0);
INSERT INTO `Order Details` Values (10281, 19, 7.3, 1, 0.0);
INSERT INTO `Order Details` Values (10281, 24, 3.6, 6, 0.0);
INSERT INTO `Order Details` Values (10281, 35, 14.4, 4, 0.0);
INSERT INTO `Order Details` Values (10282, 30, 20.7, 6, 0.0);
INSERT INTO `Order Details` Values (10282, 57, 15.6, 2, 0.0);
INSERT INTO `Order Details` Values (10283, 15, 12.4, 20, 0.0);
INSERT INTO `Order Details` Values (10283, 19, 7.3, 18, 0.0);
INSERT INTO `Order Details` Values (10283, 60, 27.2, 35, 0.0);
INSERT INTO `Order Details` Values (10283, 72, 27.8, 3, 0.0);
INSERT INTO `Order Details` Values (10284, 27, 35.1, 15, 0.25);
INSERT INTO `Order Details` Values (10284, 44, 15.5, 21, 0.0);
INSERT INTO `Order Details` Values (10284, 60, 27.2, 20, 0.25);
INSERT INTO `Order Details` Values (10284, 67, 11.2, 5, 0.25);
INSERT INTO `Order Details` Values (10285, 1, 14.4, 45, 0.2);
INSERT INTO `Order Details` Values (10285, 40, 14.7, 40, 0.2);
INSERT INTO `Order Details` Values (10285, 53, 26.2, 36, 0.2);
INSERT INTO `Order Details` Values (10286, 35, 14.4, 100, 0.0);
INSERT INTO `Order Details` Values (10286, 62, 39.4, 40, 0.0);
INSERT INTO `Order Details` Values (10287, 16, 13.9, 40, 0.15);
INSERT INTO `Order Details` Values (10287, 34, 11.2, 20, 0.0);
INSERT INTO `Order Details` Values (10287, 46, 9.6, 15, 0.15);
INSERT INTO `Order Details` Values (10288, 54, 5.9, 10, 0.1);
INSERT INTO `Order Details` Values (10288, 68, 10, 3, 0.1);
INSERT INTO `Order Details` Values (10289, 3, 8, 30, 0.0);
INSERT INTO `Order Details` Values (10289, 64, 26.6, 9, 0.0);
INSERT INTO `Order Details` Values (10290, 5, 17, 20, 0.0);
INSERT INTO `Order Details` Values (10290, 29, 99, 15, 0.0);
INSERT INTO `Order Details` Values (10290, 49, 16, 15, 0.0);
INSERT INTO `Order Details` Values (10290, 77, 10.4, 10, 0.0);
INSERT INTO `Order Details` Values (10291, 13, 4.8, 20, 0.1);
INSERT INTO `Order Details` Values (10291, 44, 15.5, 24, 0.1);
INSERT INTO `Order Details` Values (10291, 51, 42.4, 2, 0.1);
INSERT INTO `Order Details` Values (10292, 20, 64.8, 20, 0.0);
INSERT INTO `Order Details` Values (10293, 18, 50, 12, 0.0);
INSERT INTO `Order Details` Values (10293, 24, 3.6, 10, 0.0);
INSERT INTO `Order Details` Values (10293, 63, 35.1, 5, 0.0);
INSERT INTO `Order Details` Values (10293, 75, 6.2, 6, 0.0);
INSERT INTO `Order Details` Values (10294, 1, 14.4, 18, 0.0);
INSERT INTO `Order Details` Values (10294, 17, 31.2, 15, 0.0);
INSERT INTO `Order Details` Values (10294, 43, 36.8, 15, 0.0);
INSERT INTO `Order Details` Values (10294, 60, 27.2, 21, 0.0);
INSERT INTO `Order Details` Values (10294, 75, 6.2, 6, 0.0);
INSERT INTO `Order Details` Values (10295, 56, 30.4, 4, 0.0);
INSERT INTO `Order Details` Values (10296, 11, 16.8, 12, 0.0);
INSERT INTO `Order Details` Values (10296, 16, 13.9, 30, 0.0);
INSERT INTO `Order Details` Values (10296, 69, 28.8, 15, 0.0);
INSERT INTO `Order Details` Values (10297, 39, 14.4, 60, 0.0);
INSERT INTO `Order Details` Values (10297, 72, 27.8, 20, 0.0);
INSERT INTO `Order Details` Values (10298, 2, 15.2, 40, 0.0);
INSERT INTO `Order Details` Values (10298, 36, 15.2, 40, 0.25);
INSERT INTO `Order Details` Values (10298, 59, 44, 30, 0.25);
INSERT INTO `Order Details` Values (10298, 62, 39.4, 15, 0.0);
INSERT INTO `Order Details` Values (10299, 19, 7.3, 15, 0.0);
INSERT INTO `Order Details` Values (10299, 70, 12, 20, 0.0);
INSERT INTO `Order Details` Values (10300, 66, 13.6, 30, 0.0);
INSERT INTO `Order Details` Values (10300, 68, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10301, 40, 14.7, 10, 0.0);
INSERT INTO `Order Details` Values (10301, 56, 30.4, 20, 0.0);
INSERT INTO `Order Details` Values (10302, 17, 31.2, 40, 0.0);
INSERT INTO `Order Details` Values (10302, 28, 36.4, 28, 0.0);
INSERT INTO `Order Details` Values (10302, 43, 36.8, 12, 0.0);
INSERT INTO `Order Details` Values (10303, 40, 14.7, 40, 0.1);
INSERT INTO `Order Details` Values (10303, 65, 16.8, 30, 0.1);
INSERT INTO `Order Details` Values (10303, 68, 10, 15, 0.1);
INSERT INTO `Order Details` Values (10304, 49, 16, 30, 0.0);
INSERT INTO `Order Details` Values (10304, 59, 44, 10, 0.0);
INSERT INTO `Order Details` Values (10304, 71, 17.2, 2, 0.0);
INSERT INTO `Order Details` Values (10305, 18, 50, 25, 0.1);
INSERT INTO `Order Details` Values (10305, 29, 99, 25, 0.1);
INSERT INTO `Order Details` Values (10305, 39, 14.4, 30, 0.1);
INSERT INTO `Order Details` Values (10306, 30, 20.7, 10, 0.0);
INSERT INTO `Order Details` Values (10306, 53, 26.2, 10, 0.0);
INSERT INTO `Order Details` Values (10306, 54, 5.9, 5, 0.0);
INSERT INTO `Order Details` Values (10307, 62, 39.4, 10, 0.0);
INSERT INTO `Order Details` Values (10307, 68, 10, 3, 0.0);
INSERT INTO `Order Details` Values (10308, 69, 28.8, 1, 0.0);
INSERT INTO `Order Details` Values (10308, 70, 12, 5, 0.0);
INSERT INTO `Order Details` Values (10309, 4, 17.6, 20, 0.0);
INSERT INTO `Order Details` Values (10309, 6, 20, 30, 0.0);
INSERT INTO `Order Details` Values (10309, 42, 11.2, 2, 0.0);
INSERT INTO `Order Details` Values (10309, 43, 36.8, 20, 0.0);
INSERT INTO `Order Details` Values (10309, 71, 17.2, 3, 0.0);
INSERT INTO `Order Details` Values (10310, 16, 13.9, 10, 0.0);
INSERT INTO `Order Details` Values (10310, 62, 39.4, 5, 0.0);
INSERT INTO `Order Details` Values (10311, 42, 11.2, 6, 0.0);
INSERT INTO `Order Details` Values (10311, 69, 28.8, 7, 0.0);
INSERT INTO `Order Details` Values (10312, 28, 36.4, 4, 0.0);
INSERT INTO `Order Details` Values (10312, 43, 36.8, 24, 0.0);
INSERT INTO `Order Details` Values (10312, 53, 26.2, 20, 0.0);
INSERT INTO `Order Details` Values (10312, 75, 6.2, 10, 0.0);
INSERT INTO `Order Details` Values (10313, 36, 15.2, 12, 0.0);
INSERT INTO `Order Details` Values (10314, 32, 25.6, 40, 0.1);
INSERT INTO `Order Details` Values (10314, 58, 10.6, 30, 0.1);
INSERT INTO `Order Details` Values (10314, 62, 39.4, 25, 0.1);
INSERT INTO `Order Details` Values (10315, 34, 11.2, 14, 0.0);
INSERT INTO `Order Details` Values (10315, 70, 12, 30, 0.0);
INSERT INTO `Order Details` Values (10316, 41, 7.7, 10, 0.0);
INSERT INTO `Order Details` Values (10316, 62, 39.4, 70, 0.0);
INSERT INTO `Order Details` Values (10317, 1, 14.4, 20, 0.0);
INSERT INTO `Order Details` Values (10318, 41, 7.7, 20, 0.0);
INSERT INTO `Order Details` Values (10318, 76, 14.4, 6, 0.0);
INSERT INTO `Order Details` Values (10319, 17, 31.2, 8, 0.0);
INSERT INTO `Order Details` Values (10319, 28, 36.4, 14, 0.0);
INSERT INTO `Order Details` Values (10319, 76, 14.4, 30, 0.0);
INSERT INTO `Order Details` Values (10320, 71, 17.2, 30, 0.0);
INSERT INTO `Order Details` Values (10321, 35, 14.4, 10, 0.0);
INSERT INTO `Order Details` Values (10322, 52, 5.6, 20, 0.0);
INSERT INTO `Order Details` Values (10323, 15, 12.4, 5, 0.0);
INSERT INTO `Order Details` Values (10323, 25, 11.2, 4, 0.0);
INSERT INTO `Order Details` Values (10323, 39, 14.4, 4, 0.0);
INSERT INTO `Order Details` Values (10324, 16, 13.9, 21, 0.15);
INSERT INTO `Order Details` Values (10324, 35, 14.4, 70, 0.15);
INSERT INTO `Order Details` Values (10324, 46, 9.6, 30, 0.0);
INSERT INTO `Order Details` Values (10324, 59, 44, 40, 0.15);
INSERT INTO `Order Details` Values (10324, 63, 35.1, 80, 0.15);
INSERT INTO `Order Details` Values (10325, 6, 20, 6, 0.0);
INSERT INTO `Order Details` Values (10325, 13, 4.8, 12, 0.0);
INSERT INTO `Order Details` Values (10325, 14, 18.6, 9, 0.0);
INSERT INTO `Order Details` Values (10325, 31, 10, 4, 0.0);
INSERT INTO `Order Details` Values (10325, 72, 27.8, 40, 0.0);
INSERT INTO `Order Details` Values (10326, 4, 17.6, 24, 0.0);
INSERT INTO `Order Details` Values (10326, 57, 15.6, 16, 0.0);
INSERT INTO `Order Details` Values (10326, 75, 6.2, 50, 0.0);
INSERT INTO `Order Details` Values (10327, 2, 15.2, 25, 0.2);
INSERT INTO `Order Details` Values (10327, 11, 16.8, 50, 0.2);
INSERT INTO `Order Details` Values (10327, 30, 20.7, 35, 0.2);
INSERT INTO `Order Details` Values (10327, 58, 10.6, 30, 0.2);
INSERT INTO `Order Details` Values (10328, 59, 44, 9, 0.0);
INSERT INTO `Order Details` Values (10328, 65, 16.8, 40, 0.0);
INSERT INTO `Order Details` Values (10328, 68, 10, 10, 0.0);
INSERT INTO `Order Details` Values (10329, 19, 7.3, 10, 0.05);
INSERT INTO `Order Details` Values (10329, 30, 20.7, 8, 0.05);
INSERT INTO `Order Details` Values (10329, 38, 210.8, 20, 0.05);
INSERT INTO `Order Details` Values (10329, 56, 30.4, 12, 0.05);
INSERT INTO `Order Details` Values (10330, 26, 24.9, 50, 0.15);
INSERT INTO `Order Details` Values (10330, 72, 27.8, 25, 0.15);
INSERT INTO `Order Details` Values (10331, 54, 5.9, 15, 0.0);
INSERT INTO `Order Details` Values (10332, 18, 50, 40, 0.2);
INSERT INTO `Order Details` Values (10332, 42, 11.2, 10, 0.2);
INSERT INTO `Order Details` Values (10332, 47, 7.6, 16, 0.2);
INSERT INTO `Order Details` Values (10333, 14, 18.6, 10, 0.0);
INSERT INTO `Order Details` Values (10333, 21, 8, 10, 0.1);
INSERT INTO `Order Details` Values (10333, 71, 17.2, 40, 0.1);
INSERT INTO `Order Details` Values (10334, 52, 5.6, 8, 0.0);
INSERT INTO `Order Details` Values (10334, 68, 10, 10, 0.0);
INSERT INTO `Order Details` Values (10335, 2, 15.2, 7, 0.2);
INSERT INTO `Order Details` Values (10335, 31, 10, 25, 0.2);
INSERT INTO `Order Details` Values (10335, 32, 25.6, 6, 0.2);
INSERT INTO `Order Details` Values (10335, 51, 42.4, 48, 0.2);
INSERT INTO `Order Details` Values (10336, 4, 17.6, 18, 0.1);
INSERT INTO `Order Details` Values (10337, 23, 7.2, 40, 0.0);
INSERT INTO `Order Details` Values (10337, 26, 24.9, 24, 0.0);
INSERT INTO `Order Details` Values (10337, 36, 15.2, 20, 0.0);
INSERT INTO `Order Details` Values (10337, 37, 20.8, 28, 0.0);
INSERT INTO `Order Details` Values (10337, 72, 27.8, 25, 0.0);
INSERT INTO `Order Details` Values (10338, 17, 31.2, 20, 0.0);
INSERT INTO `Order Details` Values (10338, 30, 20.7, 15, 0.0);
INSERT INTO `Order Details` Values (10339, 4, 17.6, 10, 0.0);
INSERT INTO `Order Details` Values (10339, 17, 31.2, 70, 0.05);
INSERT INTO `Order Details` Values (10339, 62, 39.4, 28, 0.0);
INSERT INTO `Order Details` Values (10340, 18, 50, 20, 0.05);
INSERT INTO `Order Details` Values (10340, 41, 7.7, 12, 0.05);
INSERT INTO `Order Details` Values (10340, 43, 36.8, 40, 0.05);
INSERT INTO `Order Details` Values (10341, 33, 2, 8, 0.0);
INSERT INTO `Order Details` Values (10341, 59, 44, 9, 0.15);
INSERT INTO `Order Details` Values (10342, 2, 15.2, 24, 0.2);
INSERT INTO `Order Details` Values (10342, 31, 10, 56, 0.2);
INSERT INTO `Order Details` Values (10342, 36, 15.2, 40, 0.2);
INSERT INTO `Order Details` Values (10342, 55, 19.2, 40, 0.2);
INSERT INTO `Order Details` Values (10343, 64, 26.6, 50, 0.0);
INSERT INTO `Order Details` Values (10343, 68, 10, 4, 0.05);
INSERT INTO `Order Details` Values (10343, 76, 14.4, 15, 0.0);
INSERT INTO `Order Details` Values (10344, 4, 17.6, 35, 0.0);
INSERT INTO `Order Details` Values (10344, 8, 32, 70, 0.25);
INSERT INTO `Order Details` Values (10345, 8, 32, 70, 0.0);
INSERT INTO `Order Details` Values (10345, 19, 7.3, 80, 0.0);
INSERT INTO `Order Details` Values (10345, 42, 11.2, 9, 0.0);
INSERT INTO `Order Details` Values (10346, 17, 31.2, 36, 0.1);
INSERT INTO `Order Details` Values (10346, 56, 30.4, 20, 0.0);
INSERT INTO `Order Details` Values (10347, 25, 11.2, 10, 0.0);
INSERT INTO `Order Details` Values (10347, 39, 14.4, 50, 0.15);
INSERT INTO `Order Details` Values (10347, 40, 14.7, 4, 0.0);
INSERT INTO `Order Details` Values (10347, 75, 6.2, 6, 0.15);
INSERT INTO `Order Details` Values (10348, 1, 14.4, 15, 0.15);
INSERT INTO `Order Details` Values (10348, 23, 7.2, 25, 0.0);
INSERT INTO `Order Details` Values (10349, 54, 5.9, 24, 0.0);
INSERT INTO `Order Details` Values (10350, 50, 13, 15, 0.1);
INSERT INTO `Order Details` Values (10350, 69, 28.8, 18, 0.1);
INSERT INTO `Order Details` Values (10351, 38, 210.8, 20, 0.05);
INSERT INTO `Order Details` Values (10351, 41, 7.7, 13, 0.0);
INSERT INTO `Order Details` Values (10351, 44, 15.5, 77, 0.05);
INSERT INTO `Order Details` Values (10351, 65, 16.8, 10, 0.05);
INSERT INTO `Order Details` Values (10352, 24, 3.6, 10, 0.0);
INSERT INTO `Order Details` Values (10352, 54, 5.9, 20, 0.15);
INSERT INTO `Order Details` Values (10353, 11, 16.8, 12, 0.2);
INSERT INTO `Order Details` Values (10353, 38, 210.8, 50, 0.2);
INSERT INTO `Order Details` Values (10354, 1, 14.4, 12, 0.0);
INSERT INTO `Order Details` Values (10354, 29, 99, 4, 0.0);
INSERT INTO `Order Details` Values (10355, 24, 3.6, 25, 0.0);
INSERT INTO `Order Details` Values (10355, 57, 15.6, 25, 0.0);
INSERT INTO `Order Details` Values (10356, 31, 10, 30, 0.0);
INSERT INTO `Order Details` Values (10356, 55, 19.2, 12, 0.0);
INSERT INTO `Order Details` Values (10356, 69, 28.8, 20, 0.0);
INSERT INTO `Order Details` Values (10357, 10, 24.8, 30, 0.2);
INSERT INTO `Order Details` Values (10357, 26, 24.9, 16, 0.0);
INSERT INTO `Order Details` Values (10357, 60, 27.2, 8, 0.2);
INSERT INTO `Order Details` Values (10358, 24, 3.6, 10, 0.05);
INSERT INTO `Order Details` Values (10358, 34, 11.2, 10, 0.05);
INSERT INTO `Order Details` Values (10358, 36, 15.2, 20, 0.05);
INSERT INTO `Order Details` Values (10359, 16, 13.9, 56, 0.05);
INSERT INTO `Order Details` Values (10359, 31, 10, 70, 0.05);
INSERT INTO `Order Details` Values (10359, 60, 27.2, 80, 0.05);
INSERT INTO `Order Details` Values (10360, 28, 36.4, 30, 0.0);
INSERT INTO `Order Details` Values (10360, 29, 99, 35, 0.0);
INSERT INTO `Order Details` Values (10360, 38, 210.8, 10, 0.0);
INSERT INTO `Order Details` Values (10360, 49, 16, 35, 0.0);
INSERT INTO `Order Details` Values (10360, 54, 5.9, 28, 0.0);
INSERT INTO `Order Details` Values (10361, 39, 14.4, 54, 0.1);
INSERT INTO `Order Details` Values (10361, 60, 27.2, 55, 0.1);
INSERT INTO `Order Details` Values (10362, 25, 11.2, 50, 0.0);
INSERT INTO `Order Details` Values (10362, 51, 42.4, 20, 0.0);
INSERT INTO `Order Details` Values (10362, 54, 5.9, 24, 0.0);
INSERT INTO `Order Details` Values (10363, 31, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10363, 75, 6.2, 12, 0.0);
INSERT INTO `Order Details` Values (10363, 76, 14.4, 12, 0.0);
INSERT INTO `Order Details` Values (10364, 69, 28.8, 30, 0.0);
INSERT INTO `Order Details` Values (10364, 71, 17.2, 5, 0.0);
INSERT INTO `Order Details` Values (10365, 11, 16.8, 24, 0.0);
INSERT INTO `Order Details` Values (10366, 65, 16.8, 5, 0.0);
INSERT INTO `Order Details` Values (10366, 77, 10.4, 5, 0.0);
INSERT INTO `Order Details` Values (10367, 34, 11.2, 36, 0.0);
INSERT INTO `Order Details` Values (10367, 54, 5.9, 18, 0.0);
INSERT INTO `Order Details` Values (10367, 65, 16.8, 15, 0.0);
INSERT INTO `Order Details` Values (10367, 77, 10.4, 7, 0.0);
INSERT INTO `Order Details` Values (10368, 21, 8, 5, 0.1);
INSERT INTO `Order Details` Values (10368, 28, 36.4, 13, 0.1);
INSERT INTO `Order Details` Values (10368, 57, 15.6, 25, 0.0);
INSERT INTO `Order Details` Values (10368, 64, 26.6, 35, 0.1);
INSERT INTO `Order Details` Values (10369, 29, 99, 20, 0.0);
INSERT INTO `Order Details` Values (10369, 56, 30.4, 18, 0.25);
INSERT INTO `Order Details` Values (10370, 1, 14.4, 15, 0.15);
INSERT INTO `Order Details` Values (10370, 64, 26.6, 30, 0.0);
INSERT INTO `Order Details` Values (10370, 74, 8, 20, 0.15);
INSERT INTO `Order Details` Values (10371, 36, 15.2, 6, 0.2);
INSERT INTO `Order Details` Values (10372, 20, 64.8, 12, 0.25);
INSERT INTO `Order Details` Values (10372, 38, 210.8, 40, 0.25);
INSERT INTO `Order Details` Values (10372, 60, 27.2, 70, 0.25);
INSERT INTO `Order Details` Values (10372, 72, 27.8, 42, 0.25);
INSERT INTO `Order Details` Values (10373, 58, 10.6, 80, 0.2);
INSERT INTO `Order Details` Values (10373, 71, 17.2, 50, 0.2);
INSERT INTO `Order Details` Values (10374, 31, 10, 30, 0.0);
INSERT INTO `Order Details` Values (10374, 58, 10.6, 15, 0.0);
INSERT INTO `Order Details` Values (10375, 14, 18.6, 15, 0.0);
INSERT INTO `Order Details` Values (10375, 54, 5.9, 10, 0.0);
INSERT INTO `Order Details` Values (10376, 31, 10, 42, 0.05);
INSERT INTO `Order Details` Values (10377, 28, 36.4, 20, 0.15);
INSERT INTO `Order Details` Values (10377, 39, 14.4, 20, 0.15);
INSERT INTO `Order Details` Values (10378, 71, 17.2, 6, 0.0);
INSERT INTO `Order Details` Values (10379, 41, 7.7, 8, 0.1);
INSERT INTO `Order Details` Values (10379, 63, 35.1, 16, 0.1);
INSERT INTO `Order Details` Values (10379, 65, 16.8, 20, 0.1);
INSERT INTO `Order Details` Values (10380, 30, 20.7, 18, 0.1);
INSERT INTO `Order Details` Values (10380, 53, 26.2, 20, 0.1);
INSERT INTO `Order Details` Values (10380, 60, 27.2, 6, 0.1);
INSERT INTO `Order Details` Values (10380, 70, 12, 30, 0.0);
INSERT INTO `Order Details` Values (10381, 74, 8, 14, 0.0);
INSERT INTO `Order Details` Values (10382, 5, 17, 32, 0.0);
INSERT INTO `Order Details` Values (10382, 18, 50, 9, 0.0);
INSERT INTO `Order Details` Values (10382, 29, 99, 14, 0.0);
INSERT INTO `Order Details` Values (10382, 33, 2, 60, 0.0);
INSERT INTO `Order Details` Values (10382, 74, 8, 50, 0.0);
INSERT INTO `Order Details` Values (10383, 13, 4.8, 20, 0.0);
INSERT INTO `Order Details` Values (10383, 50, 13, 15, 0.0);
INSERT INTO `Order Details` Values (10383, 56, 30.4, 20, 0.0);
INSERT INTO `Order Details` Values (10384, 20, 64.8, 28, 0.0);
INSERT INTO `Order Details` Values (10384, 60, 27.2, 15, 0.0);
INSERT INTO `Order Details` Values (10385, 7, 24, 10, 0.2);
INSERT INTO `Order Details` Values (10385, 60, 27.2, 20, 0.2);
INSERT INTO `Order Details` Values (10385, 68, 10, 8, 0.2);
INSERT INTO `Order Details` Values (10386, 24, 3.6, 15, 0.0);
INSERT INTO `Order Details` Values (10386, 34, 11.2, 10, 0.0);
INSERT INTO `Order Details` Values (10387, 24, 3.6, 15, 0.0);
INSERT INTO `Order Details` Values (10387, 28, 36.4, 6, 0.0);
INSERT INTO `Order Details` Values (10387, 59, 44, 12, 0.0);
INSERT INTO `Order Details` Values (10387, 71, 17.2, 15, 0.0);
INSERT INTO `Order Details` Values (10388, 45, 7.6, 15, 0.2);
INSERT INTO `Order Details` Values (10388, 52, 5.6, 20, 0.2);
INSERT INTO `Order Details` Values (10388, 53, 26.2, 40, 0.0);
INSERT INTO `Order Details` Values (10389, 10, 24.8, 16, 0.0);
INSERT INTO `Order Details` Values (10389, 55, 19.2, 15, 0.0);
INSERT INTO `Order Details` Values (10389, 62, 39.4, 20, 0.0);
INSERT INTO `Order Details` Values (10389, 70, 12, 30, 0.0);
INSERT INTO `Order Details` Values (10390, 31, 10, 60, 0.1);
INSERT INTO `Order Details` Values (10390, 35, 14.4, 40, 0.1);
INSERT INTO `Order Details` Values (10390, 46, 9.6, 45, 0.0);
INSERT INTO `Order Details` Values (10390, 72, 27.8, 24, 0.1);
INSERT INTO `Order Details` Values (10391, 13, 4.8, 18, 0.0);
INSERT INTO `Order Details` Values (10392, 69, 28.8, 50, 0.0);
INSERT INTO `Order Details` Values (10393, 2, 15.2, 25, 0.25);
INSERT INTO `Order Details` Values (10393, 14, 18.6, 42, 0.25);
INSERT INTO `Order Details` Values (10393, 25, 11.2, 7, 0.25);
INSERT INTO `Order Details` Values (10393, 26, 24.9, 70, 0.25);
INSERT INTO `Order Details` Values (10393, 31, 10, 32, 0.0);
INSERT INTO `Order Details` Values (10394, 13, 4.8, 10, 0.0);
INSERT INTO `Order Details` Values (10394, 62, 39.4, 10, 0.0);
INSERT INTO `Order Details` Values (10395, 46, 9.6, 28, 0.1);
INSERT INTO `Order Details` Values (10395, 53, 26.2, 70, 0.1);
INSERT INTO `Order Details` Values (10395, 69, 28.8, 8, 0.0);
INSERT INTO `Order Details` Values (10396, 23, 7.2, 40, 0.0);
INSERT INTO `Order Details` Values (10396, 71, 17.2, 60, 0.0);
INSERT INTO `Order Details` Values (10396, 72, 27.8, 21, 0.0);
INSERT INTO `Order Details` Values (10397, 21, 8, 10, 0.15);
INSERT INTO `Order Details` Values (10397, 51, 42.4, 18, 0.15);
INSERT INTO `Order Details` Values (10398, 35, 14.4, 30, 0.0);
INSERT INTO `Order Details` Values (10398, 55, 19.2, 120, 0.1);
INSERT INTO `Order Details` Values (10399, 68, 10, 60, 0.0);
INSERT INTO `Order Details` Values (10399, 71, 17.2, 30, 0.0);
INSERT INTO `Order Details` Values (10399, 76, 14.4, 35, 0.0);
INSERT INTO `Order Details` Values (10399, 77, 10.4, 14, 0.0);
INSERT INTO `Order Details` Values (10400, 29, 99, 21, 0.0);
INSERT INTO `Order Details` Values (10400, 35, 14.4, 35, 0.0);
INSERT INTO `Order Details` Values (10400, 49, 16, 30, 0.0);
INSERT INTO `Order Details` Values (10401, 30, 20.7, 18, 0.0);
INSERT INTO `Order Details` Values (10401, 56, 30.4, 70, 0.0);
INSERT INTO `Order Details` Values (10401, 65, 16.8, 20, 0.0);
INSERT INTO `Order Details` Values (10401, 71, 17.2, 60, 0.0);
INSERT INTO `Order Details` Values (10402, 23, 7.2, 60, 0.0);
INSERT INTO `Order Details` Values (10402, 63, 35.1, 65, 0.0);
INSERT INTO `Order Details` Values (10403, 16, 13.9, 21, 0.15);
INSERT INTO `Order Details` Values (10403, 48, 10.2, 70, 0.15);
INSERT INTO `Order Details` Values (10404, 26, 24.9, 30, 0.05);
INSERT INTO `Order Details` Values (10404, 42, 11.2, 40, 0.05);
INSERT INTO `Order Details` Values (10404, 49, 16, 30, 0.05);
INSERT INTO `Order Details` Values (10405, 3, 8, 50, 0.0);
INSERT INTO `Order Details` Values (10406, 1, 14.4, 10, 0.0);
INSERT INTO `Order Details` Values (10406, 21, 8, 30, 0.1);
INSERT INTO `Order Details` Values (10406, 28, 36.4, 42, 0.1);
INSERT INTO `Order Details` Values (10406, 36, 15.2, 5, 0.1);
INSERT INTO `Order Details` Values (10406, 40, 14.7, 2, 0.1);
INSERT INTO `Order Details` Values (10407, 11, 16.8, 30, 0.0);
INSERT INTO `Order Details` Values (10407, 69, 28.8, 15, 0.0);
INSERT INTO `Order Details` Values (10407, 71, 17.2, 15, 0.0);
INSERT INTO `Order Details` Values (10408, 37, 20.8, 10, 0.0);
INSERT INTO `Order Details` Values (10408, 54, 5.9, 6, 0.0);
INSERT INTO `Order Details` Values (10408, 62, 39.4, 35, 0.0);
INSERT INTO `Order Details` Values (10409, 14, 18.6, 12, 0.0);
INSERT INTO `Order Details` Values (10409, 21, 8, 12, 0.0);
INSERT INTO `Order Details` Values (10410, 33, 2, 49, 0.0);
INSERT INTO `Order Details` Values (10410, 59, 44, 16, 0.0);
INSERT INTO `Order Details` Values (10411, 41, 7.7, 25, 0.2);
INSERT INTO `Order Details` Values (10411, 44, 15.5, 40, 0.2);
INSERT INTO `Order Details` Values (10411, 59, 44, 9, 0.2);
INSERT INTO `Order Details` Values (10412, 14, 18.6, 20, 0.1);
INSERT INTO `Order Details` Values (10413, 1, 14.4, 24, 0.0);
INSERT INTO `Order Details` Values (10413, 62, 39.4, 40, 0.0);
INSERT INTO `Order Details` Values (10413, 76, 14.4, 14, 0.0);
INSERT INTO `Order Details` Values (10414, 19, 7.3, 18, 0.05);
INSERT INTO `Order Details` Values (10414, 33, 2, 50, 0.0);
INSERT INTO `Order Details` Values (10415, 17, 31.2, 2, 0.0);
INSERT INTO `Order Details` Values (10415, 33, 2, 20, 0.0);
INSERT INTO `Order Details` Values (10416, 19, 7.3, 20, 0.0);
INSERT INTO `Order Details` Values (10416, 53, 26.2, 10, 0.0);
INSERT INTO `Order Details` Values (10416, 57, 15.6, 20, 0.0);
INSERT INTO `Order Details` Values (10417, 38, 210.8, 50, 0.0);
INSERT INTO `Order Details` Values (10417, 46, 9.6, 2, 0.25);
INSERT INTO `Order Details` Values (10417, 68, 10, 36, 0.25);
INSERT INTO `Order Details` Values (10417, 77, 10.4, 35, 0.0);
INSERT INTO `Order Details` Values (10418, 2, 15.2, 60, 0.0);
INSERT INTO `Order Details` Values (10418, 47, 7.6, 55, 0.0);
INSERT INTO `Order Details` Values (10418, 61, 22.8, 16, 0.0);
INSERT INTO `Order Details` Values (10418, 74, 8, 15, 0.0);
INSERT INTO `Order Details` Values (10419, 60, 27.2, 60, 0.05);
INSERT INTO `Order Details` Values (10419, 69, 28.8, 20, 0.05);
INSERT INTO `Order Details` Values (10420, 9, 77.6, 20, 0.1);
INSERT INTO `Order Details` Values (10420, 13, 4.8, 2, 0.1);
INSERT INTO `Order Details` Values (10420, 70, 12, 8, 0.1);
INSERT INTO `Order Details` Values (10420, 73, 12, 20, 0.1);
INSERT INTO `Order Details` Values (10421, 19, 7.3, 4, 0.15);
INSERT INTO `Order Details` Values (10421, 26, 24.9, 30, 0.0);
INSERT INTO `Order Details` Values (10421, 53, 26.2, 15, 0.15);
INSERT INTO `Order Details` Values (10421, 77, 10.4, 10, 0.15);
INSERT INTO `Order Details` Values (10422, 26, 24.9, 2, 0.0);
INSERT INTO `Order Details` Values (10423, 31, 10, 14, 0.0);
INSERT INTO `Order Details` Values (10423, 59, 44, 20, 0.0);
INSERT INTO `Order Details` Values (10424, 35, 14.4, 60, 0.2);
INSERT INTO `Order Details` Values (10424, 38, 210.8, 49, 0.2);
INSERT INTO `Order Details` Values (10424, 68, 10, 30, 0.2);
INSERT INTO `Order Details` Values (10425, 55, 19.2, 10, 0.25);
INSERT INTO `Order Details` Values (10425, 76, 14.4, 20, 0.25);
INSERT INTO `Order Details` Values (10426, 56, 30.4, 5, 0.0);
INSERT INTO `Order Details` Values (10426, 64, 26.6, 7, 0.0);
INSERT INTO `Order Details` Values (10427, 14, 18.6, 35, 0.0);
INSERT INTO `Order Details` Values (10428, 46, 9.6, 20, 0.0);
INSERT INTO `Order Details` Values (10429, 50, 13, 40, 0.0);
INSERT INTO `Order Details` Values (10429, 63, 35.1, 35, 0.25);
INSERT INTO `Order Details` Values (10430, 17, 31.2, 45, 0.2);
INSERT INTO `Order Details` Values (10430, 21, 8, 50, 0.0);
INSERT INTO `Order Details` Values (10430, 56, 30.4, 30, 0.0);
INSERT INTO `Order Details` Values (10430, 59, 44, 70, 0.2);
INSERT INTO `Order Details` Values (10431, 17, 31.2, 50, 0.25);
INSERT INTO `Order Details` Values (10431, 40, 14.7, 50, 0.25);
INSERT INTO `Order Details` Values (10431, 47, 7.6, 30, 0.25);
INSERT INTO `Order Details` Values (10432, 26, 24.9, 10, 0.0);
INSERT INTO `Order Details` Values (10432, 54, 5.9, 40, 0.0);
INSERT INTO `Order Details` Values (10433, 56, 30.4, 28, 0.0);
INSERT INTO `Order Details` Values (10434, 11, 16.8, 6, 0.0);
INSERT INTO `Order Details` Values (10434, 76, 14.4, 18, 0.15);
INSERT INTO `Order Details` Values (10435, 2, 15.2, 10, 0.0);
INSERT INTO `Order Details` Values (10435, 22, 16.8, 12, 0.0);
INSERT INTO `Order Details` Values (10435, 72, 27.8, 10, 0.0);
INSERT INTO `Order Details` Values (10436, 46, 9.6, 5, 0.0);
INSERT INTO `Order Details` Values (10436, 56, 30.4, 40, 0.1);
INSERT INTO `Order Details` Values (10436, 64, 26.6, 30, 0.1);
INSERT INTO `Order Details` Values (10436, 75, 6.2, 24, 0.1);
INSERT INTO `Order Details` Values (10437, 53, 26.2, 15, 0.0);
INSERT INTO `Order Details` Values (10438, 19, 7.3, 15, 0.2);
INSERT INTO `Order Details` Values (10438, 34, 11.2, 20, 0.2);
INSERT INTO `Order Details` Values (10438, 57, 15.6, 15, 0.2);
INSERT INTO `Order Details` Values (10439, 12, 30.4, 15, 0.0);
INSERT INTO `Order Details` Values (10439, 16, 13.9, 16, 0.0);
INSERT INTO `Order Details` Values (10439, 64, 26.6, 6, 0.0);
INSERT INTO `Order Details` Values (10439, 74, 8, 30, 0.0);
INSERT INTO `Order Details` Values (10440, 2, 15.2, 45, 0.15);
INSERT INTO `Order Details` Values (10440, 16, 13.9, 49, 0.15);
INSERT INTO `Order Details` Values (10440, 29, 99, 24, 0.15);
INSERT INTO `Order Details` Values (10440, 61, 22.8, 90, 0.15);
INSERT INTO `Order Details` Values (10441, 27, 35.1, 50, 0.0);
INSERT INTO `Order Details` Values (10442, 11, 16.8, 30, 0.0);
INSERT INTO `Order Details` Values (10442, 54, 5.9, 80, 0.0);
INSERT INTO `Order Details` Values (10442, 66, 13.6, 60, 0.0);
INSERT INTO `Order Details` Values (10443, 11, 16.8, 6, 0.2);
INSERT INTO `Order Details` Values (10443, 28, 36.4, 12, 0.0);
INSERT INTO `Order Details` Values (10444, 17, 31.2, 10, 0.0);
INSERT INTO `Order Details` Values (10444, 26, 24.9, 15, 0.0);
INSERT INTO `Order Details` Values (10444, 35, 14.4, 8, 0.0);
INSERT INTO `Order Details` Values (10444, 41, 7.7, 30, 0.0);
INSERT INTO `Order Details` Values (10445, 39, 14.4, 6, 0.0);
INSERT INTO `Order Details` Values (10445, 54, 5.9, 15, 0.0);
INSERT INTO `Order Details` Values (10446, 19, 7.3, 12, 0.1);
INSERT INTO `Order Details` Values (10446, 24, 3.6, 20, 0.1);
INSERT INTO `Order Details` Values (10446, 31, 10, 3, 0.1);
INSERT INTO `Order Details` Values (10446, 52, 5.6, 15, 0.1);
INSERT INTO `Order Details` Values (10447, 19, 7.3, 40, 0.0);
INSERT INTO `Order Details` Values (10447, 65, 16.8, 35, 0.0);
INSERT INTO `Order Details` Values (10447, 71, 17.2, 2, 0.0);
INSERT INTO `Order Details` Values (10448, 26, 24.9, 6, 0.0);
INSERT INTO `Order Details` Values (10448, 40, 14.7, 20, 0.0);
INSERT INTO `Order Details` Values (10449, 10, 24.8, 14, 0.0);
INSERT INTO `Order Details` Values (10449, 52, 5.6, 20, 0.0);
INSERT INTO `Order Details` Values (10449, 62, 39.4, 35, 0.0);
INSERT INTO `Order Details` Values (10450, 10, 24.8, 20, 0.2);
INSERT INTO `Order Details` Values (10450, 54, 5.9, 6, 0.2);
INSERT INTO `Order Details` Values (10451, 55, 19.2, 120, 0.1);
INSERT INTO `Order Details` Values (10451, 64, 26.6, 35, 0.1);
INSERT INTO `Order Details` Values (10451, 65, 16.8, 28, 0.1);
INSERT INTO `Order Details` Values (10451, 77, 10.4, 55, 0.1);
INSERT INTO `Order Details` Values (10452, 28, 36.4, 15, 0.0);
INSERT INTO `Order Details` Values (10452, 44, 15.5, 100, 0.05);
INSERT INTO `Order Details` Values (10453, 48, 10.2, 15, 0.1);
INSERT INTO `Order Details` Values (10453, 70, 12, 25, 0.1);
INSERT INTO `Order Details` Values (10454, 16, 13.9, 20, 0.2);
INSERT INTO `Order Details` Values (10454, 33, 2, 20, 0.2);
INSERT INTO `Order Details` Values (10454, 46, 9.6, 10, 0.2);
INSERT INTO `Order Details` Values (10455, 39, 14.4, 20, 0.0);
INSERT INTO `Order Details` Values (10455, 53, 26.2, 50, 0.0);
INSERT INTO `Order Details` Values (10455, 61, 22.8, 25, 0.0);
INSERT INTO `Order Details` Values (10455, 71, 17.2, 30, 0.0);
INSERT INTO `Order Details` Values (10456, 21, 8, 40, 0.15);
INSERT INTO `Order Details` Values (10456, 49, 16, 21, 0.15);
INSERT INTO `Order Details` Values (10457, 59, 44, 36, 0.0);
INSERT INTO `Order Details` Values (10458, 26, 24.9, 30, 0.0);
INSERT INTO `Order Details` Values (10458, 28, 36.4, 30, 0.0);
INSERT INTO `Order Details` Values (10458, 43, 36.8, 20, 0.0);
INSERT INTO `Order Details` Values (10458, 56, 30.4, 15, 0.0);
INSERT INTO `Order Details` Values (10458, 71, 17.2, 50, 0.0);
INSERT INTO `Order Details` Values (10459, 7, 24, 16, 0.05);
INSERT INTO `Order Details` Values (10459, 46, 9.6, 20, 0.05);
INSERT INTO `Order Details` Values (10459, 72, 27.8, 40, 0.0);
INSERT INTO `Order Details` Values (10460, 68, 10, 21, 0.25);
INSERT INTO `Order Details` Values (10460, 75, 6.2, 4, 0.25);
INSERT INTO `Order Details` Values (10461, 21, 8, 40, 0.25);
INSERT INTO `Order Details` Values (10461, 30, 20.7, 28, 0.25);
INSERT INTO `Order Details` Values (10461, 55, 19.2, 60, 0.25);
INSERT INTO `Order Details` Values (10462, 13, 4.8, 1, 0.0);
INSERT INTO `Order Details` Values (10462, 23, 7.2, 21, 0.0);
INSERT INTO `Order Details` Values (10463, 19, 7.3, 21, 0.0);
INSERT INTO `Order Details` Values (10463, 42, 11.2, 50, 0.0);
INSERT INTO `Order Details` Values (10464, 4, 17.6, 16, 0.2);
INSERT INTO `Order Details` Values (10464, 43, 36.8, 3, 0.0);
INSERT INTO `Order Details` Values (10464, 56, 30.4, 30, 0.2);
INSERT INTO `Order Details` Values (10464, 60, 27.2, 20, 0.0);
INSERT INTO `Order Details` Values (10465, 24, 3.6, 25, 0.0);
INSERT INTO `Order Details` Values (10465, 29, 99, 18, 0.1);
INSERT INTO `Order Details` Values (10465, 40, 14.7, 20, 0.0);
INSERT INTO `Order Details` Values (10465, 45, 7.6, 30, 0.1);
INSERT INTO `Order Details` Values (10465, 50, 13, 25, 0.0);
INSERT INTO `Order Details` Values (10466, 11, 16.8, 10, 0.0);
INSERT INTO `Order Details` Values (10466, 46, 9.6, 5, 0.0);
INSERT INTO `Order Details` Values (10467, 24, 3.6, 28, 0.0);
INSERT INTO `Order Details` Values (10467, 25, 11.2, 12, 0.0);
INSERT INTO `Order Details` Values (10468, 30, 20.7, 8, 0.0);
INSERT INTO `Order Details` Values (10468, 43, 36.8, 15, 0.0);
INSERT INTO `Order Details` Values (10469, 2, 15.2, 40, 0.15);
INSERT INTO `Order Details` Values (10469, 16, 13.9, 35, 0.15);
INSERT INTO `Order Details` Values (10469, 44, 15.5, 2, 0.15);
INSERT INTO `Order Details` Values (10470, 18, 50, 30, 0.0);
INSERT INTO `Order Details` Values (10470, 23, 7.2, 15, 0.0);
INSERT INTO `Order Details` Values (10470, 64, 26.6, 8, 0.0);
INSERT INTO `Order Details` Values (10471, 7, 24, 30, 0.0);
INSERT INTO `Order Details` Values (10471, 56, 30.4, 20, 0.0);
INSERT INTO `Order Details` Values (10472, 24, 3.6, 80, 0.05);
INSERT INTO `Order Details` Values (10472, 51, 42.4, 18, 0.0);
INSERT INTO `Order Details` Values (10473, 33, 2, 12, 0.0);
INSERT INTO `Order Details` Values (10473, 71, 17.2, 12, 0.0);
INSERT INTO `Order Details` Values (10474, 14, 18.6, 12, 0.0);
INSERT INTO `Order Details` Values (10474, 28, 36.4, 18, 0.0);
INSERT INTO `Order Details` Values (10474, 40, 14.7, 21, 0.0);
INSERT INTO `Order Details` Values (10474, 75, 6.2, 10, 0.0);
INSERT INTO `Order Details` Values (10475, 31, 10, 35, 0.15);
INSERT INTO `Order Details` Values (10475, 66, 13.6, 60, 0.15);
INSERT INTO `Order Details` Values (10475, 76, 14.4, 42, 0.15);
INSERT INTO `Order Details` Values (10476, 55, 19.2, 2, 0.05);
INSERT INTO `Order Details` Values (10476, 70, 12, 12, 0.0);
INSERT INTO `Order Details` Values (10477, 1, 14.4, 15, 0.0);
INSERT INTO `Order Details` Values (10477, 21, 8, 21, 0.25);
INSERT INTO `Order Details` Values (10477, 39, 14.4, 20, 0.25);
INSERT INTO `Order Details` Values (10478, 10, 24.8, 20, 0.05);
INSERT INTO `Order Details` Values (10479, 38, 210.8, 30, 0.0);
INSERT INTO `Order Details` Values (10479, 53, 26.2, 28, 0.0);
INSERT INTO `Order Details` Values (10479, 59, 44, 60, 0.0);
INSERT INTO `Order Details` Values (10479, 64, 26.6, 30, 0.0);
INSERT INTO `Order Details` Values (10480, 47, 7.6, 30, 0.0);
INSERT INTO `Order Details` Values (10480, 59, 44, 12, 0.0);
INSERT INTO `Order Details` Values (10481, 49, 16, 24, 0.0);
INSERT INTO `Order Details` Values (10481, 60, 27.2, 40, 0.0);
INSERT INTO `Order Details` Values (10482, 40, 14.7, 10, 0.0);
INSERT INTO `Order Details` Values (10483, 34, 11.2, 35, 0.05);
INSERT INTO `Order Details` Values (10483, 77, 10.4, 30, 0.05);
INSERT INTO `Order Details` Values (10484, 21, 8, 14, 0.0);
INSERT INTO `Order Details` Values (10484, 40, 14.7, 10, 0.0);
INSERT INTO `Order Details` Values (10484, 51, 42.4, 3, 0.0);
INSERT INTO `Order Details` Values (10485, 2, 15.2, 20, 0.1);
INSERT INTO `Order Details` Values (10485, 3, 8, 20, 0.1);
INSERT INTO `Order Details` Values (10485, 55, 19.2, 30, 0.1);
INSERT INTO `Order Details` Values (10485, 70, 12, 60, 0.1);
INSERT INTO `Order Details` Values (10486, 11, 16.8, 5, 0.0);
INSERT INTO `Order Details` Values (10486, 51, 42.4, 25, 0.0);
INSERT INTO `Order Details` Values (10486, 74, 8, 16, 0.0);
INSERT INTO `Order Details` Values (10487, 19, 7.3, 5, 0.0);
INSERT INTO `Order Details` Values (10487, 26, 24.9, 30, 0.0);
INSERT INTO `Order Details` Values (10487, 54, 5.9, 24, 0.25);
INSERT INTO `Order Details` Values (10488, 59, 44, 30, 0.0);
INSERT INTO `Order Details` Values (10488, 73, 12, 20, 0.2);
INSERT INTO `Order Details` Values (10489, 11, 16.8, 15, 0.25);
INSERT INTO `Order Details` Values (10489, 16, 13.9, 18, 0.0);
INSERT INTO `Order Details` Values (10490, 59, 44, 60, 0.0);
INSERT INTO `Order Details` Values (10490, 68, 10, 30, 0.0);
INSERT INTO `Order Details` Values (10490, 75, 6.2, 36, 0.0);
INSERT INTO `Order Details` Values (10491, 44, 15.5, 15, 0.15);
INSERT INTO `Order Details` Values (10491, 77, 10.4, 7, 0.15);
INSERT INTO `Order Details` Values (10492, 25, 11.2, 60, 0.05);
INSERT INTO `Order Details` Values (10492, 42, 11.2, 20, 0.05);
INSERT INTO `Order Details` Values (10493, 65, 16.8, 15, 0.1);
INSERT INTO `Order Details` Values (10493, 66, 13.6, 10, 0.1);
INSERT INTO `Order Details` Values (10493, 69, 28.8, 10, 0.1);
INSERT INTO `Order Details` Values (10494, 56, 30.4, 30, 0.0);
INSERT INTO `Order Details` Values (10495, 23, 7.2, 10, 0.0);
INSERT INTO `Order Details` Values (10495, 41, 7.7, 20, 0.0);
INSERT INTO `Order Details` Values (10495, 77, 10.4, 5, 0.0);
INSERT INTO `Order Details` Values (10496, 31, 10, 20, 0.05);
INSERT INTO `Order Details` Values (10497, 56, 30.4, 14, 0.0);
INSERT INTO `Order Details` Values (10497, 72, 27.8, 25, 0.0);
INSERT INTO `Order Details` Values (10497, 77, 10.4, 25, 0.0);
INSERT INTO `Order Details` Values (10498, 24, 4.5, 14, 0.0);
INSERT INTO `Order Details` Values (10498, 40, 18.4, 5, 0.0);
INSERT INTO `Order Details` Values (10498, 42, 14, 30, 0.0);
INSERT INTO `Order Details` Values (10499, 28, 45.6, 20, 0.0);
INSERT INTO `Order Details` Values (10499, 49, 20, 25, 0.0);
INSERT INTO `Order Details` Values (10500, 15, 15.5, 12, 0.05);
INSERT INTO `Order Details` Values (10500, 28, 45.6, 8, 0.05);
INSERT INTO `Order Details` Values (10501, 54, 7.45, 20, 0.0);
INSERT INTO `Order Details` Values (10502, 45, 9.5, 21, 0.0);
INSERT INTO `Order Details` Values (10502, 53, 32.8, 6, 0.0);
INSERT INTO `Order Details` Values (10502, 67, 14, 30, 0.0);
INSERT INTO `Order Details` Values (10503, 14, 23.25, 70, 0.0);
INSERT INTO `Order Details` Values (10503, 65, 21.05, 20, 0.0);
INSERT INTO `Order Details` Values (10504, 2, 19, 12, 0.0);
INSERT INTO `Order Details` Values (10504, 21, 10, 12, 0.0);
INSERT INTO `Order Details` Values (10504, 53, 32.8, 10, 0.0);
INSERT INTO `Order Details` Values (10504, 61, 28.5, 25, 0.0);
INSERT INTO `Order Details` Values (10505, 62, 49.3, 3, 0.0);
INSERT INTO `Order Details` Values (10506, 25, 14, 18, 0.1);
INSERT INTO `Order Details` Values (10506, 70, 15, 14, 0.1);
INSERT INTO `Order Details` Values (10507, 43, 46, 15, 0.15);
INSERT INTO `Order Details` Values (10507, 48, 12.75, 15, 0.15);
INSERT INTO `Order Details` Values (10508, 13, 6, 10, 0.0);
INSERT INTO `Order Details` Values (10508, 39, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10509, 28, 45.6, 3, 0.0);
INSERT INTO `Order Details` Values (10510, 29, 123.79, 36, 0.0);
INSERT INTO `Order Details` Values (10510, 75, 7.75, 36, 0.1);
INSERT INTO `Order Details` Values (10511, 4, 22, 50, 0.15);
INSERT INTO `Order Details` Values (10511, 7, 30, 50, 0.15);
INSERT INTO `Order Details` Values (10511, 8, 40, 10, 0.15);
INSERT INTO `Order Details` Values (10512, 24, 4.5, 10, 0.15);
INSERT INTO `Order Details` Values (10512, 46, 12, 9, 0.15);
INSERT INTO `Order Details` Values (10512, 47, 9.5, 6, 0.15);
INSERT INTO `Order Details` Values (10512, 60, 34, 12, 0.15);
INSERT INTO `Order Details` Values (10513, 21, 10, 40, 0.2);
INSERT INTO `Order Details` Values (10513, 32, 32, 50, 0.2);
INSERT INTO `Order Details` Values (10513, 61, 28.5, 15, 0.2);
INSERT INTO `Order Details` Values (10514, 20, 81, 39, 0.0);
INSERT INTO `Order Details` Values (10514, 28, 45.6, 35, 0.0);
INSERT INTO `Order Details` Values (10514, 56, 38, 70, 0.0);
INSERT INTO `Order Details` Values (10514, 65, 21.05, 39, 0.0);
INSERT INTO `Order Details` Values (10514, 75, 7.75, 50, 0.0);
INSERT INTO `Order Details` Values (10515, 9, 97, 16, 0.15);
INSERT INTO `Order Details` Values (10515, 16, 17.45, 50, 0.0);
INSERT INTO `Order Details` Values (10515, 27, 43.9, 120, 0.0);
INSERT INTO `Order Details` Values (10515, 33, 2.5, 16, 0.15);
INSERT INTO `Order Details` Values (10515, 60, 34, 84, 0.15);
INSERT INTO `Order Details` Values (10516, 18, 62.5, 25, 0.1);
INSERT INTO `Order Details` Values (10516, 41, 9.65, 80, 0.1);
INSERT INTO `Order Details` Values (10516, 42, 14, 20, 0.0);
INSERT INTO `Order Details` Values (10517, 52, 7, 6, 0.0);
INSERT INTO `Order Details` Values (10517, 59, 55, 4, 0.0);
INSERT INTO `Order Details` Values (10517, 70, 15, 6, 0.0);
INSERT INTO `Order Details` Values (10518, 24, 4.5, 5, 0.0);
INSERT INTO `Order Details` Values (10518, 38, 263.5, 15, 0.0);
INSERT INTO `Order Details` Values (10518, 44, 19.45, 9, 0.0);
INSERT INTO `Order Details` Values (10519, 10, 31, 16, 0.05);
INSERT INTO `Order Details` Values (10519, 56, 38, 40, 0.0);
INSERT INTO `Order Details` Values (10519, 60, 34, 10, 0.05);
INSERT INTO `Order Details` Values (10520, 24, 4.5, 8, 0.0);
INSERT INTO `Order Details` Values (10520, 53, 32.8, 5, 0.0);
INSERT INTO `Order Details` Values (10521, 35, 18, 3, 0.0);
INSERT INTO `Order Details` Values (10521, 41, 9.65, 10, 0.0);
INSERT INTO `Order Details` Values (10521, 68, 12.5, 6, 0.0);
INSERT INTO `Order Details` Values (10522, 1, 18, 40, 0.2);
INSERT INTO `Order Details` Values (10522, 8, 40, 24, 0.0);
INSERT INTO `Order Details` Values (10522, 30, 25.89, 20, 0.2);
INSERT INTO `Order Details` Values (10522, 40, 18.4, 25, 0.2);
INSERT INTO `Order Details` Values (10523, 17, 39, 25, 0.1);
INSERT INTO `Order Details` Values (10523, 20, 81, 15, 0.1);
INSERT INTO `Order Details` Values (10523, 37, 26, 18, 0.1);
INSERT INTO `Order Details` Values (10523, 41, 9.65, 6, 0.1);
INSERT INTO `Order Details` Values (10524, 10, 31, 2, 0.0);
INSERT INTO `Order Details` Values (10524, 30, 25.89, 10, 0.0);
INSERT INTO `Order Details` Values (10524, 43, 46, 60, 0.0);
INSERT INTO `Order Details` Values (10524, 54, 7.45, 15, 0.0);
INSERT INTO `Order Details` Values (10525, 36, 19, 30, 0.0);
INSERT INTO `Order Details` Values (10525, 40, 18.4, 15, 0.1);
INSERT INTO `Order Details` Values (10526, 1, 18, 8, 0.15);
INSERT INTO `Order Details` Values (10526, 13, 6, 10, 0.0);
INSERT INTO `Order Details` Values (10526, 56, 38, 30, 0.15);
INSERT INTO `Order Details` Values (10527, 4, 22, 50, 0.1);
INSERT INTO `Order Details` Values (10527, 36, 19, 30, 0.1);
INSERT INTO `Order Details` Values (10528, 11, 21, 3, 0.0);
INSERT INTO `Order Details` Values (10528, 33, 2.5, 8, 0.2);
INSERT INTO `Order Details` Values (10528, 72, 34.8, 9, 0.0);
INSERT INTO `Order Details` Values (10529, 55, 24, 14, 0.0);
INSERT INTO `Order Details` Values (10529, 68, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10529, 69, 36, 10, 0.0);
INSERT INTO `Order Details` Values (10530, 17, 39, 40, 0.0);
INSERT INTO `Order Details` Values (10530, 43, 46, 25, 0.0);
INSERT INTO `Order Details` Values (10530, 61, 28.5, 20, 0.0);
INSERT INTO `Order Details` Values (10530, 76, 18, 50, 0.0);
INSERT INTO `Order Details` Values (10531, 59, 55, 2, 0.0);
INSERT INTO `Order Details` Values (10532, 30, 25.89, 15, 0.0);
INSERT INTO `Order Details` Values (10532, 66, 17, 24, 0.0);
INSERT INTO `Order Details` Values (10533, 4, 22, 50, 0.05);
INSERT INTO `Order Details` Values (10533, 72, 34.8, 24, 0.0);
INSERT INTO `Order Details` Values (10533, 73, 15, 24, 0.05);
INSERT INTO `Order Details` Values (10534, 30, 25.89, 10, 0.0);
INSERT INTO `Order Details` Values (10534, 40, 18.4, 10, 0.2);
INSERT INTO `Order Details` Values (10534, 54, 7.45, 10, 0.2);
INSERT INTO `Order Details` Values (10535, 11, 21, 50, 0.1);
INSERT INTO `Order Details` Values (10535, 40, 18.4, 10, 0.1);
INSERT INTO `Order Details` Values (10535, 57, 19.5, 5, 0.1);
INSERT INTO `Order Details` Values (10535, 59, 55, 15, 0.1);
INSERT INTO `Order Details` Values (10536, 12, 38, 15, 0.25);
INSERT INTO `Order Details` Values (10536, 31, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10536, 33, 2.5, 30, 0.0);
INSERT INTO `Order Details` Values (10536, 60, 34, 35, 0.25);
INSERT INTO `Order Details` Values (10537, 31, 12.5, 30, 0.0);
INSERT INTO `Order Details` Values (10537, 51, 53, 6, 0.0);
INSERT INTO `Order Details` Values (10537, 58, 13.25, 20, 0.0);
INSERT INTO `Order Details` Values (10537, 72, 34.8, 21, 0.0);
INSERT INTO `Order Details` Values (10537, 73, 15, 9, 0.0);
INSERT INTO `Order Details` Values (10538, 70, 15, 7, 0.0);
INSERT INTO `Order Details` Values (10538, 72, 34.8, 1, 0.0);
INSERT INTO `Order Details` Values (10539, 13, 6, 8, 0.0);
INSERT INTO `Order Details` Values (10539, 21, 10, 15, 0.0);
INSERT INTO `Order Details` Values (10539, 33, 2.5, 15, 0.0);
INSERT INTO `Order Details` Values (10539, 49, 20, 6, 0.0);
INSERT INTO `Order Details` Values (10540, 3, 10, 60, 0.0);
INSERT INTO `Order Details` Values (10540, 26, 31.23, 40, 0.0);
INSERT INTO `Order Details` Values (10540, 38, 263.5, 30, 0.0);
INSERT INTO `Order Details` Values (10540, 68, 12.5, 35, 0.0);
INSERT INTO `Order Details` Values (10541, 24, 4.5, 35, 0.1);
INSERT INTO `Order Details` Values (10541, 38, 263.5, 4, 0.1);
INSERT INTO `Order Details` Values (10541, 65, 21.05, 36, 0.1);
INSERT INTO `Order Details` Values (10541, 71, 21.5, 9, 0.1);
INSERT INTO `Order Details` Values (10542, 11, 21, 15, 0.05);
INSERT INTO `Order Details` Values (10542, 54, 7.45, 24, 0.05);
INSERT INTO `Order Details` Values (10543, 12, 38, 30, 0.15);
INSERT INTO `Order Details` Values (10543, 23, 9, 70, 0.15);
INSERT INTO `Order Details` Values (10544, 28, 45.6, 7, 0.0);
INSERT INTO `Order Details` Values (10544, 67, 14, 7, 0.0);
INSERT INTO `Order Details` Values (10545, 11, 21, 10, 0.0);
INSERT INTO `Order Details` Values (10546, 7, 30, 10, 0.0);
INSERT INTO `Order Details` Values (10546, 35, 18, 30, 0.0);
INSERT INTO `Order Details` Values (10546, 62, 49.3, 40, 0.0);
INSERT INTO `Order Details` Values (10547, 32, 32, 24, 0.15);
INSERT INTO `Order Details` Values (10547, 36, 19, 60, 0.0);
INSERT INTO `Order Details` Values (10548, 34, 14, 10, 0.25);
INSERT INTO `Order Details` Values (10548, 41, 9.65, 14, 0.0);
INSERT INTO `Order Details` Values (10549, 31, 12.5, 55, 0.15);
INSERT INTO `Order Details` Values (10549, 45, 9.5, 100, 0.15);
INSERT INTO `Order Details` Values (10549, 51, 53, 48, 0.15);
INSERT INTO `Order Details` Values (10550, 17, 39, 8, 0.1);
INSERT INTO `Order Details` Values (10550, 19, 9.2, 10, 0.0);
INSERT INTO `Order Details` Values (10550, 21, 10, 6, 0.1);
INSERT INTO `Order Details` Values (10550, 61, 28.5, 10, 0.1);
INSERT INTO `Order Details` Values (10551, 16, 17.45, 40, 0.15);
INSERT INTO `Order Details` Values (10551, 35, 18, 20, 0.15);
INSERT INTO `Order Details` Values (10551, 44, 19.45, 40, 0.0);
INSERT INTO `Order Details` Values (10552, 69, 36, 18, 0.0);
INSERT INTO `Order Details` Values (10552, 75, 7.75, 30, 0.0);
INSERT INTO `Order Details` Values (10553, 11, 21, 15, 0.0);
INSERT INTO `Order Details` Values (10553, 16, 17.45, 14, 0.0);
INSERT INTO `Order Details` Values (10553, 22, 21, 24, 0.0);
INSERT INTO `Order Details` Values (10553, 31, 12.5, 30, 0.0);
INSERT INTO `Order Details` Values (10553, 35, 18, 6, 0.0);
INSERT INTO `Order Details` Values (10554, 16, 17.45, 30, 0.05);
INSERT INTO `Order Details` Values (10554, 23, 9, 20, 0.05);
INSERT INTO `Order Details` Values (10554, 62, 49.3, 20, 0.05);
INSERT INTO `Order Details` Values (10554, 77, 13, 10, 0.05);
INSERT INTO `Order Details` Values (10555, 14, 23.25, 30, 0.2);
INSERT INTO `Order Details` Values (10555, 19, 9.2, 35, 0.2);
INSERT INTO `Order Details` Values (10555, 24, 4.5, 18, 0.2);
INSERT INTO `Order Details` Values (10555, 51, 53, 20, 0.2);
INSERT INTO `Order Details` Values (10555, 56, 38, 40, 0.2);
INSERT INTO `Order Details` Values (10556, 72, 34.8, 24, 0.0);
INSERT INTO `Order Details` Values (10557, 64, 33.25, 30, 0.0);
INSERT INTO `Order Details` Values (10557, 75, 7.75, 20, 0.0);
INSERT INTO `Order Details` Values (10558, 47, 9.5, 25, 0.0);
INSERT INTO `Order Details` Values (10558, 51, 53, 20, 0.0);
INSERT INTO `Order Details` Values (10558, 52, 7, 30, 0.0);
INSERT INTO `Order Details` Values (10558, 53, 32.8, 18, 0.0);
INSERT INTO `Order Details` Values (10558, 73, 15, 3, 0.0);
INSERT INTO `Order Details` Values (10559, 41, 9.65, 12, 0.05);
INSERT INTO `Order Details` Values (10559, 55, 24, 18, 0.05);
INSERT INTO `Order Details` Values (10560, 30, 25.89, 20, 0.0);
INSERT INTO `Order Details` Values (10560, 62, 49.3, 15, 0.25);
INSERT INTO `Order Details` Values (10561, 44, 19.45, 10, 0.0);
INSERT INTO `Order Details` Values (10561, 51, 53, 50, 0.0);
INSERT INTO `Order Details` Values (10562, 33, 2.5, 20, 0.1);
INSERT INTO `Order Details` Values (10562, 62, 49.3, 10, 0.1);
INSERT INTO `Order Details` Values (10563, 36, 19, 25, 0.0);
INSERT INTO `Order Details` Values (10563, 52, 7, 70, 0.0);
INSERT INTO `Order Details` Values (10564, 17, 39, 16, 0.05);
INSERT INTO `Order Details` Values (10564, 31, 12.5, 6, 0.05);
INSERT INTO `Order Details` Values (10564, 55, 24, 25, 0.05);
INSERT INTO `Order Details` Values (10565, 24, 4.5, 25, 0.1);
INSERT INTO `Order Details` Values (10565, 64, 33.25, 18, 0.1);
INSERT INTO `Order Details` Values (10566, 11, 21, 35, 0.15);
INSERT INTO `Order Details` Values (10566, 18, 62.5, 18, 0.15);
INSERT INTO `Order Details` Values (10566, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10567, 31, 12.5, 60, 0.2);
INSERT INTO `Order Details` Values (10567, 51, 53, 3, 0.0);
INSERT INTO `Order Details` Values (10567, 59, 55, 40, 0.2);
INSERT INTO `Order Details` Values (10568, 10, 31, 5, 0.0);
INSERT INTO `Order Details` Values (10569, 31, 12.5, 35, 0.2);
INSERT INTO `Order Details` Values (10569, 76, 18, 30, 0.0);
INSERT INTO `Order Details` Values (10570, 11, 21, 15, 0.05);
INSERT INTO `Order Details` Values (10570, 56, 38, 60, 0.05);
INSERT INTO `Order Details` Values (10571, 14, 23.25, 11, 0.15);
INSERT INTO `Order Details` Values (10571, 42, 14, 28, 0.15);
INSERT INTO `Order Details` Values (10572, 16, 17.45, 12, 0.1);
INSERT INTO `Order Details` Values (10572, 32, 32, 10, 0.1);
INSERT INTO `Order Details` Values (10572, 40, 18.4, 50, 0.0);
INSERT INTO `Order Details` Values (10572, 75, 7.75, 15, 0.1);
INSERT INTO `Order Details` Values (10573, 17, 39, 18, 0.0);
INSERT INTO `Order Details` Values (10573, 34, 14, 40, 0.0);
INSERT INTO `Order Details` Values (10573, 53, 32.8, 25, 0.0);
INSERT INTO `Order Details` Values (10574, 33, 2.5, 14, 0.0);
INSERT INTO `Order Details` Values (10574, 40, 18.4, 2, 0.0);
INSERT INTO `Order Details` Values (10574, 62, 49.3, 10, 0.0);
INSERT INTO `Order Details` Values (10574, 64, 33.25, 6, 0.0);
INSERT INTO `Order Details` Values (10575, 59, 55, 12, 0.0);
INSERT INTO `Order Details` Values (10575, 63, 43.9, 6, 0.0);
INSERT INTO `Order Details` Values (10575, 72, 34.8, 30, 0.0);
INSERT INTO `Order Details` Values (10575, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10576, 1, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10576, 31, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10576, 44, 19.45, 21, 0.0);
INSERT INTO `Order Details` Values (10577, 39, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10577, 75, 7.75, 20, 0.0);
INSERT INTO `Order Details` Values (10577, 77, 13, 18, 0.0);
INSERT INTO `Order Details` Values (10578, 35, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10578, 57, 19.5, 6, 0.0);
INSERT INTO `Order Details` Values (10579, 15, 15.5, 10, 0.0);
INSERT INTO `Order Details` Values (10579, 75, 7.75, 21, 0.0);
INSERT INTO `Order Details` Values (10580, 14, 23.25, 15, 0.05);
INSERT INTO `Order Details` Values (10580, 41, 9.65, 9, 0.05);
INSERT INTO `Order Details` Values (10580, 65, 21.05, 30, 0.05);
INSERT INTO `Order Details` Values (10581, 75, 7.75, 50, 0.2);
INSERT INTO `Order Details` Values (10582, 57, 19.5, 4, 0.0);
INSERT INTO `Order Details` Values (10582, 76, 18, 14, 0.0);
INSERT INTO `Order Details` Values (10583, 29, 123.79, 10, 0.0);
INSERT INTO `Order Details` Values (10583, 60, 34, 24, 0.15);
INSERT INTO `Order Details` Values (10583, 69, 36, 10, 0.15);
INSERT INTO `Order Details` Values (10584, 31, 12.5, 50, 0.05);
INSERT INTO `Order Details` Values (10585, 47, 9.5, 15, 0.0);
INSERT INTO `Order Details` Values (10586, 52, 7, 4, 0.15);
INSERT INTO `Order Details` Values (10587, 26, 31.23, 6, 0.0);
INSERT INTO `Order Details` Values (10587, 35, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10587, 77, 13, 20, 0.0);
INSERT INTO `Order Details` Values (10588, 18, 62.5, 40, 0.2);
INSERT INTO `Order Details` Values (10588, 42, 14, 100, 0.2);
INSERT INTO `Order Details` Values (10589, 35, 18, 4, 0.0);
INSERT INTO `Order Details` Values (10590, 1, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10590, 77, 13, 60, 0.05);
INSERT INTO `Order Details` Values (10591, 3, 10, 14, 0.0);
INSERT INTO `Order Details` Values (10591, 7, 30, 10, 0.0);
INSERT INTO `Order Details` Values (10591, 54, 7.45, 50, 0.0);
INSERT INTO `Order Details` Values (10592, 15, 15.5, 25, 0.05);
INSERT INTO `Order Details` Values (10592, 26, 31.23, 5, 0.05);
INSERT INTO `Order Details` Values (10593, 20, 81, 21, 0.2);
INSERT INTO `Order Details` Values (10593, 69, 36, 20, 0.2);
INSERT INTO `Order Details` Values (10593, 76, 18, 4, 0.2);
INSERT INTO `Order Details` Values (10594, 52, 7, 24, 0.0);
INSERT INTO `Order Details` Values (10594, 58, 13.25, 30, 0.0);
INSERT INTO `Order Details` Values (10595, 35, 18, 30, 0.25);
INSERT INTO `Order Details` Values (10595, 61, 28.5, 120, 0.25);
INSERT INTO `Order Details` Values (10595, 69, 36, 65, 0.25);
INSERT INTO `Order Details` Values (10596, 56, 38, 5, 0.2);
INSERT INTO `Order Details` Values (10596, 63, 43.9, 24, 0.2);
INSERT INTO `Order Details` Values (10596, 75, 7.75, 30, 0.2);
INSERT INTO `Order Details` Values (10597, 24, 4.5, 35, 0.2);
INSERT INTO `Order Details` Values (10597, 57, 19.5, 20, 0.0);
INSERT INTO `Order Details` Values (10597, 65, 21.05, 12, 0.2);
INSERT INTO `Order Details` Values (10598, 27, 43.9, 50, 0.0);
INSERT INTO `Order Details` Values (10598, 71, 21.5, 9, 0.0);
INSERT INTO `Order Details` Values (10599, 62, 49.3, 10, 0.0);
INSERT INTO `Order Details` Values (10600, 54, 7.45, 4, 0.0);
INSERT INTO `Order Details` Values (10600, 73, 15, 30, 0.0);
INSERT INTO `Order Details` Values (10601, 13, 6, 60, 0.0);
INSERT INTO `Order Details` Values (10601, 59, 55, 35, 0.0);
INSERT INTO `Order Details` Values (10602, 77, 13, 5, 0.25);
INSERT INTO `Order Details` Values (10603, 22, 21, 48, 0.0);
INSERT INTO `Order Details` Values (10603, 49, 20, 25, 0.05);
INSERT INTO `Order Details` Values (10604, 48, 12.75, 6, 0.1);
INSERT INTO `Order Details` Values (10604, 76, 18, 10, 0.1);
INSERT INTO `Order Details` Values (10605, 16, 17.45, 30, 0.05);
INSERT INTO `Order Details` Values (10605, 59, 55, 20, 0.05);
INSERT INTO `Order Details` Values (10605, 60, 34, 70, 0.05);
INSERT INTO `Order Details` Values (10605, 71, 21.5, 15, 0.05);
INSERT INTO `Order Details` Values (10606, 4, 22, 20, 0.2);
INSERT INTO `Order Details` Values (10606, 55, 24, 20, 0.2);
INSERT INTO `Order Details` Values (10606, 62, 49.3, 10, 0.2);
INSERT INTO `Order Details` Values (10607, 7, 30, 45, 0.0);
INSERT INTO `Order Details` Values (10607, 17, 39, 100, 0.0);
INSERT INTO `Order Details` Values (10607, 33, 2.5, 14, 0.0);
INSERT INTO `Order Details` Values (10607, 40, 18.4, 42, 0.0);
INSERT INTO `Order Details` Values (10607, 72, 34.8, 12, 0.0);
INSERT INTO `Order Details` Values (10608, 56, 38, 28, 0.0);
INSERT INTO `Order Details` Values (10609, 1, 18, 3, 0.0);
INSERT INTO `Order Details` Values (10609, 10, 31, 10, 0.0);
INSERT INTO `Order Details` Values (10609, 21, 10, 6, 0.0);
INSERT INTO `Order Details` Values (10610, 36, 19, 21, 0.25);
INSERT INTO `Order Details` Values (10611, 1, 18, 6, 0.0);
INSERT INTO `Order Details` Values (10611, 2, 19, 10, 0.0);
INSERT INTO `Order Details` Values (10611, 60, 34, 15, 0.0);
INSERT INTO `Order Details` Values (10612, 10, 31, 70, 0.0);
INSERT INTO `Order Details` Values (10612, 36, 19, 55, 0.0);
INSERT INTO `Order Details` Values (10612, 49, 20, 18, 0.0);
INSERT INTO `Order Details` Values (10612, 60, 34, 40, 0.0);
INSERT INTO `Order Details` Values (10612, 76, 18, 80, 0.0);
INSERT INTO `Order Details` Values (10613, 13, 6, 8, 0.1);
INSERT INTO `Order Details` Values (10613, 75, 7.75, 40, 0.0);
INSERT INTO `Order Details` Values (10614, 11, 21, 14, 0.0);
INSERT INTO `Order Details` Values (10614, 21, 10, 8, 0.0);
INSERT INTO `Order Details` Values (10614, 39, 18, 5, 0.0);
INSERT INTO `Order Details` Values (10615, 55, 24, 5, 0.0);
INSERT INTO `Order Details` Values (10616, 38, 263.5, 15, 0.05);
INSERT INTO `Order Details` Values (10616, 56, 38, 14, 0.0);
INSERT INTO `Order Details` Values (10616, 70, 15, 15, 0.05);
INSERT INTO `Order Details` Values (10616, 71, 21.5, 15, 0.05);
INSERT INTO `Order Details` Values (10617, 59, 55, 30, 0.15);
INSERT INTO `Order Details` Values (10618, 6, 25, 70, 0.0);
INSERT INTO `Order Details` Values (10618, 56, 38, 20, 0.0);
INSERT INTO `Order Details` Values (10618, 68, 12.5, 15, 0.0);
INSERT INTO `Order Details` Values (10619, 21, 10, 42, 0.0);
INSERT INTO `Order Details` Values (10619, 22, 21, 40, 0.0);
INSERT INTO `Order Details` Values (10620, 24, 4.5, 5, 0.0);
INSERT INTO `Order Details` Values (10620, 52, 7, 5, 0.0);
INSERT INTO `Order Details` Values (10621, 19, 9.2, 5, 0.0);
INSERT INTO `Order Details` Values (10621, 23, 9, 10, 0.0);
INSERT INTO `Order Details` Values (10621, 70, 15, 20, 0.0);
INSERT INTO `Order Details` Values (10621, 71, 21.5, 15, 0.0);
INSERT INTO `Order Details` Values (10622, 2, 19, 20, 0.0);
INSERT INTO `Order Details` Values (10622, 68, 12.5, 18, 0.2);
INSERT INTO `Order Details` Values (10623, 14, 23.25, 21, 0.0);
INSERT INTO `Order Details` Values (10623, 19, 9.2, 15, 0.1);
INSERT INTO `Order Details` Values (10623, 21, 10, 25, 0.1);
INSERT INTO `Order Details` Values (10623, 24, 4.5, 3, 0.0);
INSERT INTO `Order Details` Values (10623, 35, 18, 30, 0.1);
INSERT INTO `Order Details` Values (10624, 28, 45.6, 10, 0.0);
INSERT INTO `Order Details` Values (10624, 29, 123.79, 6, 0.0);
INSERT INTO `Order Details` Values (10624, 44, 19.45, 10, 0.0);
INSERT INTO `Order Details` Values (10625, 14, 23.25, 3, 0.0);
INSERT INTO `Order Details` Values (10625, 42, 14, 5, 0.0);
INSERT INTO `Order Details` Values (10625, 60, 34, 10, 0.0);
INSERT INTO `Order Details` Values (10626, 53, 32.8, 12, 0.0);
INSERT INTO `Order Details` Values (10626, 60, 34, 20, 0.0);
INSERT INTO `Order Details` Values (10626, 71, 21.5, 20, 0.0);
INSERT INTO `Order Details` Values (10627, 62, 49.3, 15, 0.0);
INSERT INTO `Order Details` Values (10627, 73, 15, 35, 0.15);
INSERT INTO `Order Details` Values (10628, 1, 18, 25, 0.0);
INSERT INTO `Order Details` Values (10629, 29, 123.79, 20, 0.0);
INSERT INTO `Order Details` Values (10629, 64, 33.25, 9, 0.0);
INSERT INTO `Order Details` Values (10630, 55, 24, 12, 0.05);
INSERT INTO `Order Details` Values (10630, 76, 18, 35, 0.0);
INSERT INTO `Order Details` Values (10631, 75, 7.75, 8, 0.1);
INSERT INTO `Order Details` Values (10632, 2, 19, 30, 0.05);
INSERT INTO `Order Details` Values (10632, 33, 2.5, 20, 0.05);
INSERT INTO `Order Details` Values (10633, 12, 38, 36, 0.15);
INSERT INTO `Order Details` Values (10633, 13, 6, 13, 0.15);
INSERT INTO `Order Details` Values (10633, 26, 31.23, 35, 0.15);
INSERT INTO `Order Details` Values (10633, 62, 49.3, 80, 0.15);
INSERT INTO `Order Details` Values (10634, 7, 30, 35, 0.0);
INSERT INTO `Order Details` Values (10634, 18, 62.5, 50, 0.0);
INSERT INTO `Order Details` Values (10634, 51, 53, 15, 0.0);
INSERT INTO `Order Details` Values (10634, 75, 7.75, 2, 0.0);
INSERT INTO `Order Details` Values (10635, 4, 22, 10, 0.1);
INSERT INTO `Order Details` Values (10635, 5, 21.35, 15, 0.1);
INSERT INTO `Order Details` Values (10635, 22, 21, 40, 0.0);
INSERT INTO `Order Details` Values (10636, 4, 22, 25, 0.0);
INSERT INTO `Order Details` Values (10636, 58, 13.25, 6, 0.0);
INSERT INTO `Order Details` Values (10637, 11, 21, 10, 0.0);
INSERT INTO `Order Details` Values (10637, 50, 16.25, 25, 0.05);
INSERT INTO `Order Details` Values (10637, 56, 38, 60, 0.05);
INSERT INTO `Order Details` Values (10638, 45, 9.5, 20, 0.0);
INSERT INTO `Order Details` Values (10638, 65, 21.05, 21, 0.0);
INSERT INTO `Order Details` Values (10638, 72, 34.8, 60, 0.0);
INSERT INTO `Order Details` Values (10639, 18, 62.5, 8, 0.0);
INSERT INTO `Order Details` Values (10640, 69, 36, 20, 0.25);
INSERT INTO `Order Details` Values (10640, 70, 15, 15, 0.25);
INSERT INTO `Order Details` Values (10641, 2, 19, 50, 0.0);
INSERT INTO `Order Details` Values (10641, 40, 18.4, 60, 0.0);
INSERT INTO `Order Details` Values (10642, 21, 10, 30, 0.2);
INSERT INTO `Order Details` Values (10642, 61, 28.5, 20, 0.2);
INSERT INTO `Order Details` Values (10643, 28, 45.6, 15, 0.25);
INSERT INTO `Order Details` Values (10643, 39, 18, 21, 0.25);
INSERT INTO `Order Details` Values (10643, 46, 12, 2, 0.25);
INSERT INTO `Order Details` Values (10644, 18, 62.5, 4, 0.1);
INSERT INTO `Order Details` Values (10644, 43, 46, 20, 0.0);
INSERT INTO `Order Details` Values (10644, 46, 12, 21, 0.1);
INSERT INTO `Order Details` Values (10645, 18, 62.5, 20, 0.0);
INSERT INTO `Order Details` Values (10645, 36, 19, 15, 0.0);
INSERT INTO `Order Details` Values (10646, 1, 18, 15, 0.25);
INSERT INTO `Order Details` Values (10646, 10, 31, 18, 0.25);
INSERT INTO `Order Details` Values (10646, 71, 21.5, 30, 0.25);
INSERT INTO `Order Details` Values (10646, 77, 13, 35, 0.25);
INSERT INTO `Order Details` Values (10647, 19, 9.2, 30, 0.0);
INSERT INTO `Order Details` Values (10647, 39, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10648, 22, 21, 15, 0.0);
INSERT INTO `Order Details` Values (10648, 24, 4.5, 15, 0.15);
INSERT INTO `Order Details` Values (10649, 28, 45.6, 20, 0.0);
INSERT INTO `Order Details` Values (10649, 72, 34.8, 15, 0.0);
INSERT INTO `Order Details` Values (10650, 30, 25.89, 30, 0.0);
INSERT INTO `Order Details` Values (10650, 53, 32.8, 25, 0.05);
INSERT INTO `Order Details` Values (10650, 54, 7.45, 30, 0.0);
INSERT INTO `Order Details` Values (10651, 19, 9.2, 12, 0.25);
INSERT INTO `Order Details` Values (10651, 22, 21, 20, 0.25);
INSERT INTO `Order Details` Values (10652, 30, 25.89, 2, 0.25);
INSERT INTO `Order Details` Values (10652, 42, 14, 20, 0.0);
INSERT INTO `Order Details` Values (10653, 16, 17.45, 30, 0.1);
INSERT INTO `Order Details` Values (10653, 60, 34, 20, 0.1);
INSERT INTO `Order Details` Values (10654, 4, 22, 12, 0.1);
INSERT INTO `Order Details` Values (10654, 39, 18, 20, 0.1);
INSERT INTO `Order Details` Values (10654, 54, 7.45, 6, 0.1);
INSERT INTO `Order Details` Values (10655, 41, 9.65, 20, 0.2);
INSERT INTO `Order Details` Values (10656, 14, 23.25, 3, 0.1);
INSERT INTO `Order Details` Values (10656, 44, 19.45, 28, 0.1);
INSERT INTO `Order Details` Values (10656, 47, 9.5, 6, 0.1);
INSERT INTO `Order Details` Values (10657, 15, 15.5, 50, 0.0);
INSERT INTO `Order Details` Values (10657, 41, 9.65, 24, 0.0);
INSERT INTO `Order Details` Values (10657, 46, 12, 45, 0.0);
INSERT INTO `Order Details` Values (10657, 47, 9.5, 10, 0.0);
INSERT INTO `Order Details` Values (10657, 56, 38, 45, 0.0);
INSERT INTO `Order Details` Values (10657, 60, 34, 30, 0.0);
INSERT INTO `Order Details` Values (10658, 21, 10, 60, 0.0);
INSERT INTO `Order Details` Values (10658, 40, 18.4, 70, 0.05);
INSERT INTO `Order Details` Values (10658, 60, 34, 55, 0.05);
INSERT INTO `Order Details` Values (10658, 77, 13, 70, 0.05);
INSERT INTO `Order Details` Values (10659, 31, 12.5, 20, 0.05);
INSERT INTO `Order Details` Values (10659, 40, 18.4, 24, 0.05);
INSERT INTO `Order Details` Values (10659, 70, 15, 40, 0.05);
INSERT INTO `Order Details` Values (10660, 20, 81, 21, 0.0);
INSERT INTO `Order Details` Values (10661, 39, 18, 3, 0.2);
INSERT INTO `Order Details` Values (10661, 58, 13.25, 49, 0.2);
INSERT INTO `Order Details` Values (10662, 68, 12.5, 10, 0.0);
INSERT INTO `Order Details` Values (10663, 40, 18.4, 30, 0.05);
INSERT INTO `Order Details` Values (10663, 42, 14, 30, 0.05);
INSERT INTO `Order Details` Values (10663, 51, 53, 20, 0.05);
INSERT INTO `Order Details` Values (10664, 10, 31, 24, 0.15);
INSERT INTO `Order Details` Values (10664, 56, 38, 12, 0.15);
INSERT INTO `Order Details` Values (10664, 65, 21.05, 15, 0.15);
INSERT INTO `Order Details` Values (10665, 51, 53, 20, 0.0);
INSERT INTO `Order Details` Values (10665, 59, 55, 1, 0.0);
INSERT INTO `Order Details` Values (10665, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10666, 29, 123.79, 36, 0.0);
INSERT INTO `Order Details` Values (10666, 65, 21.05, 10, 0.0);
INSERT INTO `Order Details` Values (10667, 69, 36, 45, 0.2);
INSERT INTO `Order Details` Values (10667, 71, 21.5, 14, 0.2);
INSERT INTO `Order Details` Values (10668, 31, 12.5, 8, 0.1);
INSERT INTO `Order Details` Values (10668, 55, 24, 4, 0.1);
INSERT INTO `Order Details` Values (10668, 64, 33.25, 15, 0.1);
INSERT INTO `Order Details` Values (10669, 36, 19, 30, 0.0);
INSERT INTO `Order Details` Values (10670, 23, 9, 32, 0.0);
INSERT INTO `Order Details` Values (10670, 46, 12, 60, 0.0);
INSERT INTO `Order Details` Values (10670, 67, 14, 25, 0.0);
INSERT INTO `Order Details` Values (10670, 73, 15, 50, 0.0);
INSERT INTO `Order Details` Values (10670, 75, 7.75, 25, 0.0);
INSERT INTO `Order Details` Values (10671, 16, 17.45, 10, 0.0);
INSERT INTO `Order Details` Values (10671, 62, 49.3, 10, 0.0);
INSERT INTO `Order Details` Values (10671, 65, 21.05, 12, 0.0);
INSERT INTO `Order Details` Values (10672, 38, 263.5, 15, 0.1);
INSERT INTO `Order Details` Values (10672, 71, 21.5, 12, 0.0);
INSERT INTO `Order Details` Values (10673, 16, 17.45, 3, 0.0);
INSERT INTO `Order Details` Values (10673, 42, 14, 6, 0.0);
INSERT INTO `Order Details` Values (10673, 43, 46, 6, 0.0);
INSERT INTO `Order Details` Values (10674, 23, 9, 5, 0.0);
INSERT INTO `Order Details` Values (10675, 14, 23.25, 30, 0.0);
INSERT INTO `Order Details` Values (10675, 53, 32.8, 10, 0.0);
INSERT INTO `Order Details` Values (10675, 58, 13.25, 30, 0.0);
INSERT INTO `Order Details` Values (10676, 10, 31, 2, 0.0);
INSERT INTO `Order Details` Values (10676, 19, 9.2, 7, 0.0);
INSERT INTO `Order Details` Values (10676, 44, 19.45, 21, 0.0);
INSERT INTO `Order Details` Values (10677, 26, 31.23, 30, 0.15);
INSERT INTO `Order Details` Values (10677, 33, 2.5, 8, 0.15);
INSERT INTO `Order Details` Values (10678, 12, 38, 100, 0.0);
INSERT INTO `Order Details` Values (10678, 33, 2.5, 30, 0.0);
INSERT INTO `Order Details` Values (10678, 41, 9.65, 120, 0.0);
INSERT INTO `Order Details` Values (10678, 54, 7.45, 30, 0.0);
INSERT INTO `Order Details` Values (10679, 59, 55, 12, 0.0);
INSERT INTO `Order Details` Values (10680, 16, 17.45, 50, 0.25);
INSERT INTO `Order Details` Values (10680, 31, 12.5, 20, 0.25);
INSERT INTO `Order Details` Values (10680, 42, 14, 40, 0.25);
INSERT INTO `Order Details` Values (10681, 19, 9.2, 30, 0.1);
INSERT INTO `Order Details` Values (10681, 21, 10, 12, 0.1);
INSERT INTO `Order Details` Values (10681, 64, 33.25, 28, 0.0);
INSERT INTO `Order Details` Values (10682, 33, 2.5, 30, 0.0);
INSERT INTO `Order Details` Values (10682, 66, 17, 4, 0.0);
INSERT INTO `Order Details` Values (10682, 75, 7.75, 30, 0.0);
INSERT INTO `Order Details` Values (10683, 52, 7, 9, 0.0);
INSERT INTO `Order Details` Values (10684, 40, 18.4, 20, 0.0);
INSERT INTO `Order Details` Values (10684, 47, 9.5, 40, 0.0);
INSERT INTO `Order Details` Values (10684, 60, 34, 30, 0.0);
INSERT INTO `Order Details` Values (10685, 10, 31, 20, 0.0);
INSERT INTO `Order Details` Values (10685, 41, 9.65, 4, 0.0);
INSERT INTO `Order Details` Values (10685, 47, 9.5, 15, 0.0);
INSERT INTO `Order Details` Values (10686, 17, 39, 30, 0.2);
INSERT INTO `Order Details` Values (10686, 26, 31.23, 15, 0.0);
INSERT INTO `Order Details` Values (10687, 9, 97, 50, 0.25);
INSERT INTO `Order Details` Values (10687, 29, 123.79, 10, 0.0);
INSERT INTO `Order Details` Values (10687, 36, 19, 6, 0.25);
INSERT INTO `Order Details` Values (10688, 10, 31, 18, 0.1);
INSERT INTO `Order Details` Values (10688, 28, 45.6, 60, 0.1);
INSERT INTO `Order Details` Values (10688, 34, 14, 14, 0.0);
INSERT INTO `Order Details` Values (10689, 1, 18, 35, 0.25);
INSERT INTO `Order Details` Values (10690, 56, 38, 20, 0.25);
INSERT INTO `Order Details` Values (10690, 77, 13, 30, 0.25);
INSERT INTO `Order Details` Values (10691, 1, 18, 30, 0.0);
INSERT INTO `Order Details` Values (10691, 29, 123.79, 40, 0.0);
INSERT INTO `Order Details` Values (10691, 43, 46, 40, 0.0);
INSERT INTO `Order Details` Values (10691, 44, 19.45, 24, 0.0);
INSERT INTO `Order Details` Values (10691, 62, 49.3, 48, 0.0);
INSERT INTO `Order Details` Values (10692, 63, 43.9, 20, 0.0);
INSERT INTO `Order Details` Values (10693, 9, 97, 6, 0.0);
INSERT INTO `Order Details` Values (10693, 54, 7.45, 60, 0.15);
INSERT INTO `Order Details` Values (10693, 69, 36, 30, 0.15);
INSERT INTO `Order Details` Values (10693, 73, 15, 15, 0.15);
INSERT INTO `Order Details` Values (10694, 7, 30, 90, 0.0);
INSERT INTO `Order Details` Values (10694, 59, 55, 25, 0.0);
INSERT INTO `Order Details` Values (10694, 70, 15, 50, 0.0);
INSERT INTO `Order Details` Values (10695, 8, 40, 10, 0.0);
INSERT INTO `Order Details` Values (10695, 12, 38, 4, 0.0);
INSERT INTO `Order Details` Values (10695, 24, 4.5, 20, 0.0);
INSERT INTO `Order Details` Values (10696, 17, 39, 20, 0.0);
INSERT INTO `Order Details` Values (10696, 46, 12, 18, 0.0);
INSERT INTO `Order Details` Values (10697, 19, 9.2, 7, 0.25);
INSERT INTO `Order Details` Values (10697, 35, 18, 9, 0.25);
INSERT INTO `Order Details` Values (10697, 58, 13.25, 30, 0.25);
INSERT INTO `Order Details` Values (10697, 70, 15, 30, 0.25);
INSERT INTO `Order Details` Values (10698, 11, 21, 15, 0.0);
INSERT INTO `Order Details` Values (10698, 17, 39, 8, 0.05);
INSERT INTO `Order Details` Values (10698, 29, 123.79, 12, 0.05);
INSERT INTO `Order Details` Values (10698, 65, 21.05, 65, 0.05);
INSERT INTO `Order Details` Values (10698, 70, 15, 8, 0.05);
INSERT INTO `Order Details` Values (10699, 47, 9.5, 12, 0.0);
INSERT INTO `Order Details` Values (10700, 1, 18, 5, 0.2);
INSERT INTO `Order Details` Values (10700, 34, 14, 12, 0.2);
INSERT INTO `Order Details` Values (10700, 68, 12.5, 40, 0.2);
INSERT INTO `Order Details` Values (10700, 71, 21.5, 60, 0.2);
INSERT INTO `Order Details` Values (10701, 59, 55, 42, 0.15);
INSERT INTO `Order Details` Values (10701, 71, 21.5, 20, 0.15);
INSERT INTO `Order Details` Values (10701, 76, 18, 35, 0.15);
INSERT INTO `Order Details` Values (10702, 3, 10, 6, 0.0);
INSERT INTO `Order Details` Values (10702, 76, 18, 15, 0.0);
INSERT INTO `Order Details` Values (10703, 2, 19, 5, 0.0);
INSERT INTO `Order Details` Values (10703, 59, 55, 35, 0.0);
INSERT INTO `Order Details` Values (10703, 73, 15, 35, 0.0);
INSERT INTO `Order Details` Values (10704, 4, 22, 6, 0.0);
INSERT INTO `Order Details` Values (10704, 24, 4.5, 35, 0.0);
INSERT INTO `Order Details` Values (10704, 48, 12.75, 24, 0.0);
INSERT INTO `Order Details` Values (10705, 31, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10705, 32, 32, 4, 0.0);
INSERT INTO `Order Details` Values (10706, 16, 17.45, 20, 0.0);
INSERT INTO `Order Details` Values (10706, 43, 46, 24, 0.0);
INSERT INTO `Order Details` Values (10706, 59, 55, 8, 0.0);
INSERT INTO `Order Details` Values (10707, 55, 24, 21, 0.0);
INSERT INTO `Order Details` Values (10707, 57, 19.5, 40, 0.0);
INSERT INTO `Order Details` Values (10707, 70, 15, 28, 0.15);
INSERT INTO `Order Details` Values (10708, 5, 21.35, 4, 0.0);
INSERT INTO `Order Details` Values (10708, 36, 19, 5, 0.0);
INSERT INTO `Order Details` Values (10709, 8, 40, 40, 0.0);
INSERT INTO `Order Details` Values (10709, 51, 53, 28, 0.0);
INSERT INTO `Order Details` Values (10709, 60, 34, 10, 0.0);
INSERT INTO `Order Details` Values (10710, 19, 9.2, 5, 0.0);
INSERT INTO `Order Details` Values (10710, 47, 9.5, 5, 0.0);
INSERT INTO `Order Details` Values (10711, 19, 9.2, 12, 0.0);
INSERT INTO `Order Details` Values (10711, 41, 9.65, 42, 0.0);
INSERT INTO `Order Details` Values (10711, 53, 32.8, 120, 0.0);
INSERT INTO `Order Details` Values (10712, 53, 32.8, 3, 0.05);
INSERT INTO `Order Details` Values (10712, 56, 38, 30, 0.0);
INSERT INTO `Order Details` Values (10713, 10, 31, 18, 0.0);
INSERT INTO `Order Details` Values (10713, 26, 31.23, 30, 0.0);
INSERT INTO `Order Details` Values (10713, 45, 9.5, 110, 0.0);
INSERT INTO `Order Details` Values (10713, 46, 12, 24, 0.0);
INSERT INTO `Order Details` Values (10714, 2, 19, 30, 0.25);
INSERT INTO `Order Details` Values (10714, 17, 39, 27, 0.25);
INSERT INTO `Order Details` Values (10714, 47, 9.5, 50, 0.25);
INSERT INTO `Order Details` Values (10714, 56, 38, 18, 0.25);
INSERT INTO `Order Details` Values (10714, 58, 13.25, 12, 0.25);
INSERT INTO `Order Details` Values (10715, 10, 31, 21, 0.0);
INSERT INTO `Order Details` Values (10715, 71, 21.5, 30, 0.0);
INSERT INTO `Order Details` Values (10716, 21, 10, 5, 0.0);
INSERT INTO `Order Details` Values (10716, 51, 53, 7, 0.0);
INSERT INTO `Order Details` Values (10716, 61, 28.5, 10, 0.0);
INSERT INTO `Order Details` Values (10717, 21, 10, 32, 0.05);
INSERT INTO `Order Details` Values (10717, 54, 7.45, 15, 0.0);
INSERT INTO `Order Details` Values (10717, 69, 36, 25, 0.05);
INSERT INTO `Order Details` Values (10718, 12, 38, 36, 0.0);
INSERT INTO `Order Details` Values (10718, 16, 17.45, 20, 0.0);
INSERT INTO `Order Details` Values (10718, 36, 19, 40, 0.0);
INSERT INTO `Order Details` Values (10718, 62, 49.3, 20, 0.0);
INSERT INTO `Order Details` Values (10719, 18, 62.5, 12, 0.25);
INSERT INTO `Order Details` Values (10719, 30, 25.89, 3, 0.25);
INSERT INTO `Order Details` Values (10719, 54, 7.45, 40, 0.25);
INSERT INTO `Order Details` Values (10720, 35, 18, 21, 0.0);
INSERT INTO `Order Details` Values (10720, 71, 21.5, 8, 0.0);
INSERT INTO `Order Details` Values (10721, 44, 19.45, 50, 0.05);
INSERT INTO `Order Details` Values (10722, 2, 19, 3, 0.0);
INSERT INTO `Order Details` Values (10722, 31, 12.5, 50, 0.0);
INSERT INTO `Order Details` Values (10722, 68, 12.5, 45, 0.0);
INSERT INTO `Order Details` Values (10722, 75, 7.75, 42, 0.0);
INSERT INTO `Order Details` Values (10723, 26, 31.23, 15, 0.0);
INSERT INTO `Order Details` Values (10724, 10, 31, 16, 0.0);
INSERT INTO `Order Details` Values (10724, 61, 28.5, 5, 0.0);
INSERT INTO `Order Details` Values (10725, 41, 9.65, 12, 0.0);
INSERT INTO `Order Details` Values (10725, 52, 7, 4, 0.0);
INSERT INTO `Order Details` Values (10725, 55, 24, 6, 0.0);
INSERT INTO `Order Details` Values (10726, 4, 22, 25, 0.0);
INSERT INTO `Order Details` Values (10726, 11, 21, 5, 0.0);
INSERT INTO `Order Details` Values (10727, 17, 39, 20, 0.05);
INSERT INTO `Order Details` Values (10727, 56, 38, 10, 0.05);
INSERT INTO `Order Details` Values (10727, 59, 55, 10, 0.05);
INSERT INTO `Order Details` Values (10728, 30, 25.89, 15, 0.0);
INSERT INTO `Order Details` Values (10728, 40, 18.4, 6, 0.0);
INSERT INTO `Order Details` Values (10728, 55, 24, 12, 0.0);
INSERT INTO `Order Details` Values (10728, 60, 34, 15, 0.0);
INSERT INTO `Order Details` Values (10729, 1, 18, 50, 0.0);
INSERT INTO `Order Details` Values (10729, 21, 10, 30, 0.0);
INSERT INTO `Order Details` Values (10729, 50, 16.25, 40, 0.0);
INSERT INTO `Order Details` Values (10730, 16, 17.45, 15, 0.05);
INSERT INTO `Order Details` Values (10730, 31, 12.5, 3, 0.05);
INSERT INTO `Order Details` Values (10730, 65, 21.05, 10, 0.05);
INSERT INTO `Order Details` Values (10731, 21, 10, 40, 0.05);
INSERT INTO `Order Details` Values (10731, 51, 53, 30, 0.05);
INSERT INTO `Order Details` Values (10732, 76, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10733, 14, 23.25, 16, 0.0);
INSERT INTO `Order Details` Values (10733, 28, 45.6, 20, 0.0);
INSERT INTO `Order Details` Values (10733, 52, 7, 25, 0.0);
INSERT INTO `Order Details` Values (10734, 6, 25, 30, 0.0);
INSERT INTO `Order Details` Values (10734, 30, 25.89, 15, 0.0);
INSERT INTO `Order Details` Values (10734, 76, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10735, 61, 28.5, 20, 0.1);
INSERT INTO `Order Details` Values (10735, 77, 13, 2, 0.1);
INSERT INTO `Order Details` Values (10736, 65, 21.05, 40, 0.0);
INSERT INTO `Order Details` Values (10736, 75, 7.75, 20, 0.0);
INSERT INTO `Order Details` Values (10737, 13, 6, 4, 0.0);
INSERT INTO `Order Details` Values (10737, 41, 9.65, 12, 0.0);
INSERT INTO `Order Details` Values (10738, 16, 17.45, 3, 0.0);
INSERT INTO `Order Details` Values (10739, 36, 19, 6, 0.0);
INSERT INTO `Order Details` Values (10739, 52, 7, 18, 0.0);
INSERT INTO `Order Details` Values (10740, 28, 45.6, 5, 0.2);
INSERT INTO `Order Details` Values (10740, 35, 18, 35, 0.2);
INSERT INTO `Order Details` Values (10740, 45, 9.5, 40, 0.2);
INSERT INTO `Order Details` Values (10740, 56, 38, 14, 0.2);
INSERT INTO `Order Details` Values (10741, 2, 19, 15, 0.2);
INSERT INTO `Order Details` Values (10742, 3, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10742, 60, 34, 50, 0.0);
INSERT INTO `Order Details` Values (10742, 72, 34.8, 35, 0.0);
INSERT INTO `Order Details` Values (10743, 46, 12, 28, 0.05);
INSERT INTO `Order Details` Values (10744, 40, 18.4, 50, 0.2);
INSERT INTO `Order Details` Values (10745, 18, 62.5, 24, 0.0);
INSERT INTO `Order Details` Values (10745, 44, 19.45, 16, 0.0);
INSERT INTO `Order Details` Values (10745, 59, 55, 45, 0.0);
INSERT INTO `Order Details` Values (10745, 72, 34.8, 7, 0.0);
INSERT INTO `Order Details` Values (10746, 13, 6, 6, 0.0);
INSERT INTO `Order Details` Values (10746, 42, 14, 28, 0.0);
INSERT INTO `Order Details` Values (10746, 62, 49.3, 9, 0.0);
INSERT INTO `Order Details` Values (10746, 69, 36, 40, 0.0);
INSERT INTO `Order Details` Values (10747, 31, 12.5, 8, 0.0);
INSERT INTO `Order Details` Values (10747, 41, 9.65, 35, 0.0);
INSERT INTO `Order Details` Values (10747, 63, 43.9, 9, 0.0);
INSERT INTO `Order Details` Values (10747, 69, 36, 30, 0.0);
INSERT INTO `Order Details` Values (10748, 23, 9, 44, 0.0);
INSERT INTO `Order Details` Values (10748, 40, 18.4, 40, 0.0);
INSERT INTO `Order Details` Values (10748, 56, 38, 28, 0.0);
INSERT INTO `Order Details` Values (10749, 56, 38, 15, 0.0);
INSERT INTO `Order Details` Values (10749, 59, 55, 6, 0.0);
INSERT INTO `Order Details` Values (10749, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10750, 14, 23.25, 5, 0.15);
INSERT INTO `Order Details` Values (10750, 45, 9.5, 40, 0.15);
INSERT INTO `Order Details` Values (10750, 59, 55, 25, 0.15);
INSERT INTO `Order Details` Values (10751, 26, 31.23, 12, 0.1);
INSERT INTO `Order Details` Values (10751, 30, 25.89, 30, 0.0);
INSERT INTO `Order Details` Values (10751, 50, 16.25, 20, 0.1);
INSERT INTO `Order Details` Values (10751, 73, 15, 15, 0.0);
INSERT INTO `Order Details` Values (10752, 1, 18, 8, 0.0);
INSERT INTO `Order Details` Values (10752, 69, 36, 3, 0.0);
INSERT INTO `Order Details` Values (10753, 45, 9.5, 4, 0.0);
INSERT INTO `Order Details` Values (10753, 74, 10, 5, 0.0);
INSERT INTO `Order Details` Values (10754, 40, 18.4, 3, 0.0);
INSERT INTO `Order Details` Values (10755, 47, 9.5, 30, 0.25);
INSERT INTO `Order Details` Values (10755, 56, 38, 30, 0.25);
INSERT INTO `Order Details` Values (10755, 57, 19.5, 14, 0.25);
INSERT INTO `Order Details` Values (10755, 69, 36, 25, 0.25);
INSERT INTO `Order Details` Values (10756, 18, 62.5, 21, 0.2);
INSERT INTO `Order Details` Values (10756, 36, 19, 20, 0.2);
INSERT INTO `Order Details` Values (10756, 68, 12.5, 6, 0.2);
INSERT INTO `Order Details` Values (10756, 69, 36, 20, 0.2);
INSERT INTO `Order Details` Values (10757, 34, 14, 30, 0.0);
INSERT INTO `Order Details` Values (10757, 59, 55, 7, 0.0);
INSERT INTO `Order Details` Values (10757, 62, 49.3, 30, 0.0);
INSERT INTO `Order Details` Values (10757, 64, 33.25, 24, 0.0);
INSERT INTO `Order Details` Values (10758, 26, 31.23, 20, 0.0);
INSERT INTO `Order Details` Values (10758, 52, 7, 60, 0.0);
INSERT INTO `Order Details` Values (10758, 70, 15, 40, 0.0);
INSERT INTO `Order Details` Values (10759, 32, 32, 10, 0.0);
INSERT INTO `Order Details` Values (10760, 25, 14, 12, 0.25);
INSERT INTO `Order Details` Values (10760, 27, 43.9, 40, 0.0);
INSERT INTO `Order Details` Values (10760, 43, 46, 30, 0.25);
INSERT INTO `Order Details` Values (10761, 25, 14, 35, 0.25);
INSERT INTO `Order Details` Values (10761, 75, 7.75, 18, 0.0);
INSERT INTO `Order Details` Values (10762, 39, 18, 16, 0.0);
INSERT INTO `Order Details` Values (10762, 47, 9.5, 30, 0.0);
INSERT INTO `Order Details` Values (10762, 51, 53, 28, 0.0);
INSERT INTO `Order Details` Values (10762, 56, 38, 60, 0.0);
INSERT INTO `Order Details` Values (10763, 21, 10, 40, 0.0);
INSERT INTO `Order Details` Values (10763, 22, 21, 6, 0.0);
INSERT INTO `Order Details` Values (10763, 24, 4.5, 20, 0.0);
INSERT INTO `Order Details` Values (10764, 3, 10, 20, 0.1);
INSERT INTO `Order Details` Values (10764, 39, 18, 130, 0.1);
INSERT INTO `Order Details` Values (10765, 65, 21.05, 80, 0.1);
INSERT INTO `Order Details` Values (10766, 2, 19, 40, 0.0);
INSERT INTO `Order Details` Values (10766, 7, 30, 35, 0.0);
INSERT INTO `Order Details` Values (10766, 68, 12.5, 40, 0.0);
INSERT INTO `Order Details` Values (10767, 42, 14, 2, 0.0);
INSERT INTO `Order Details` Values (10768, 22, 21, 4, 0.0);
INSERT INTO `Order Details` Values (10768, 31, 12.5, 50, 0.0);
INSERT INTO `Order Details` Values (10768, 60, 34, 15, 0.0);
INSERT INTO `Order Details` Values (10768, 71, 21.5, 12, 0.0);
INSERT INTO `Order Details` Values (10769, 41, 9.65, 30, 0.05);
INSERT INTO `Order Details` Values (10769, 52, 7, 15, 0.05);
INSERT INTO `Order Details` Values (10769, 61, 28.5, 20, 0.0);
INSERT INTO `Order Details` Values (10769, 62, 49.3, 15, 0.0);
INSERT INTO `Order Details` Values (10770, 11, 21, 15, 0.25);
INSERT INTO `Order Details` Values (10771, 71, 21.5, 16, 0.0);
INSERT INTO `Order Details` Values (10772, 29, 123.79, 18, 0.0);
INSERT INTO `Order Details` Values (10772, 59, 55, 25, 0.0);
INSERT INTO `Order Details` Values (10773, 17, 39, 33, 0.0);
INSERT INTO `Order Details` Values (10773, 31, 12.5, 70, 0.2);
INSERT INTO `Order Details` Values (10773, 75, 7.75, 7, 0.2);
INSERT INTO `Order Details` Values (10774, 31, 12.5, 2, 0.25);
INSERT INTO `Order Details` Values (10774, 66, 17, 50, 0.0);
INSERT INTO `Order Details` Values (10775, 10, 31, 6, 0.0);
INSERT INTO `Order Details` Values (10775, 67, 14, 3, 0.0);
INSERT INTO `Order Details` Values (10776, 31, 12.5, 16, 0.05);
INSERT INTO `Order Details` Values (10776, 42, 14, 12, 0.05);
INSERT INTO `Order Details` Values (10776, 45, 9.5, 27, 0.05);
INSERT INTO `Order Details` Values (10776, 51, 53, 120, 0.05);
INSERT INTO `Order Details` Values (10777, 42, 14, 20, 0.2);
INSERT INTO `Order Details` Values (10778, 41, 9.65, 10, 0.0);
INSERT INTO `Order Details` Values (10779, 16, 17.45, 20, 0.0);
INSERT INTO `Order Details` Values (10779, 62, 49.3, 20, 0.0);
INSERT INTO `Order Details` Values (10780, 70, 15, 35, 0.0);
INSERT INTO `Order Details` Values (10780, 77, 13, 15, 0.0);
INSERT INTO `Order Details` Values (10781, 54, 7.45, 3, 0.2);
INSERT INTO `Order Details` Values (10781, 56, 38, 20, 0.2);
INSERT INTO `Order Details` Values (10781, 74, 10, 35, 0.0);
INSERT INTO `Order Details` Values (10782, 31, 12.5, 1, 0.0);
INSERT INTO `Order Details` Values (10783, 31, 12.5, 10, 0.0);
INSERT INTO `Order Details` Values (10783, 38, 263.5, 5, 0.0);
INSERT INTO `Order Details` Values (10784, 36, 19, 30, 0.0);
INSERT INTO `Order Details` Values (10784, 39, 18, 2, 0.15);
INSERT INTO `Order Details` Values (10784, 72, 34.8, 30, 0.15);
INSERT INTO `Order Details` Values (10785, 10, 31, 10, 0.0);
INSERT INTO `Order Details` Values (10785, 75, 7.75, 10, 0.0);
INSERT INTO `Order Details` Values (10786, 8, 40, 30, 0.2);
INSERT INTO `Order Details` Values (10786, 30, 25.89, 15, 0.2);
INSERT INTO `Order Details` Values (10786, 75, 7.75, 42, 0.2);
INSERT INTO `Order Details` Values (10787, 2, 19, 15, 0.05);
INSERT INTO `Order Details` Values (10787, 29, 123.79, 20, 0.05);
INSERT INTO `Order Details` Values (10788, 19, 9.2, 50, 0.05);
INSERT INTO `Order Details` Values (10788, 75, 7.75, 40, 0.05);
INSERT INTO `Order Details` Values (10789, 18, 62.5, 30, 0.0);
INSERT INTO `Order Details` Values (10789, 35, 18, 15, 0.0);
INSERT INTO `Order Details` Values (10789, 63, 43.9, 30, 0.0);
INSERT INTO `Order Details` Values (10789, 68, 12.5, 18, 0.0);
INSERT INTO `Order Details` Values (10790, 7, 30, 3, 0.15);
INSERT INTO `Order Details` Values (10790, 56, 38, 20, 0.15);
INSERT INTO `Order Details` Values (10791, 29, 123.79, 14, 0.05);
INSERT INTO `Order Details` Values (10791, 41, 9.65, 20, 0.05);
INSERT INTO `Order Details` Values (10792, 2, 19, 10, 0.0);
INSERT INTO `Order Details` Values (10792, 54, 7.45, 3, 0.0);
INSERT INTO `Order Details` Values (10792, 68, 12.5, 15, 0.0);
INSERT INTO `Order Details` Values (10793, 41, 9.65, 14, 0.0);
INSERT INTO `Order Details` Values (10793, 52, 7, 8, 0.0);
INSERT INTO `Order Details` Values (10794, 14, 23.25, 15, 0.2);
INSERT INTO `Order Details` Values (10794, 54, 7.45, 6, 0.2);
INSERT INTO `Order Details` Values (10795, 16, 17.45, 65, 0.0);
INSERT INTO `Order Details` Values (10795, 17, 39, 35, 0.25);
INSERT INTO `Order Details` Values (10796, 26, 31.23, 21, 0.2);
INSERT INTO `Order Details` Values (10796, 44, 19.45, 10, 0.0);
INSERT INTO `Order Details` Values (10796, 64, 33.25, 35, 0.2);
INSERT INTO `Order Details` Values (10796, 69, 36, 24, 0.2);
INSERT INTO `Order Details` Values (10797, 11, 21, 20, 0.0);
INSERT INTO `Order Details` Values (10798, 62, 49.3, 2, 0.0);
INSERT INTO `Order Details` Values (10798, 72, 34.8, 10, 0.0);
INSERT INTO `Order Details` Values (10799, 13, 6, 20, 0.15);
INSERT INTO `Order Details` Values (10799, 24, 4.5, 20, 0.15);
INSERT INTO `Order Details` Values (10799, 59, 55, 25, 0.0);
INSERT INTO `Order Details` Values (10800, 11, 21, 50, 0.1);
INSERT INTO `Order Details` Values (10800, 51, 53, 10, 0.1);
INSERT INTO `Order Details` Values (10800, 54, 7.45, 7, 0.1);
INSERT INTO `Order Details` Values (10801, 17, 39, 40, 0.25);
INSERT INTO `Order Details` Values (10801, 29, 123.79, 20, 0.25);
INSERT INTO `Order Details` Values (10802, 30, 25.89, 25, 0.25);
INSERT INTO `Order Details` Values (10802, 51, 53, 30, 0.25);
INSERT INTO `Order Details` Values (10802, 55, 24, 60, 0.25);
INSERT INTO `Order Details` Values (10802, 62, 49.3, 5, 0.25);
INSERT INTO `Order Details` Values (10803, 19, 9.2, 24, 0.05);
INSERT INTO `Order Details` Values (10803, 25, 14, 15, 0.05);
INSERT INTO `Order Details` Values (10803, 59, 55, 15, 0.05);
INSERT INTO `Order Details` Values (10804, 10, 31, 36, 0.0);
INSERT INTO `Order Details` Values (10804, 28, 45.6, 24, 0.0);
INSERT INTO `Order Details` Values (10804, 49, 20, 4, 0.15);
INSERT INTO `Order Details` Values (10805, 34, 14, 10, 0.0);
INSERT INTO `Order Details` Values (10805, 38, 263.5, 10, 0.0);
INSERT INTO `Order Details` Values (10806, 2, 19, 20, 0.25);
INSERT INTO `Order Details` Values (10806, 65, 21.05, 2, 0.0);
INSERT INTO `Order Details` Values (10806, 74, 10, 15, 0.25);
INSERT INTO `Order Details` Values (10807, 40, 18.4, 1, 0.0);
INSERT INTO `Order Details` Values (10808, 56, 38, 20, 0.15);
INSERT INTO `Order Details` Values (10808, 76, 18, 50, 0.15);
INSERT INTO `Order Details` Values (10809, 52, 7, 20, 0.0);
INSERT INTO `Order Details` Values (10810, 13, 6, 7, 0.0);
INSERT INTO `Order Details` Values (10810, 25, 14, 5, 0.0);
INSERT INTO `Order Details` Values (10810, 70, 15, 5, 0.0);
INSERT INTO `Order Details` Values (10811, 19, 9.2, 15, 0.0);
INSERT INTO `Order Details` Values (10811, 23, 9, 18, 0.0);
INSERT INTO `Order Details` Values (10811, 40, 18.4, 30, 0.0);
INSERT INTO `Order Details` Values (10812, 31, 12.5, 16, 0.1);
INSERT INTO `Order Details` Values (10812, 72, 34.8, 40, 0.1);
INSERT INTO `Order Details` Values (10812, 77, 13, 20, 0.0);
INSERT INTO `Order Details` Values (10813, 2, 19, 12, 0.2);
INSERT INTO `Order Details` Values (10813, 46, 12, 35, 0.0);
INSERT INTO `Order Details` Values (10814, 41, 9.65, 20, 0.0);
INSERT INTO `Order Details` Values (10814, 43, 46, 20, 0.15);
INSERT INTO `Order Details` Values (10814, 48, 12.75, 8, 0.15);
INSERT INTO `Order Details` Values (10814, 61, 28.5, 30, 0.15);
INSERT INTO `Order Details` Values (10815, 33, 2.5, 16, 0.0);
INSERT INTO `Order Details` Values (10816, 38, 263.5, 30, 0.05);
INSERT INTO `Order Details` Values (10816, 62, 49.3, 20, 0.05);
INSERT INTO `Order Details` Values (10817, 26, 31.23, 40, 0.15);
INSERT INTO `Order Details` Values (10817, 38, 263.5, 30, 0.0);
INSERT INTO `Order Details` Values (10817, 40, 18.4, 60, 0.15);
INSERT INTO `Order Details` Values (10817, 62, 49.3, 25, 0.15);
INSERT INTO `Order Details` Values (10818, 32, 32, 20, 0.0);
INSERT INTO `Order Details` Values (10818, 41, 9.65, 20, 0.0);
INSERT INTO `Order Details` Values (10819, 43, 46, 7, 0.0);
INSERT INTO `Order Details` Values (10819, 75, 7.75, 20, 0.0);
INSERT INTO `Order Details` Values (10820, 56, 38, 30, 0.0);
INSERT INTO `Order Details` Values (10821, 35, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10821, 51, 53, 6, 0.0);
INSERT INTO `Order Details` Values (10822, 62, 49.3, 3, 0.0);
INSERT INTO `Order Details` Values (10822, 70, 15, 6, 0.0);
INSERT INTO `Order Details` Values (10823, 11, 21, 20, 0.1);
INSERT INTO `Order Details` Values (10823, 57, 19.5, 15, 0.0);
INSERT INTO `Order Details` Values (10823, 59, 55, 40, 0.1);
INSERT INTO `Order Details` Values (10823, 77, 13, 15, 0.1);
INSERT INTO `Order Details` Values (10824, 41, 9.65, 12, 0.0);
INSERT INTO `Order Details` Values (10824, 70, 15, 9, 0.0);
INSERT INTO `Order Details` Values (10825, 26, 31.23, 12, 0.0);
INSERT INTO `Order Details` Values (10825, 53, 32.8, 20, 0.0);
INSERT INTO `Order Details` Values (10826, 31, 12.5, 35, 0.0);
INSERT INTO `Order Details` Values (10826, 57, 19.5, 15, 0.0);
INSERT INTO `Order Details` Values (10827, 10, 31, 15, 0.0);
INSERT INTO `Order Details` Values (10827, 39, 18, 21, 0.0);
INSERT INTO `Order Details` Values (10828, 20, 81, 5, 0.0);
INSERT INTO `Order Details` Values (10828, 38, 263.5, 2, 0.0);
INSERT INTO `Order Details` Values (10829, 2, 19, 10, 0.0);
INSERT INTO `Order Details` Values (10829, 8, 40, 20, 0.0);
INSERT INTO `Order Details` Values (10829, 13, 6, 10, 0.0);
INSERT INTO `Order Details` Values (10829, 60, 34, 21, 0.0);
INSERT INTO `Order Details` Values (10830, 6, 25, 6, 0.0);
INSERT INTO `Order Details` Values (10830, 39, 18, 28, 0.0);
INSERT INTO `Order Details` Values (10830, 60, 34, 30, 0.0);
INSERT INTO `Order Details` Values (10830, 68, 12.5, 24, 0.0);
INSERT INTO `Order Details` Values (10831, 19, 9.2, 2, 0.0);
INSERT INTO `Order Details` Values (10831, 35, 18, 8, 0.0);
INSERT INTO `Order Details` Values (10831, 38, 263.5, 8, 0.0);
INSERT INTO `Order Details` Values (10831, 43, 46, 9, 0.0);
INSERT INTO `Order Details` Values (10832, 13, 6, 3, 0.2);
INSERT INTO `Order Details` Values (10832, 25, 14, 10, 0.2);
INSERT INTO `Order Details` Values (10832, 44, 19.45, 16, 0.2);
INSERT INTO `Order Details` Values (10832, 64, 33.25, 3, 0.0);
INSERT INTO `Order Details` Values (10833, 7, 30, 20, 0.1);
INSERT INTO `Order Details` Values (10833, 31, 12.5, 9, 0.1);
INSERT INTO `Order Details` Values (10833, 53, 32.8, 9, 0.1);
INSERT INTO `Order Details` Values (10834, 29, 123.79, 8, 0.05);
INSERT INTO `Order Details` Values (10834, 30, 25.89, 20, 0.05);
INSERT INTO `Order Details` Values (10835, 59, 55, 15, 0.0);
INSERT INTO `Order Details` Values (10835, 77, 13, 2, 0.2);
INSERT INTO `Order Details` Values (10836, 22, 21, 52, 0.0);
INSERT INTO `Order Details` Values (10836, 35, 18, 6, 0.0);
INSERT INTO `Order Details` Values (10836, 57, 19.5, 24, 0.0);
INSERT INTO `Order Details` Values (10836, 60, 34, 60, 0.0);
INSERT INTO `Order Details` Values (10836, 64, 33.25, 30, 0.0);
INSERT INTO `Order Details` Values (10837, 13, 6, 6, 0.0);
INSERT INTO `Order Details` Values (10837, 40, 18.4, 25, 0.0);
INSERT INTO `Order Details` Values (10837, 47, 9.5, 40, 0.25);
INSERT INTO `Order Details` Values (10837, 76, 18, 21, 0.25);
INSERT INTO `Order Details` Values (10838, 1, 18, 4, 0.25);
INSERT INTO `Order Details` Values (10838, 18, 62.5, 25, 0.25);
INSERT INTO `Order Details` Values (10838, 36, 19, 50, 0.25);
INSERT INTO `Order Details` Values (10839, 58, 13.25, 30, 0.1);
INSERT INTO `Order Details` Values (10839, 72, 34.8, 15, 0.1);
INSERT INTO `Order Details` Values (10840, 25, 14, 6, 0.2);
INSERT INTO `Order Details` Values (10840, 39, 18, 10, 0.2);
INSERT INTO `Order Details` Values (10841, 10, 31, 16, 0.0);
INSERT INTO `Order Details` Values (10841, 56, 38, 30, 0.0);
INSERT INTO `Order Details` Values (10841, 59, 55, 50, 0.0);
INSERT INTO `Order Details` Values (10841, 77, 13, 15, 0.0);
INSERT INTO `Order Details` Values (10842, 11, 21, 15, 0.0);
INSERT INTO `Order Details` Values (10842, 43, 46, 5, 0.0);
INSERT INTO `Order Details` Values (10842, 68, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10842, 70, 15, 12, 0.0);
INSERT INTO `Order Details` Values (10843, 51, 53, 4, 0.25);
INSERT INTO `Order Details` Values (10844, 22, 21, 35, 0.0);
INSERT INTO `Order Details` Values (10845, 23, 9, 70, 0.1);
INSERT INTO `Order Details` Values (10845, 35, 18, 25, 0.1);
INSERT INTO `Order Details` Values (10845, 42, 14, 42, 0.1);
INSERT INTO `Order Details` Values (10845, 58, 13.25, 60, 0.1);
INSERT INTO `Order Details` Values (10845, 64, 33.25, 48, 0.0);
INSERT INTO `Order Details` Values (10846, 4, 22, 21, 0.0);
INSERT INTO `Order Details` Values (10846, 70, 15, 30, 0.0);
INSERT INTO `Order Details` Values (10846, 74, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10847, 1, 18, 80, 0.2);
INSERT INTO `Order Details` Values (10847, 19, 9.2, 12, 0.2);
INSERT INTO `Order Details` Values (10847, 37, 26, 60, 0.2);
INSERT INTO `Order Details` Values (10847, 45, 9.5, 36, 0.2);
INSERT INTO `Order Details` Values (10847, 60, 34, 45, 0.2);
INSERT INTO `Order Details` Values (10847, 71, 21.5, 55, 0.2);
INSERT INTO `Order Details` Values (10848, 5, 21.35, 30, 0.0);
INSERT INTO `Order Details` Values (10848, 9, 97, 3, 0.0);
INSERT INTO `Order Details` Values (10849, 3, 10, 49, 0.0);
INSERT INTO `Order Details` Values (10849, 26, 31.23, 18, 0.15);
INSERT INTO `Order Details` Values (10850, 25, 14, 20, 0.15);
INSERT INTO `Order Details` Values (10850, 33, 2.5, 4, 0.15);
INSERT INTO `Order Details` Values (10850, 70, 15, 30, 0.15);
INSERT INTO `Order Details` Values (10851, 2, 19, 5, 0.05);
INSERT INTO `Order Details` Values (10851, 25, 14, 10, 0.05);
INSERT INTO `Order Details` Values (10851, 57, 19.5, 10, 0.05);
INSERT INTO `Order Details` Values (10851, 59, 55, 42, 0.05);
INSERT INTO `Order Details` Values (10852, 2, 19, 15, 0.0);
INSERT INTO `Order Details` Values (10852, 17, 39, 6, 0.0);
INSERT INTO `Order Details` Values (10852, 62, 49.3, 50, 0.0);
INSERT INTO `Order Details` Values (10853, 18, 62.5, 10, 0.0);
INSERT INTO `Order Details` Values (10854, 10, 31, 100, 0.15);
INSERT INTO `Order Details` Values (10854, 13, 6, 65, 0.15);
INSERT INTO `Order Details` Values (10855, 16, 17.45, 50, 0.0);
INSERT INTO `Order Details` Values (10855, 31, 12.5, 14, 0.0);
INSERT INTO `Order Details` Values (10855, 56, 38, 24, 0.0);
INSERT INTO `Order Details` Values (10855, 65, 21.05, 15, 0.15);
INSERT INTO `Order Details` Values (10856, 2, 19, 20, 0.0);
INSERT INTO `Order Details` Values (10856, 42, 14, 20, 0.0);
INSERT INTO `Order Details` Values (10857, 3, 10, 30, 0.0);
INSERT INTO `Order Details` Values (10857, 26, 31.23, 35, 0.25);
INSERT INTO `Order Details` Values (10857, 29, 123.79, 10, 0.25);
INSERT INTO `Order Details` Values (10858, 7, 30, 5, 0.0);
INSERT INTO `Order Details` Values (10858, 27, 43.9, 10, 0.0);
INSERT INTO `Order Details` Values (10858, 70, 15, 4, 0.0);
INSERT INTO `Order Details` Values (10859, 24, 4.5, 40, 0.25);
INSERT INTO `Order Details` Values (10859, 54, 7.45, 35, 0.25);
INSERT INTO `Order Details` Values (10859, 64, 33.25, 30, 0.25);
INSERT INTO `Order Details` Values (10860, 51, 53, 3, 0.0);
INSERT INTO `Order Details` Values (10860, 76, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10861, 17, 39, 42, 0.0);
INSERT INTO `Order Details` Values (10861, 18, 62.5, 20, 0.0);
INSERT INTO `Order Details` Values (10861, 21, 10, 40, 0.0);
INSERT INTO `Order Details` Values (10861, 33, 2.5, 35, 0.0);
INSERT INTO `Order Details` Values (10861, 62, 49.3, 3, 0.0);
INSERT INTO `Order Details` Values (10862, 11, 21, 25, 0.0);
INSERT INTO `Order Details` Values (10862, 52, 7, 8, 0.0);
INSERT INTO `Order Details` Values (10863, 1, 18, 20, 0.15);
INSERT INTO `Order Details` Values (10863, 58, 13.25, 12, 0.15);
INSERT INTO `Order Details` Values (10864, 35, 18, 4, 0.0);
INSERT INTO `Order Details` Values (10864, 67, 14, 15, 0.0);
INSERT INTO `Order Details` Values (10865, 38, 263.5, 60, 0.05);
INSERT INTO `Order Details` Values (10865, 39, 18, 80, 0.05);
INSERT INTO `Order Details` Values (10866, 2, 19, 21, 0.25);
INSERT INTO `Order Details` Values (10866, 24, 4.5, 6, 0.25);
INSERT INTO `Order Details` Values (10866, 30, 25.89, 40, 0.25);
INSERT INTO `Order Details` Values (10867, 53, 32.8, 3, 0.0);
INSERT INTO `Order Details` Values (10868, 26, 31.23, 20, 0.0);
INSERT INTO `Order Details` Values (10868, 35, 18, 30, 0.0);
INSERT INTO `Order Details` Values (10868, 49, 20, 42, 0.1);
INSERT INTO `Order Details` Values (10869, 1, 18, 40, 0.0);
INSERT INTO `Order Details` Values (10869, 11, 21, 10, 0.0);
INSERT INTO `Order Details` Values (10869, 23, 9, 50, 0.0);
INSERT INTO `Order Details` Values (10869, 68, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10870, 35, 18, 3, 0.0);
INSERT INTO `Order Details` Values (10870, 51, 53, 2, 0.0);
INSERT INTO `Order Details` Values (10871, 6, 25, 50, 0.05);
INSERT INTO `Order Details` Values (10871, 16, 17.45, 12, 0.05);
INSERT INTO `Order Details` Values (10871, 17, 39, 16, 0.05);
INSERT INTO `Order Details` Values (10872, 55, 24, 10, 0.05);
INSERT INTO `Order Details` Values (10872, 62, 49.3, 20, 0.05);
INSERT INTO `Order Details` Values (10872, 64, 33.25, 15, 0.05);
INSERT INTO `Order Details` Values (10872, 65, 21.05, 21, 0.05);
INSERT INTO `Order Details` Values (10873, 21, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10873, 28, 45.6, 3, 0.0);
INSERT INTO `Order Details` Values (10874, 10, 31, 10, 0.0);
INSERT INTO `Order Details` Values (10875, 19, 9.2, 25, 0.0);
INSERT INTO `Order Details` Values (10875, 47, 9.5, 21, 0.1);
INSERT INTO `Order Details` Values (10875, 49, 20, 15, 0.0);
INSERT INTO `Order Details` Values (10876, 46, 12, 21, 0.0);
INSERT INTO `Order Details` Values (10876, 64, 33.25, 20, 0.0);
INSERT INTO `Order Details` Values (10877, 16, 17.45, 30, 0.25);
INSERT INTO `Order Details` Values (10877, 18, 62.5, 25, 0.0);
INSERT INTO `Order Details` Values (10878, 20, 81, 20, 0.05);
INSERT INTO `Order Details` Values (10879, 40, 18.4, 12, 0.0);
INSERT INTO `Order Details` Values (10879, 65, 21.05, 10, 0.0);
INSERT INTO `Order Details` Values (10879, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10880, 23, 9, 30, 0.2);
INSERT INTO `Order Details` Values (10880, 61, 28.5, 30, 0.2);
INSERT INTO `Order Details` Values (10880, 70, 15, 50, 0.2);
INSERT INTO `Order Details` Values (10881, 73, 15, 10, 0.0);
INSERT INTO `Order Details` Values (10882, 42, 14, 25, 0.0);
INSERT INTO `Order Details` Values (10882, 49, 20, 20, 0.15);
INSERT INTO `Order Details` Values (10882, 54, 7.45, 32, 0.15);
INSERT INTO `Order Details` Values (10883, 24, 4.5, 8, 0.0);
INSERT INTO `Order Details` Values (10884, 21, 10, 40, 0.05);
INSERT INTO `Order Details` Values (10884, 56, 38, 21, 0.05);
INSERT INTO `Order Details` Values (10884, 65, 21.05, 12, 0.05);
INSERT INTO `Order Details` Values (10885, 2, 19, 20, 0.0);
INSERT INTO `Order Details` Values (10885, 24, 4.5, 12, 0.0);
INSERT INTO `Order Details` Values (10885, 70, 15, 30, 0.0);
INSERT INTO `Order Details` Values (10885, 77, 13, 25, 0.0);
INSERT INTO `Order Details` Values (10886, 10, 31, 70, 0.0);
INSERT INTO `Order Details` Values (10886, 31, 12.5, 35, 0.0);
INSERT INTO `Order Details` Values (10886, 77, 13, 40, 0.0);
INSERT INTO `Order Details` Values (10887, 25, 14, 5, 0.0);
INSERT INTO `Order Details` Values (10888, 2, 19, 20, 0.0);
INSERT INTO `Order Details` Values (10888, 68, 12.5, 18, 0.0);
INSERT INTO `Order Details` Values (10889, 11, 21, 40, 0.0);
INSERT INTO `Order Details` Values (10889, 38, 263.5, 40, 0.0);
INSERT INTO `Order Details` Values (10890, 17, 39, 15, 0.0);
INSERT INTO `Order Details` Values (10890, 34, 14, 10, 0.0);
INSERT INTO `Order Details` Values (10890, 41, 9.65, 14, 0.0);
INSERT INTO `Order Details` Values (10891, 30, 25.89, 15, 0.05);
INSERT INTO `Order Details` Values (10892, 59, 55, 40, 0.05);
INSERT INTO `Order Details` Values (10893, 8, 40, 30, 0.0);
INSERT INTO `Order Details` Values (10893, 24, 4.5, 10, 0.0);
INSERT INTO `Order Details` Values (10893, 29, 123.79, 24, 0.0);
INSERT INTO `Order Details` Values (10893, 30, 25.89, 35, 0.0);
INSERT INTO `Order Details` Values (10893, 36, 19, 20, 0.0);
INSERT INTO `Order Details` Values (10894, 13, 6, 28, 0.05);
INSERT INTO `Order Details` Values (10894, 69, 36, 50, 0.05);
INSERT INTO `Order Details` Values (10894, 75, 7.75, 120, 0.05);
INSERT INTO `Order Details` Values (10895, 24, 4.5, 110, 0.0);
INSERT INTO `Order Details` Values (10895, 39, 18, 45, 0.0);
INSERT INTO `Order Details` Values (10895, 40, 18.4, 91, 0.0);
INSERT INTO `Order Details` Values (10895, 60, 34, 100, 0.0);
INSERT INTO `Order Details` Values (10896, 45, 9.5, 15, 0.0);
INSERT INTO `Order Details` Values (10896, 56, 38, 16, 0.0);
INSERT INTO `Order Details` Values (10897, 29, 123.79, 80, 0.0);
INSERT INTO `Order Details` Values (10897, 30, 25.89, 36, 0.0);
INSERT INTO `Order Details` Values (10898, 13, 6, 5, 0.0);
INSERT INTO `Order Details` Values (10899, 39, 18, 8, 0.15);
INSERT INTO `Order Details` Values (10900, 70, 15, 3, 0.25);
INSERT INTO `Order Details` Values (10901, 41, 9.65, 30, 0.0);
INSERT INTO `Order Details` Values (10901, 71, 21.5, 30, 0.0);
INSERT INTO `Order Details` Values (10902, 55, 24, 30, 0.15);
INSERT INTO `Order Details` Values (10902, 62, 49.3, 6, 0.15);
INSERT INTO `Order Details` Values (10903, 13, 6, 40, 0.0);
INSERT INTO `Order Details` Values (10903, 65, 21.05, 21, 0.0);
INSERT INTO `Order Details` Values (10903, 68, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (10904, 58, 13.25, 15, 0.0);
INSERT INTO `Order Details` Values (10904, 62, 49.3, 35, 0.0);
INSERT INTO `Order Details` Values (10905, 1, 18, 20, 0.05);
INSERT INTO `Order Details` Values (10906, 61, 28.5, 15, 0.0);
INSERT INTO `Order Details` Values (10907, 75, 7.75, 14, 0.0);
INSERT INTO `Order Details` Values (10908, 7, 30, 20, 0.05);
INSERT INTO `Order Details` Values (10908, 52, 7, 14, 0.05);
INSERT INTO `Order Details` Values (10909, 7, 30, 12, 0.0);
INSERT INTO `Order Details` Values (10909, 16, 17.45, 15, 0.0);
INSERT INTO `Order Details` Values (10909, 41, 9.65, 5, 0.0);
INSERT INTO `Order Details` Values (10910, 19, 9.2, 12, 0.0);
INSERT INTO `Order Details` Values (10910, 49, 20, 10, 0.0);
INSERT INTO `Order Details` Values (10910, 61, 28.5, 5, 0.0);
INSERT INTO `Order Details` Values (10911, 1, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10911, 17, 39, 12, 0.0);
INSERT INTO `Order Details` Values (10911, 67, 14, 15, 0.0);
INSERT INTO `Order Details` Values (10912, 11, 21, 40, 0.25);
INSERT INTO `Order Details` Values (10912, 29, 123.79, 60, 0.25);
INSERT INTO `Order Details` Values (10913, 4, 22, 30, 0.25);
INSERT INTO `Order Details` Values (10913, 33, 2.5, 40, 0.25);
INSERT INTO `Order Details` Values (10913, 58, 13.25, 15, 0.0);
INSERT INTO `Order Details` Values (10914, 71, 21.5, 25, 0.0);
INSERT INTO `Order Details` Values (10915, 17, 39, 10, 0.0);
INSERT INTO `Order Details` Values (10915, 33, 2.5, 30, 0.0);
INSERT INTO `Order Details` Values (10915, 54, 7.45, 10, 0.0);
INSERT INTO `Order Details` Values (10916, 16, 17.45, 6, 0.0);
INSERT INTO `Order Details` Values (10916, 32, 32, 6, 0.0);
INSERT INTO `Order Details` Values (10916, 57, 19.5, 20, 0.0);
INSERT INTO `Order Details` Values (10917, 30, 25.89, 1, 0.0);
INSERT INTO `Order Details` Values (10917, 60, 34, 10, 0.0);
INSERT INTO `Order Details` Values (10918, 1, 18, 60, 0.25);
INSERT INTO `Order Details` Values (10918, 60, 34, 25, 0.25);
INSERT INTO `Order Details` Values (10919, 16, 17.45, 24, 0.0);
INSERT INTO `Order Details` Values (10919, 25, 14, 24, 0.0);
INSERT INTO `Order Details` Values (10919, 40, 18.4, 20, 0.0);
INSERT INTO `Order Details` Values (10920, 50, 16.25, 24, 0.0);
INSERT INTO `Order Details` Values (10921, 35, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10921, 63, 43.9, 40, 0.0);
INSERT INTO `Order Details` Values (10922, 17, 39, 15, 0.0);
INSERT INTO `Order Details` Values (10922, 24, 4.5, 35, 0.0);
INSERT INTO `Order Details` Values (10923, 42, 14, 10, 0.2);
INSERT INTO `Order Details` Values (10923, 43, 46, 10, 0.2);
INSERT INTO `Order Details` Values (10923, 67, 14, 24, 0.2);
INSERT INTO `Order Details` Values (10924, 10, 31, 20, 0.1);
INSERT INTO `Order Details` Values (10924, 28, 45.6, 30, 0.1);
INSERT INTO `Order Details` Values (10924, 75, 7.75, 6, 0.0);
INSERT INTO `Order Details` Values (10925, 36, 19, 25, 0.15);
INSERT INTO `Order Details` Values (10925, 52, 7, 12, 0.15);
INSERT INTO `Order Details` Values (10926, 11, 21, 2, 0.0);
INSERT INTO `Order Details` Values (10926, 13, 6, 10, 0.0);
INSERT INTO `Order Details` Values (10926, 19, 9.2, 7, 0.0);
INSERT INTO `Order Details` Values (10926, 72, 34.8, 10, 0.0);
INSERT INTO `Order Details` Values (10927, 20, 81, 5, 0.0);
INSERT INTO `Order Details` Values (10927, 52, 7, 5, 0.0);
INSERT INTO `Order Details` Values (10927, 76, 18, 20, 0.0);
INSERT INTO `Order Details` Values (10928, 47, 9.5, 5, 0.0);
INSERT INTO `Order Details` Values (10928, 76, 18, 5, 0.0);
INSERT INTO `Order Details` Values (10929, 21, 10, 60, 0.0);
INSERT INTO `Order Details` Values (10929, 75, 7.75, 49, 0.0);
INSERT INTO `Order Details` Values (10929, 77, 13, 15, 0.0);
INSERT INTO `Order Details` Values (10930, 21, 10, 36, 0.0);
INSERT INTO `Order Details` Values (10930, 27, 43.9, 25, 0.0);
INSERT INTO `Order Details` Values (10930, 55, 24, 25, 0.2);
INSERT INTO `Order Details` Values (10930, 58, 13.25, 30, 0.2);
INSERT INTO `Order Details` Values (10931, 13, 6, 42, 0.15);
INSERT INTO `Order Details` Values (10931, 57, 19.5, 30, 0.0);
INSERT INTO `Order Details` Values (10932, 16, 17.45, 30, 0.1);
INSERT INTO `Order Details` Values (10932, 62, 49.3, 14, 0.1);
INSERT INTO `Order Details` Values (10932, 72, 34.8, 16, 0.0);
INSERT INTO `Order Details` Values (10932, 75, 7.75, 20, 0.1);
INSERT INTO `Order Details` Values (10933, 53, 32.8, 2, 0.0);
INSERT INTO `Order Details` Values (10933, 61, 28.5, 30, 0.0);
INSERT INTO `Order Details` Values (10934, 6, 25, 20, 0.0);
INSERT INTO `Order Details` Values (10935, 1, 18, 21, 0.0);
INSERT INTO `Order Details` Values (10935, 18, 62.5, 4, 0.25);
INSERT INTO `Order Details` Values (10935, 23, 9, 8, 0.25);
INSERT INTO `Order Details` Values (10936, 36, 19, 30, 0.2);
INSERT INTO `Order Details` Values (10937, 28, 45.6, 8, 0.0);
INSERT INTO `Order Details` Values (10937, 34, 14, 20, 0.0);
INSERT INTO `Order Details` Values (10938, 13, 6, 20, 0.25);
INSERT INTO `Order Details` Values (10938, 43, 46, 24, 0.25);
INSERT INTO `Order Details` Values (10938, 60, 34, 49, 0.25);
INSERT INTO `Order Details` Values (10938, 71, 21.5, 35, 0.25);
INSERT INTO `Order Details` Values (10939, 2, 19, 10, 0.15);
INSERT INTO `Order Details` Values (10939, 67, 14, 40, 0.15);
INSERT INTO `Order Details` Values (10940, 7, 30, 8, 0.0);
INSERT INTO `Order Details` Values (10940, 13, 6, 20, 0.0);
INSERT INTO `Order Details` Values (10941, 31, 12.5, 44, 0.25);
INSERT INTO `Order Details` Values (10941, 62, 49.3, 30, 0.25);
INSERT INTO `Order Details` Values (10941, 68, 12.5, 80, 0.25);
INSERT INTO `Order Details` Values (10941, 72, 34.8, 50, 0.0);
INSERT INTO `Order Details` Values (10942, 49, 20, 28, 0.0);
INSERT INTO `Order Details` Values (10943, 13, 6, 15, 0.0);
INSERT INTO `Order Details` Values (10943, 22, 21, 21, 0.0);
INSERT INTO `Order Details` Values (10943, 46, 12, 15, 0.0);
INSERT INTO `Order Details` Values (10944, 11, 21, 5, 0.25);
INSERT INTO `Order Details` Values (10944, 44, 19.45, 18, 0.25);
INSERT INTO `Order Details` Values (10944, 56, 38, 18, 0.0);
INSERT INTO `Order Details` Values (10945, 13, 6, 20, 0.0);
INSERT INTO `Order Details` Values (10945, 31, 12.5, 10, 0.0);
INSERT INTO `Order Details` Values (10946, 10, 31, 25, 0.0);
INSERT INTO `Order Details` Values (10946, 24, 4.5, 25, 0.0);
INSERT INTO `Order Details` Values (10946, 77, 13, 40, 0.0);
INSERT INTO `Order Details` Values (10947, 59, 55, 4, 0.0);
INSERT INTO `Order Details` Values (10948, 50, 16.25, 9, 0.0);
INSERT INTO `Order Details` Values (10948, 51, 53, 40, 0.0);
INSERT INTO `Order Details` Values (10948, 55, 24, 4, 0.0);
INSERT INTO `Order Details` Values (10949, 6, 25, 12, 0.0);
INSERT INTO `Order Details` Values (10949, 10, 31, 30, 0.0);
INSERT INTO `Order Details` Values (10949, 17, 39, 6, 0.0);
INSERT INTO `Order Details` Values (10949, 62, 49.3, 60, 0.0);
INSERT INTO `Order Details` Values (10950, 4, 22, 5, 0.0);
INSERT INTO `Order Details` Values (10951, 33, 2.5, 15, 0.05);
INSERT INTO `Order Details` Values (10951, 41, 9.65, 6, 0.05);
INSERT INTO `Order Details` Values (10951, 75, 7.75, 50, 0.05);
INSERT INTO `Order Details` Values (10952, 6, 25, 16, 0.05);
INSERT INTO `Order Details` Values (10952, 28, 45.6, 2, 0.0);
INSERT INTO `Order Details` Values (10953, 20, 81, 50, 0.05);
INSERT INTO `Order Details` Values (10953, 31, 12.5, 50, 0.05);
INSERT INTO `Order Details` Values (10954, 16, 17.45, 28, 0.15);
INSERT INTO `Order Details` Values (10954, 31, 12.5, 25, 0.15);
INSERT INTO `Order Details` Values (10954, 45, 9.5, 30, 0.0);
INSERT INTO `Order Details` Values (10954, 60, 34, 24, 0.15);
INSERT INTO `Order Details` Values (10955, 75, 7.75, 12, 0.2);
INSERT INTO `Order Details` Values (10956, 21, 10, 12, 0.0);
INSERT INTO `Order Details` Values (10956, 47, 9.5, 14, 0.0);
INSERT INTO `Order Details` Values (10956, 51, 53, 8, 0.0);
INSERT INTO `Order Details` Values (10957, 30, 25.89, 30, 0.0);
INSERT INTO `Order Details` Values (10957, 35, 18, 40, 0.0);
INSERT INTO `Order Details` Values (10957, 64, 33.25, 8, 0.0);
INSERT INTO `Order Details` Values (10958, 5, 21.35, 20, 0.0);
INSERT INTO `Order Details` Values (10958, 7, 30, 6, 0.0);
INSERT INTO `Order Details` Values (10958, 72, 34.8, 5, 0.0);
INSERT INTO `Order Details` Values (10959, 75, 7.75, 20, 0.15);
INSERT INTO `Order Details` Values (10960, 24, 4.5, 10, 0.25);
INSERT INTO `Order Details` Values (10960, 41, 9.65, 24, 0.0);
INSERT INTO `Order Details` Values (10961, 52, 7, 6, 0.05);
INSERT INTO `Order Details` Values (10961, 76, 18, 60, 0.0);
INSERT INTO `Order Details` Values (10962, 7, 30, 45, 0.0);
INSERT INTO `Order Details` Values (10962, 13, 6, 77, 0.0);
INSERT INTO `Order Details` Values (10962, 53, 32.8, 20, 0.0);
INSERT INTO `Order Details` Values (10962, 69, 36, 9, 0.0);
INSERT INTO `Order Details` Values (10962, 76, 18, 44, 0.0);
INSERT INTO `Order Details` Values (10963, 60, 34, 2, 0.15);
INSERT INTO `Order Details` Values (10964, 18, 62.5, 6, 0.0);
INSERT INTO `Order Details` Values (10964, 38, 263.5, 5, 0.0);
INSERT INTO `Order Details` Values (10964, 69, 36, 10, 0.0);
INSERT INTO `Order Details` Values (10965, 51, 53, 16, 0.0);
INSERT INTO `Order Details` Values (10966, 37, 26, 8, 0.0);
INSERT INTO `Order Details` Values (10966, 56, 38, 12, 0.15);
INSERT INTO `Order Details` Values (10966, 62, 49.3, 12, 0.15);
INSERT INTO `Order Details` Values (10967, 19, 9.2, 12, 0.0);
INSERT INTO `Order Details` Values (10967, 49, 20, 40, 0.0);
INSERT INTO `Order Details` Values (10968, 12, 38, 30, 0.0);
INSERT INTO `Order Details` Values (10968, 24, 4.5, 30, 0.0);
INSERT INTO `Order Details` Values (10968, 64, 33.25, 4, 0.0);
INSERT INTO `Order Details` Values (10969, 46, 12, 9, 0.0);
INSERT INTO `Order Details` Values (10970, 52, 7, 40, 0.2);
INSERT INTO `Order Details` Values (10971, 29, 123.79, 14, 0.0);
INSERT INTO `Order Details` Values (10972, 17, 39, 6, 0.0);
INSERT INTO `Order Details` Values (10972, 33, 2.5, 7, 0.0);
INSERT INTO `Order Details` Values (10973, 26, 31.23, 5, 0.0);
INSERT INTO `Order Details` Values (10973, 41, 9.65, 6, 0.0);
INSERT INTO `Order Details` Values (10973, 75, 7.75, 10, 0.0);
INSERT INTO `Order Details` Values (10974, 63, 43.9, 10, 0.0);
INSERT INTO `Order Details` Values (10975, 8, 40, 16, 0.0);
INSERT INTO `Order Details` Values (10975, 75, 7.75, 10, 0.0);
INSERT INTO `Order Details` Values (10976, 28, 45.6, 20, 0.0);
INSERT INTO `Order Details` Values (10977, 39, 18, 30, 0.0);
INSERT INTO `Order Details` Values (10977, 47, 9.5, 30, 0.0);
INSERT INTO `Order Details` Values (10977, 51, 53, 10, 0.0);
INSERT INTO `Order Details` Values (10977, 63, 43.9, 20, 0.0);
INSERT INTO `Order Details` Values (10978, 8, 40, 20, 0.15);
INSERT INTO `Order Details` Values (10978, 21, 10, 40, 0.15);
INSERT INTO `Order Details` Values (10978, 40, 18.4, 10, 0.0);
INSERT INTO `Order Details` Values (10978, 44, 19.45, 6, 0.15);
INSERT INTO `Order Details` Values (10979, 7, 30, 18, 0.0);
INSERT INTO `Order Details` Values (10979, 12, 38, 20, 0.0);
INSERT INTO `Order Details` Values (10979, 24, 4.5, 80, 0.0);
INSERT INTO `Order Details` Values (10979, 27, 43.9, 30, 0.0);
INSERT INTO `Order Details` Values (10979, 31, 12.5, 24, 0.0);
INSERT INTO `Order Details` Values (10979, 63, 43.9, 35, 0.0);
INSERT INTO `Order Details` Values (10980, 75, 7.75, 40, 0.2);
INSERT INTO `Order Details` Values (10981, 38, 263.5, 60, 0.0);
INSERT INTO `Order Details` Values (10982, 7, 30, 20, 0.0);
INSERT INTO `Order Details` Values (10982, 43, 46, 9, 0.0);
INSERT INTO `Order Details` Values (10983, 13, 6, 84, 0.15);
INSERT INTO `Order Details` Values (10983, 57, 19.5, 15, 0.0);
INSERT INTO `Order Details` Values (10984, 16, 17.45, 55, 0.0);
INSERT INTO `Order Details` Values (10984, 24, 4.5, 20, 0.0);
INSERT INTO `Order Details` Values (10984, 36, 19, 40, 0.0);
INSERT INTO `Order Details` Values (10985, 16, 17.45, 36, 0.1);
INSERT INTO `Order Details` Values (10985, 18, 62.5, 8, 0.1);
INSERT INTO `Order Details` Values (10985, 32, 32, 35, 0.1);
INSERT INTO `Order Details` Values (10986, 11, 21, 30, 0.0);
INSERT INTO `Order Details` Values (10986, 20, 81, 15, 0.0);
INSERT INTO `Order Details` Values (10986, 76, 18, 10, 0.0);
INSERT INTO `Order Details` Values (10986, 77, 13, 15, 0.0);
INSERT INTO `Order Details` Values (10987, 7, 30, 60, 0.0);
INSERT INTO `Order Details` Values (10987, 43, 46, 6, 0.0);
INSERT INTO `Order Details` Values (10987, 72, 34.8, 20, 0.0);
INSERT INTO `Order Details` Values (10988, 7, 30, 60, 0.0);
INSERT INTO `Order Details` Values (10988, 62, 49.3, 40, 0.1);
INSERT INTO `Order Details` Values (10989, 6, 25, 40, 0.0);
INSERT INTO `Order Details` Values (10989, 11, 21, 15, 0.0);
INSERT INTO `Order Details` Values (10989, 41, 9.65, 4, 0.0);
INSERT INTO `Order Details` Values (10990, 21, 10, 65, 0.0);
INSERT INTO `Order Details` Values (10990, 34, 14, 60, 0.15);
INSERT INTO `Order Details` Values (10990, 55, 24, 65, 0.15);
INSERT INTO `Order Details` Values (10990, 61, 28.5, 66, 0.15);
INSERT INTO `Order Details` Values (10991, 2, 19, 50, 0.2);
INSERT INTO `Order Details` Values (10991, 70, 15, 20, 0.2);
INSERT INTO `Order Details` Values (10991, 76, 18, 90, 0.2);
INSERT INTO `Order Details` Values (10992, 72, 34.8, 2, 0.0);
INSERT INTO `Order Details` Values (10993, 29, 123.79, 50, 0.25);
INSERT INTO `Order Details` Values (10993, 41, 9.65, 35, 0.25);
INSERT INTO `Order Details` Values (10994, 59, 55, 18, 0.05);
INSERT INTO `Order Details` Values (10995, 51, 53, 20, 0.0);
INSERT INTO `Order Details` Values (10995, 60, 34, 4, 0.0);
INSERT INTO `Order Details` Values (10996, 42, 14, 40, 0.0);
INSERT INTO `Order Details` Values (10997, 32, 32, 50, 0.0);
INSERT INTO `Order Details` Values (10997, 46, 12, 20, 0.25);
INSERT INTO `Order Details` Values (10997, 52, 7, 20, 0.25);
INSERT INTO `Order Details` Values (10998, 24, 4.5, 12, 0.0);
INSERT INTO `Order Details` Values (10998, 61, 28.5, 7, 0.0);
INSERT INTO `Order Details` Values (10998, 74, 10, 20, 0.0);
INSERT INTO `Order Details` Values (10998, 75, 7.75, 30, 0.0);
INSERT INTO `Order Details` Values (10999, 41, 9.65, 20, 0.05);
INSERT INTO `Order Details` Values (10999, 51, 53, 15, 0.05);
INSERT INTO `Order Details` Values (10999, 77, 13, 21, 0.05);
INSERT INTO `Order Details` Values (11000, 4, 22, 25, 0.25);
INSERT INTO `Order Details` Values (11000, 24, 4.5, 30, 0.25);
INSERT INTO `Order Details` Values (11000, 77, 13, 30, 0.0);
INSERT INTO `Order Details` Values (11001, 7, 30, 60, 0.0);
INSERT INTO `Order Details` Values (11001, 22, 21, 25, 0.0);
INSERT INTO `Order Details` Values (11001, 46, 12, 25, 0.0);
INSERT INTO `Order Details` Values (11001, 55, 24, 6, 0.0);
INSERT INTO `Order Details` Values (11002, 13, 6, 56, 0.0);
INSERT INTO `Order Details` Values (11002, 35, 18, 15, 0.15);
INSERT INTO `Order Details` Values (11002, 42, 14, 24, 0.15);
INSERT INTO `Order Details` Values (11002, 55, 24, 40, 0.0);
INSERT INTO `Order Details` Values (11003, 1, 18, 4, 0.0);
INSERT INTO `Order Details` Values (11003, 40, 18.4, 10, 0.0);
INSERT INTO `Order Details` Values (11003, 52, 7, 10, 0.0);
INSERT INTO `Order Details` Values (11004, 26, 31.23, 6, 0.0);
INSERT INTO `Order Details` Values (11004, 76, 18, 6, 0.0);
INSERT INTO `Order Details` Values (11005, 1, 18, 2, 0.0);
INSERT INTO `Order Details` Values (11005, 59, 55, 10, 0.0);
INSERT INTO `Order Details` Values (11006, 1, 18, 8, 0.0);
INSERT INTO `Order Details` Values (11006, 29, 123.79, 2, 0.25);
INSERT INTO `Order Details` Values (11007, 8, 40, 30, 0.0);
INSERT INTO `Order Details` Values (11007, 29, 123.79, 10, 0.0);
INSERT INTO `Order Details` Values (11007, 42, 14, 14, 0.0);
INSERT INTO `Order Details` Values (11008, 28, 45.6, 70, 0.05);
INSERT INTO `Order Details` Values (11008, 34, 14, 90, 0.05);
INSERT INTO `Order Details` Values (11008, 71, 21.5, 21, 0.0);
INSERT INTO `Order Details` Values (11009, 24, 4.5, 12, 0.0);
INSERT INTO `Order Details` Values (11009, 36, 19, 18, 0.25);
INSERT INTO `Order Details` Values (11009, 60, 34, 9, 0.0);
INSERT INTO `Order Details` Values (11010, 7, 30, 20, 0.0);
INSERT INTO `Order Details` Values (11010, 24, 4.5, 10, 0.0);
INSERT INTO `Order Details` Values (11011, 58, 13.25, 40, 0.05);
INSERT INTO `Order Details` Values (11011, 71, 21.5, 20, 0.0);
INSERT INTO `Order Details` Values (11012, 19, 9.2, 50, 0.05);
INSERT INTO `Order Details` Values (11012, 60, 34, 36, 0.05);
INSERT INTO `Order Details` Values (11012, 71, 21.5, 60, 0.05);
INSERT INTO `Order Details` Values (11013, 23, 9, 10, 0.0);
INSERT INTO `Order Details` Values (11013, 42, 14, 4, 0.0);
INSERT INTO `Order Details` Values (11013, 45, 9.5, 20, 0.0);
INSERT INTO `Order Details` Values (11013, 68, 12.5, 2, 0.0);
INSERT INTO `Order Details` Values (11014, 41, 9.65, 28, 0.1);
INSERT INTO `Order Details` Values (11015, 30, 25.89, 15, 0.0);
INSERT INTO `Order Details` Values (11015, 77, 13, 18, 0.0);
INSERT INTO `Order Details` Values (11016, 31, 12.5, 15, 0.0);
INSERT INTO `Order Details` Values (11016, 36, 19, 16, 0.0);
INSERT INTO `Order Details` Values (11017, 3, 10, 25, 0.0);
INSERT INTO `Order Details` Values (11017, 59, 55, 110, 0.0);
INSERT INTO `Order Details` Values (11017, 70, 15, 30, 0.0);
INSERT INTO `Order Details` Values (11018, 12, 38, 20, 0.0);
INSERT INTO `Order Details` Values (11018, 18, 62.5, 10, 0.0);
INSERT INTO `Order Details` Values (11018, 56, 38, 5, 0.0);
INSERT INTO `Order Details` Values (11019, 46, 12, 3, 0.0);
INSERT INTO `Order Details` Values (11019, 49, 20, 2, 0.0);
INSERT INTO `Order Details` Values (11020, 10, 31, 24, 0.15);
INSERT INTO `Order Details` Values (11021, 2, 19, 11, 0.25);
INSERT INTO `Order Details` Values (11021, 20, 81, 15, 0.0);
INSERT INTO `Order Details` Values (11021, 26, 31.23, 63, 0.0);
INSERT INTO `Order Details` Values (11021, 51, 53, 44, 0.25);
INSERT INTO `Order Details` Values (11021, 72, 34.8, 35, 0.0);
INSERT INTO `Order Details` Values (11022, 19, 9.2, 35, 0.0);
INSERT INTO `Order Details` Values (11022, 69, 36, 30, 0.0);
INSERT INTO `Order Details` Values (11023, 7, 30, 4, 0.0);
INSERT INTO `Order Details` Values (11023, 43, 46, 30, 0.0);
INSERT INTO `Order Details` Values (11024, 26, 31.23, 12, 0.0);
INSERT INTO `Order Details` Values (11024, 33, 2.5, 30, 0.0);
INSERT INTO `Order Details` Values (11024, 65, 21.05, 21, 0.0);
INSERT INTO `Order Details` Values (11024, 71, 21.5, 50, 0.0);
INSERT INTO `Order Details` Values (11025, 1, 18, 10, 0.1);
INSERT INTO `Order Details` Values (11025, 13, 6, 20, 0.1);
INSERT INTO `Order Details` Values (11026, 18, 62.5, 8, 0.0);
INSERT INTO `Order Details` Values (11026, 51, 53, 10, 0.0);
INSERT INTO `Order Details` Values (11027, 24, 4.5, 30, 0.25);
INSERT INTO `Order Details` Values (11027, 62, 49.3, 21, 0.25);
INSERT INTO `Order Details` Values (11028, 55, 24, 35, 0.0);
INSERT INTO `Order Details` Values (11028, 59, 55, 24, 0.0);
INSERT INTO `Order Details` Values (11029, 56, 38, 20, 0.0);
INSERT INTO `Order Details` Values (11029, 63, 43.9, 12, 0.0);
INSERT INTO `Order Details` Values (11030, 2, 19, 100, 0.25);
INSERT INTO `Order Details` Values (11030, 5, 21.35, 70, 0.0);
INSERT INTO `Order Details` Values (11030, 29, 123.79, 60, 0.25);
INSERT INTO `Order Details` Values (11030, 59, 55, 100, 0.25);
INSERT INTO `Order Details` Values (11031, 1, 18, 45, 0.0);
INSERT INTO `Order Details` Values (11031, 13, 6, 80, 0.0);
INSERT INTO `Order Details` Values (11031, 24, 4.5, 21, 0.0);
INSERT INTO `Order Details` Values (11031, 64, 33.25, 20, 0.0);
INSERT INTO `Order Details` Values (11031, 71, 21.5, 16, 0.0);
INSERT INTO `Order Details` Values (11032, 36, 19, 35, 0.0);
INSERT INTO `Order Details` Values (11032, 38, 263.5, 25, 0.0);
INSERT INTO `Order Details` Values (11032, 59, 55, 30, 0.0);
INSERT INTO `Order Details` Values (11033, 53, 32.8, 70, 0.1);
INSERT INTO `Order Details` Values (11033, 69, 36, 36, 0.1);
INSERT INTO `Order Details` Values (11034, 21, 10, 15, 0.1);
INSERT INTO `Order Details` Values (11034, 44, 19.45, 12, 0.0);
INSERT INTO `Order Details` Values (11034, 61, 28.5, 6, 0.0);
INSERT INTO `Order Details` Values (11035, 1, 18, 10, 0.0);
INSERT INTO `Order Details` Values (11035, 35, 18, 60, 0.0);
INSERT INTO `Order Details` Values (11035, 42, 14, 30, 0.0);
INSERT INTO `Order Details` Values (11035, 54, 7.45, 10, 0.0);
INSERT INTO `Order Details` Values (11036, 13, 6, 7, 0.0);
INSERT INTO `Order Details` Values (11036, 59, 55, 30, 0.0);
INSERT INTO `Order Details` Values (11037, 70, 15, 4, 0.0);
INSERT INTO `Order Details` Values (11038, 40, 18.4, 5, 0.2);
INSERT INTO `Order Details` Values (11038, 52, 7, 2, 0.0);
INSERT INTO `Order Details` Values (11038, 71, 21.5, 30, 0.0);
INSERT INTO `Order Details` Values (11039, 28, 45.6, 20, 0.0);
INSERT INTO `Order Details` Values (11039, 35, 18, 24, 0.0);
INSERT INTO `Order Details` Values (11039, 49, 20, 60, 0.0);
INSERT INTO `Order Details` Values (11039, 57, 19.5, 28, 0.0);
INSERT INTO `Order Details` Values (11040, 21, 10, 20, 0.0);
INSERT INTO `Order Details` Values (11041, 2, 19, 30, 0.2);
INSERT INTO `Order Details` Values (11041, 63, 43.9, 30, 0.0);
INSERT INTO `Order Details` Values (11042, 44, 19.45, 15, 0.0);
INSERT INTO `Order Details` Values (11042, 61, 28.5, 4, 0.0);
INSERT INTO `Order Details` Values (11043, 11, 21, 10, 0.0);
INSERT INTO `Order Details` Values (11044, 62, 49.3, 12, 0.0);
INSERT INTO `Order Details` Values (11045, 33, 2.5, 15, 0.0);
INSERT INTO `Order Details` Values (11045, 51, 53, 24, 0.0);
INSERT INTO `Order Details` Values (11046, 12, 38, 20, 0.05);
INSERT INTO `Order Details` Values (11046, 32, 32, 15, 0.05);
INSERT INTO `Order Details` Values (11046, 35, 18, 18, 0.05);
INSERT INTO `Order Details` Values (11047, 1, 18, 25, 0.25);
INSERT INTO `Order Details` Values (11047, 5, 21.35, 30, 0.25);
INSERT INTO `Order Details` Values (11048, 68, 12.5, 42, 0.0);
INSERT INTO `Order Details` Values (11049, 2, 19, 10, 0.2);
INSERT INTO `Order Details` Values (11049, 12, 38, 4, 0.2);
INSERT INTO `Order Details` Values (11050, 76, 18, 50, 0.1);
INSERT INTO `Order Details` Values (11051, 24, 4.5, 10, 0.2);
INSERT INTO `Order Details` Values (11052, 43, 46, 30, 0.2);
INSERT INTO `Order Details` Values (11052, 61, 28.5, 10, 0.2);
INSERT INTO `Order Details` Values (11053, 18, 62.5, 35, 0.2);
INSERT INTO `Order Details` Values (11053, 32, 32, 20, 0.0);
INSERT INTO `Order Details` Values (11053, 64, 33.25, 25, 0.2);
INSERT INTO `Order Details` Values (11054, 33, 2.5, 10, 0.0);
INSERT INTO `Order Details` Values (11054, 67, 14, 20, 0.0);
INSERT INTO `Order Details` Values (11055, 24, 4.5, 15, 0.0);
INSERT INTO `Order Details` Values (11055, 25, 14, 15, 0.0);
INSERT INTO `Order Details` Values (11055, 51, 53, 20, 0.0);
INSERT INTO `Order Details` Values (11055, 57, 19.5, 20, 0.0);
INSERT INTO `Order Details` Values (11056, 7, 30, 40, 0.0);
INSERT INTO `Order Details` Values (11056, 55, 24, 35, 0.0);
INSERT INTO `Order Details` Values (11056, 60, 34, 50, 0.0);
INSERT INTO `Order Details` Values (11057, 70, 15, 3, 0.0);
INSERT INTO `Order Details` Values (11058, 21, 10, 3, 0.0);
INSERT INTO `Order Details` Values (11058, 60, 34, 21, 0.0);
INSERT INTO `Order Details` Values (11058, 61, 28.5, 4, 0.0);
INSERT INTO `Order Details` Values (11059, 13, 6, 30, 0.0);
INSERT INTO `Order Details` Values (11059, 17, 39, 12, 0.0);
INSERT INTO `Order Details` Values (11059, 60, 34, 35, 0.0);
INSERT INTO `Order Details` Values (11060, 60, 34, 4, 0.0);
INSERT INTO `Order Details` Values (11060, 77, 13, 10, 0.0);
INSERT INTO `Order Details` Values (11061, 60, 34, 15, 0.0);
INSERT INTO `Order Details` Values (11062, 53, 32.8, 10, 0.2);
INSERT INTO `Order Details` Values (11062, 70, 15, 12, 0.2);
INSERT INTO `Order Details` Values (11063, 34, 14, 30, 0.0);
INSERT INTO `Order Details` Values (11063, 40, 18.4, 40, 0.1);
INSERT INTO `Order Details` Values (11063, 41, 9.65, 30, 0.1);
INSERT INTO `Order Details` Values (11064, 17, 39, 77, 0.1);
INSERT INTO `Order Details` Values (11064, 41, 9.65, 12, 0.0);
INSERT INTO `Order Details` Values (11064, 53, 32.8, 25, 0.1);
INSERT INTO `Order Details` Values (11064, 55, 24, 4, 0.1);
INSERT INTO `Order Details` Values (11064, 68, 12.5, 55, 0.0);
INSERT INTO `Order Details` Values (11065, 30, 25.89, 4, 0.25);
INSERT INTO `Order Details` Values (11065, 54, 7.45, 20, 0.25);
INSERT INTO `Order Details` Values (11066, 16, 17.45, 3, 0.0);
INSERT INTO `Order Details` Values (11066, 19, 9.2, 42, 0.0);
INSERT INTO `Order Details` Values (11066, 34, 14, 35, 0.0);
INSERT INTO `Order Details` Values (11067, 41, 9.65, 9, 0.0);
INSERT INTO `Order Details` Values (11068, 28, 45.6, 8, 0.15);
INSERT INTO `Order Details` Values (11068, 43, 46, 36, 0.15);
INSERT INTO `Order Details` Values (11068, 77, 13, 28, 0.15);
INSERT INTO `Order Details` Values (11069, 39, 18, 20, 0.0);
INSERT INTO `Order Details` Values (11070, 1, 18, 40, 0.15);
INSERT INTO `Order Details` Values (11070, 2, 19, 20, 0.15);
INSERT INTO `Order Details` Values (11070, 16, 17.45, 30, 0.15);
INSERT INTO `Order Details` Values (11070, 31, 12.5, 20, 0.0);
INSERT INTO `Order Details` Values (11071, 7, 30, 15, 0.05);
INSERT INTO `Order Details` Values (11071, 13, 6, 10, 0.05);
INSERT INTO `Order Details` Values (11072, 2, 19, 8, 0.0);
INSERT INTO `Order Details` Values (11072, 41, 9.65, 40, 0.0);
INSERT INTO `Order Details` Values (11072, 50, 16.25, 22, 0.0);
INSERT INTO `Order Details` Values (11072, 64, 33.25, 130, 0.0);
INSERT INTO `Order Details` Values (11073, 11, 21, 10, 0.0);
INSERT INTO `Order Details` Values (11073, 24, 4.5, 20, 0.0);
INSERT INTO `Order Details` Values (11074, 16, 17.45, 14, 0.05);
INSERT INTO `Order Details` Values (11075, 2, 19, 10, 0.15);
INSERT INTO `Order Details` Values (11075, 46, 12, 30, 0.15);
INSERT INTO `Order Details` Values (11075, 76, 18, 2, 0.15);
INSERT INTO `Order Details` Values (11076, 6, 25, 20, 0.25);
INSERT INTO `Order Details` Values (11076, 14, 23.25, 20, 0.25);
INSERT INTO `Order Details` Values (11076, 19, 9.2, 10, 0.25);
INSERT INTO `Order Details` Values (11077, 2, 19, 24, 0.2);
INSERT INTO `Order Details` Values (11077, 3, 10, 4, 0.0);
INSERT INTO `Order Details` Values (11077, 4, 22, 1, 0.0);
INSERT INTO `Order Details` Values (11077, 6, 25, 1, 0.02);
INSERT INTO `Order Details` Values (11077, 7, 30, 1, 0.05);
INSERT INTO `Order Details` Values (11077, 8, 40, 2, 0.1);
INSERT INTO `Order Details` Values (11077, 10, 31, 1, 0.0);
INSERT INTO `Order Details` Values (11077, 12, 38, 2, 0.05);
INSERT INTO `Order Details` Values (11077, 13, 6, 4, 0.0);
INSERT INTO `Order Details` Values (11077, 14, 23.25, 1, 0.03);
INSERT INTO `Order Details` Values (11077, 16, 17.45, 2, 0.03);
INSERT INTO `Order Details` Values (11077, 20, 81, 1, 0.04);
INSERT INTO `Order Details` Values (11077, 23, 9, 2, 0.0);
INSERT INTO `Order Details` Values (11077, 32, 32, 1, 0.0);
INSERT INTO `Order Details` Values (11077, 39, 18, 2, 0.05);
INSERT INTO `Order Details` Values (11077, 41, 9.65, 3, 0.0);
INSERT INTO `Order Details` Values (11077, 46, 12, 3, 0.02);
INSERT INTO `Order Details` Values (11077, 52, 7, 2, 0.0);
INSERT INTO `Order Details` Values (11077, 55, 24, 2, 0.0);
INSERT INTO `Order Details` Values (11077, 60, 34, 2, 0.06);
INSERT INTO `Order Details` Values (11077, 64, 33.25, 2, 0.03);
INSERT INTO `Order Details` Values (11077, 66, 17, 1, 0.0);
INSERT INTO `Order Details` Values (11077, 73, 15, 2, 0.01);
INSERT INTO `Order Details` Values (11077, 75, 7.75, 4, 0.0);
INSERT INTO `Order Details` Values (11077, 77, 13, 2, 0.0);

# ---------------------------------------------------------------------- #
# Add info into "Products"                                               #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Products;
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(1, 'Chai', 1, 1, '10 boxes x 20 bags', 18, 39, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(2, 'Chang', 1, 1, '24 - 12 oz bottles', 19, 17, 40, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(3, 'Aniseed Syrup', 1, 2, '12 - 550 ml bottles', 10, 13, 70, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(4, 'Chef Anton''s Cajun Seasoning', 2, 2, '48 - 6 oz jars', 22, 53, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(5, 'Chef Anton''s Gumbo Mix', 2, 2, '36 boxes', 21.35, 0, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(6, 'Grandma''s Boysenberry Spread', 3, 2, '12 - 8 oz jars', 25, 120, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(7, 'Uncle Bob''s Organic Dried Pears', 3, 7, '12 - 1 lb pkgs.', 30, 15, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(8, 'Northwoods Cranberry Sauce', 3, 2, '12 - 12 oz jars', 40, 6, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(9, 'Mishi Kobe Niku', 4, 6, '18 - 500 g pkgs.', 97, 29, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(10, 'Ikura', 4, 8, '12 - 200 ml jars', 31, 31, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(11, 'Queso Cabrales', 5, 4, '1 kg pkg.', 21, 22, 30, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(12, 'Queso Manchego La Pastora', 5, 4, '10 - 500 g pkgs.', 38, 86, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(13, 'Konbu', 6, 8, '2 kg box', 6, 24, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(14, 'Tofu', 6, 7, '40 - 100 g pkgs.', 23.25, 35, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(15, 'Genen Shouyu', 6, 2, '24 - 250 ml bottles', 15.5, 39, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(16, 'Pavlova', 7, 3, '32 - 500 g boxes', 17.45, 29, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(17, 'Alice Mutton', 7, 6, '20 - 1 kg tins', 39, 0, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(18, 'Carnarvon Tigers', 7, 8, '16 kg pkg.', 62.5, 42, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(19, 'Teatime Chocolate Biscuits', 8, 3, '10 boxes x 12 pieces', 9.2, 25, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(20, 'Sir Rodney''s Marmalade', 8, 3, '30 gift boxes', 81, 40, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(21, 'Sir Rodney''s Scones', 8, 3, '24 pkgs. x 4 pieces', 10, 3, 40, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(22, 'Gustaf''s Knckebrd', 9, 5, '24 - 500 g pkgs.', 21, 104, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(23, 'Tunnbrd', 9, 5, '12 - 250 g pkgs.', 9, 61, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(24, 'Guaran Fantstica', 10, 1, '12 - 355 ml cans', 4.5, 20, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(25, 'NuNuCa Nu-Nougat-Creme', 11, 3, '20 - 450 g glasses', 14, 76, 0, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(26, 'Gumbr Gummibrchen', 11, 3, '100 - 250 g bags', 31.23, 15, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(27, 'Schoggi Schokolade', 11, 3, '100 - 100 g pieces', 43.9, 49, 0, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(28, 'Rssle Sauerkraut', 12, 7, '25 - 825 g cans', 45.6, 26, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(29, 'Thringer Rostbratwurst', 12, 6, '50 bags x 30 sausgs.', 123.79, 0, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(30, 'Nord-Ost Matjeshering', 13, 8, '10 - 200 g glasses', 25.89, 10, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(31, 'Gorgonzola Telino', 14, 4, '12 - 100 g pkgs', 12.5, 0, 70, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(32, 'Mascarpone Fabioli', 14, 4, '24 - 200 g pkgs.', 32, 9, 40, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(33, 'Geitost', 15, 4, '500 g', 2.5, 112, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(34, 'Sasquatch Ale', 16, 1, '24 - 12 oz bottles', 14, 111, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(35, 'Steeleye Stout', 16, 1, '24 - 12 oz bottles', 18, 20, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(36, 'Inlagd Sill', 17, 8, '24 - 250 g  jars', 19, 112, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(37, 'Gravad lax', 17, 8, '12 - 500 g pkgs.', 26, 11, 50, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(38, 'Cte de Blaye', 18, 1, '12 - 75 cl bottles', 263.5, 17, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(39, 'Chartreuse verte', 18, 1, '750 cc per bottle', 18, 69, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(40, 'Boston Crab Meat', 19, 8, '24 - 4 oz tins', 18.4, 123, 0, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(41, 'Jack''s New England Clam Chowder', 19, 8, '12 - 12 oz cans', 9.65, 85, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(42, 'Singaporean Hokkien Fried Mee', 20, 5, '32 - 1 kg pkgs.', 14, 26, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(43, 'Ipoh Coffee', 20, 1, '16 - 500 g tins', 46, 17, 10, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(44, 'Gula Malacca', 20, 2, '20 - 2 kg bags', 19.45, 27, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(45, 'Rogede sild', 21, 8, '1k pkg.', 9.5, 5, 70, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(46, 'Spegesild', 21, 8, '4 - 450 g glasses', 12, 95, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(47, 'Zaanse koeken', 22, 3, '10 - 4 oz boxes', 9.5, 36, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(48, 'Chocolade', 22, 3, '10 pkgs.', 12.75, 15, 70, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(49, 'Maxilaku', 23, 3, '24 - 50 g pkgs.', 20, 10, 60, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(50, 'Valkoinen suklaa', 23, 3, '12 - 100 g bars', 16.25, 65, 0, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(51, 'Manjimup Dried Apples', 24, 7, '50 - 300 g pkgs.', 53, 20, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(52, 'Filo Mix', 24, 5, '16 - 2 kg boxes', 7, 38, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(53, 'Perth Pasties', 24, 6, '48 pieces', 32.8, 0, 0, 0, 1);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(54, 'Tourtire', 25, 6, '16 pies', 7.45, 21, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(55, 'Pt chinois', 25, 6, '24 boxes x 2 pies', 24, 115, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(56, 'Gnocchi di nonna Alice', 26, 5, '24 - 250 g pkgs.', 38, 21, 10, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(57, 'Ravioli Angelo', 26, 5, '24 - 250 g pkgs.', 19.5, 36, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(58, 'Escargots de Bourgogne', 27, 8, '24 pieces', 13.25, 62, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(59, 'Raclette Courdavault', 28, 4, '5 kg pkg.', 55, 79, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(60, 'Camembert Pierrot', 28, 4, '15 - 300 g rounds', 34, 19, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(61, 'Sirop d''rable', 29, 2, '24 - 500 ml bottles', 28.5, 113, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(62, 'Tarte au sucre', 29, 3, '48 pies', 49.3, 17, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(63, 'Vegie-spread', 7, 2, '15 - 625 g jars', 43.9, 24, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(64, 'Wimmers gute Semmelkndel', 12, 5, '20 bags x 4 pieces', 33.25, 22, 80, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(65, 'Louisiana Fiery Hot Pepper Sauce', 2, 2, '32 - 8 oz bottles', 21.05, 76, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(66, 'Louisiana Hot Spiced Okra', 2, 2, '24 - 8 oz jars', 17, 4, 100, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(67, 'Laughing Lumberjack Lager', 16, 1, '24 - 12 oz bottles', 14, 52, 0, 10, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(68, 'Scottish Longbreads', 8, 3, '10 boxes x 8 pieces', 12.5, 6, 10, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(69, 'Gudbrandsdalsost', 15, 4, '10 kg pkg.', 36, 26, 0, 15, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(70, 'Outback Lager', 7, 1, '24 - 355 ml bottles', 15, 15, 10, 30, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(71, 'Flotemysost', 15, 4, '10 - 500 g pkgs.', 21.5, 26, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(72, 'Mozzarella di Giovanni', 14, 4, '24 - 200 g pkgs.', 34.8, 14, 0, 0, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(73, 'Rd Kaviar', 17, 8, '24 - 150 g jars', 15, 101, 0, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(74, 'Longlife Tofu', 4, 7, '5 kg pkg.', 10, 4, 20, 5, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(75, 'Rhnbru Klosterbier', 12, 1, '24 - 0.5 l bottles', 7.75, 125, 0, 25, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(76, 'Lakkalikri', 23, 1, '500 ml', 18, 57, 0, 20, 0);
INSERT INTO Products (ProductID, ProductName, SupplierID, CategoryID, QuantityPerUnit, UnitPrice, UnitsInStock, UnitsOnOrder, ReorderLevel, Discontinued)
VALUES(77, 'Original Frankfurter grne Soe', 12, 2, '12 boxes', 13, 32, 0, 15, 0);

# ---------------------------------------------------------------------- #
# Add info into "Region"                                                 #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Region;
INSERT INTO Region (RegionID, RegionDescription)
VALUES(1, 'Eastern                                           ');
INSERT INTO Region (RegionID, RegionDescription)
VALUES(2, 'Westerns                                           ');
INSERT INTO Region (RegionID, RegionDescription)
VALUES(3, 'Northern                                          ');
INSERT INTO Region (RegionID, RegionDescription)
VALUES(4, 'Southern                                          ');

# ---------------------------------------------------------------------- #
# Add info into "Shippers"                                               #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Shippers;
INSERT INTO Shippers (ShipperID, CompanyName, Phone)
VALUES(1, 'Speedy Express', '(503) 555-9831');
INSERT INTO Shippers (ShipperID, CompanyName, Phone)
VALUES(2, 'United Package', '(503) 555-3199');
INSERT INTO Shippers (ShipperID, CompanyName, Phone)
VALUES(3, 'Federal Shipping', '(503) 555-9931');

# ---------------------------------------------------------------------- #
# Add info into "Suppliers"                                              #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Suppliers;
INSERT INTO Suppliers VALUES(1,'Exotic Liquids','Charlotte Cooper','Purchasing Manager','49 Gilbert St.','London',NULL,'EC1 4SD','UK','(171) 555-2222',NULL,NULL);
INSERT INTO Suppliers VALUES(2,'New Orleans Cajun Delights','Shelley Burke','Order Administrator','P.O. Box 78934','New Orleans','LA','70117','USA','(100) 555-4822',NULL,'#CAJUN.HTM#');
INSERT INTO Suppliers VALUES(3,'Grandma Kelly''s Homestead','Regina Murphy','Sales Representative','707 Oxford Rd.','Ann Arbor','MI','48104','USA','(313) 555-5735','(313) 555-3349',NULL);
INSERT INTO Suppliers VALUES(4,'Tokyo Traders','Yoshi Nagase','Marketing Manager','9-8 Sekimai
Musashino-shi','Tokyo',NULL,'100','Japan','(03) 3555-5011',NULL,NULL);
INSERT INTO Suppliers VALUES(5,'Cooperativa de Quesos ''Las Cabras''','Antonio del Valle Saavedra ','Export Administrator','Calle del Rosal 4','Oviedo','Asturias','33007','Spain','(98) 598 76 54',NULL,NULL);
INSERT INTO Suppliers VALUES(6,'Mayumi''s','Mayumi Ohno','Marketing Representative','92 Setsuko
Chuo-ku','Osaka',NULL,'545','Japan','(06) 431-7877',NULL,'Mayumi''s (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/mayumi.htm#');
INSERT INTO Suppliers VALUES(7,'Pavlova, Ltd.','Ian Devling','Marketing Manager','74 Rose St.
Moonie Ponds','Melbourne','Victoria','3058','Australia','(03) 444-2343','(03) 444-6588',NULL);
INSERT INTO Suppliers VALUES(8,'Specialty Biscuits, Ltd.','Peter Wilson','Sales Representative','29 King''s Way','Manchester',NULL,'M14 GSD','UK','(161) 555-4448',NULL,NULL);
INSERT INTO Suppliers VALUES(9,'PB Knckebrd AB','Lars Peterson','Sales Agent','Kaloadagatan 13','Gteborg',NULL,'S-345 67','Sweden ','031-987 65 43','031-987 65 91',NULL);
INSERT INTO Suppliers VALUES(10,'Refrescos Americanas LTDA','Carlos Diaz','Marketing Manager','Av. das Americanas 12.890','So Paulo',NULL,'5442','Brazil','(11) 555 4640',NULL,NULL);
INSERT INTO Suppliers VALUES(11,'Heli Swaren GmbH & Co. KG','Petra Winkler','Sales Manager','Tiergartenstrae 5','Berlin',NULL,'10785','Germany','(010) 9984510',NULL,NULL);
INSERT INTO Suppliers VALUES(12,'Plutzer Lebensmittelgromrkte AG','Martin Bein','International Marketing Mgr.','Bogenallee 51','Frankfurt',NULL,'60439','Germany','(069) 992755',NULL,'Plutzer (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/plutzer.htm#');
INSERT INTO Suppliers VALUES(13,'Nord-Ost-Fisch Handelsgesellschaft mbH','Sven Petersen','Coordinator Foreign Markets','Frahmredder 112a','Cuxhaven',NULL,'27478','Germany','(04721) 8713','(04721) 8714',NULL);
INSERT INTO Suppliers VALUES(14,'Formaggi Fortini s.r.l.','Elio Rossi','Sales Representative','Viale Dante, 75','Ravenna',NULL,'48100','Italy','(0544) 60323','(0544) 60603','#FORMAGGI.HTM#');
INSERT INTO Suppliers VALUES(15,'Norske Meierier','Beate Vileid','Marketing Manager','Hatlevegen 5','Sandvika',NULL,'1320','Norway','(0)2-953010',NULL,NULL);
INSERT INTO Suppliers VALUES(16,'Bigfoot Breweries','Cheryl Saylor','Regional Account Rep.','3400 - 8th Avenue
Suite 210','Bend','OR','97101','USA','(503) 555-9931',NULL,NULL);
INSERT INTO Suppliers VALUES(17,'Svensk Sjfda AB','Michael Bjrn','Sales Representative','Brovallavgen 231','Stockholm',NULL,'S-123 45','Sweden','08-123 45 67',NULL,NULL);
INSERT INTO Suppliers VALUES(18,'Aux joyeux ecclsiastiques','Guylne Nodier','Sales Manager','203, Rue des Francs-Bourgeois','Paris',NULL,'75004','France','(1) 03.83.00.68','(1) 03.83.00.62',NULL);
INSERT INTO Suppliers VALUES(19,'New England Seafood Cannery','Robb Merchant','Wholesale Account Agent','Order Processing Dept.
2100 Paul Revere Blvd.','Boston','MA','02134','USA','(617) 555-3267','(617) 555-3389',NULL);
INSERT INTO Suppliers VALUES(20,'Leka Trading','Chandra Leka','Owner','471 Serangoon Loop, Suite #402','Singapore',NULL,'0512','Singapore','555-8787',NULL,NULL);
INSERT INTO Suppliers VALUES(21,'Lyngbysild','Niels Petersen','Sales Manager','Lyngbysild
Fiskebakken 10','Lyngby',NULL,'2800','Denmark','43844108','43844115',NULL);
INSERT INTO Suppliers VALUES(22,'Zaanse Snoepfabriek','Dirk Luchte','Accounting Manager','Verkoop
Rijnweg 22','Zaandam',NULL,'9999 ZZ','Netherlands','(12345) 1212','(12345) 1210',NULL);
INSERT INTO Suppliers VALUES(23,'Karkki Oy','Anne Heikkonen','Product Manager','Valtakatu 12','Lappeenranta',NULL,'53120','Finland','(953) 10956',NULL,NULL);
INSERT INTO Suppliers VALUES(24,'G''day, Mate','Wendy Mackenzie','Sales Representative','170 Prince Edward Parade
Hunter''s Hill','Sydney','NSW','2042','Australia','(02) 555-5914','(02) 555-4873','G''day Mate (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/gdaymate.htm#');
INSERT INTO Suppliers VALUES(25,'Ma Maison','Jean-Guy Lauzon','Marketing Manager','2960 Rue St. Laurent','Montral','Qubec','H1J 1C3','Canada','(514) 555-9022',NULL,NULL);
INSERT INTO Suppliers VALUES(26,'Pasta Buttini s.r.l.','Giovanni Giudici','Order Administrator','Via dei Gelsomini, 153','Salerno',NULL,'84100','Italy','(089) 6547665','(089) 6547667',NULL);
INSERT INTO Suppliers VALUES(27,'Escargots Nouveaux','Marie Delamare','Sales Manager','22, rue H. Voiron','Montceau',NULL,'71300','France','85.57.00.07',NULL,NULL);
INSERT INTO Suppliers VALUES(28,'Gai pturage','Eliane Noz','Sales Representative','Bat. B
3, rue des Alpes','Annecy',NULL,'74000','France','38.76.98.06','38.76.98.58',NULL);
INSERT INTO Suppliers VALUES(29,'Forts d''rables','Chantal Goulet','Accounting Manager','148 rue Chasseur','Ste-Hyacinthe','Qubec','J2S 7S8','Canada','(514) 555-2955','(514) 555-2921',NULL);

# ---------------------------------------------------------------------- #
# Add info into "Territories"                                            #
# ---------------------------------------------------------------------- #

TRUNCATE TABLE Territories;
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('01581', 'Westboro                                          ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('01730', 'Bedford                                           ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('01833', 'Georgetow                                         ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('02116', 'Boston                                            ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('02139', 'Cambridge                                         ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('02184', 'Braintree                                         ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('02903', 'Providence                                        ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('03049', 'Hollis                                            ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('03801', 'Portsmouth                                        ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('06897', 'Wilton                                            ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('07960', 'Morristown                                        ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('08837', 'Edison                                            ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('10019', 'New York                                          ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('10038', 'New York                                          ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('11747', 'Mellvile                                          ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('14450', 'Fairport                                          ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('19428', 'Philadelphia                                      ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('19713', 'Neward                                            ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('20852', 'Rockville                                         ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('27403', 'Greensboro                                        ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('27511', 'Cary                                              ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('29202', 'Columbia                                          ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('30346', 'Atlanta                                           ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('31406', 'Savannah                                          ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('32859', 'Orlando                                           ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('33607', 'Tampa                                             ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('40222', 'Louisville                                        ', 1);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('44122', 'Beachwood                                         ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('45839', 'Findlay                                           ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('48075', 'Southfield                                        ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('48084', 'Troy                                              ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('48304', 'Bloomfield Hills                                  ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('53404', 'Racine                                            ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('55113', 'Roseville                                         ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('55439', 'Minneapolis                                       ', 3);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('60179', 'Hoffman Estates                                   ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('60601', 'Chicago                                           ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('72716', 'Bentonville                                       ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('75234', 'Dallas                                            ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('78759', 'Austin                                            ', 4);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('80202', 'Denver                                            ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('80909', 'Colorado Springs                                  ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('85014', 'Phoenix                                           ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('85251', 'Scottsdale                                        ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('90405', 'Santa Monica                                      ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('94025', 'Menlo Park                                        ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('94105', 'San Francisco                                     ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('95008', 'Campbell                                          ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('95054', 'Santa Clara                                       ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('95060', 'Santa Cruz                                        ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('98004', 'Bellevue                                          ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('98052', 'Redmond                                           ', 2);
INSERT INTO Territories (TerritoryID, TerritoryDescription, RegionID)
VALUES('98104', 'Seattle                                           ', 2);


# ---------------------------------------------------------------------- #
# Foreign key constraints                                                #
# ---------------------------------------------------------------------- #

ALTER TABLE `CustomerCustomerDemo` ADD CONSTRAINT `FK_CustomerCustomerDemo` 
    FOREIGN KEY (`CustomerTypeID`) REFERENCES `CustomerDemographics` (`CustomerTypeID`);

ALTER TABLE `CustomerCustomerDemo` ADD CONSTRAINT `FK_CustomerCustomerDemo_Customers` 
    FOREIGN KEY (`CustomerID`) REFERENCES `Customers` (`CustomerID`);

ALTER TABLE `Employees` ADD CONSTRAINT `FK_Employees_Employees` 
    FOREIGN KEY (`ReportsTo`) REFERENCES `Employees` (`EmployeeID`);

ALTER TABLE `EmployeeTerritories` ADD CONSTRAINT `FK_EmployeeTerritories_Employees` 
    FOREIGN KEY (`EmployeeID`) REFERENCES `Employees` (`EmployeeID`);

ALTER TABLE `EmployeeTerritories` ADD CONSTRAINT `FK_EmployeeTerritories_Territories` 
    FOREIGN KEY (`TerritoryID`) REFERENCES `Territories` (`TerritoryID`);

ALTER TABLE `Order Details` ADD CONSTRAINT `FK_Order_Details_Orders` 
    FOREIGN KEY (`OrderID`) REFERENCES `Orders` (`OrderID`);

ALTER TABLE `Order Details` ADD CONSTRAINT `FK_Order_Details_Products` 
    FOREIGN KEY (`ProductID`) REFERENCES `Products` (`ProductID`);

ALTER TABLE `Orders` ADD CONSTRAINT `FK_Orders_Customers` 
    FOREIGN KEY (`CustomerID`) REFERENCES `Customers` (`CustomerID`);

ALTER TABLE `Orders` ADD CONSTRAINT `FK_Orders_Employees` 
    FOREIGN KEY (`EmployeeID`) REFERENCES `Employees` (`EmployeeID`);

ALTER TABLE `Orders` ADD CONSTRAINT `FK_Orders_Shippers` 
    FOREIGN KEY (`ShipVia`) REFERENCES `Shippers` (`ShipperID`);

ALTER TABLE `Products` ADD CONSTRAINT `FK_Products_Categories` 
    FOREIGN KEY (`CategoryID`) REFERENCES `Categories` (`CategoryID`);

ALTER TABLE `Products` ADD CONSTRAINT `FK_Products_Suppliers` 
    FOREIGN KEY (`SupplierID`) REFERENCES `Suppliers` (`SupplierID`);

ALTER TABLE `Territories` ADD CONSTRAINT `FK_Territories_Region` 
    FOREIGN KEY (`RegionID`) REFERENCES `Region` (`RegionID`);
    
# ---------------------------------------------------------------------- #
# Add View "Alphabetical list of products"                               #
# ---------------------------------------------------------------------- #

CREATE VIEW `Alphabetical list of products`
AS
SELECT Products.*, 
       Categories.CategoryName
FROM Categories 
   INNER JOIN Products ON Categories.CategoryID = Products.CategoryID
WHERE (((Products.Discontinued)=0));

# ---------------------------------------------------------------------- #
# Add View "Current Product List"                                        #
# ---------------------------------------------------------------------- #

CREATE VIEW `Current Product List`
AS
SELECT ProductID,
       ProductName 
FROM Products 
WHERE Discontinued=0;

# ---------------------------------------------------------------------- #
# Add View "Customer and Suppliers by City"                              #
# ---------------------------------------------------------------------- #

CREATE VIEW `Customer and Suppliers by City`
AS
SELECT City, 
       CompanyName, 
       ContactName, 
       'Customers' AS Relationship 
FROM Customers
UNION 
SELECT City, 
       CompanyName, 
       ContactName, 
       'Suppliers'
FROM Suppliers 
ORDER BY City, CompanyName;

# ---------------------------------------------------------------------- #
# Add View "Invoices"                                                    #
# ---------------------------------------------------------------------- #

CREATE VIEW `Invoices`
AS
SELECT Orders.ShipName,
       Orders.ShipAddress,
       Orders.ShipCity,
       Orders.ShipRegion, 
       Orders.ShipPostalCode,
       Orders.ShipCountry,
       Orders.CustomerID,
       Customers.CompanyName AS CustomerName, 
       Customers.Address,
       Customers.City,
       Customers.Region,
       Customers.PostalCode,
       Customers.Country,
       (Employees.FirstName + ' ' + Employees.LastName) AS Salesperson, 
       Orders.OrderID,
       Orders.OrderDate,
       Orders.RequiredDate,
       Orders.ShippedDate, 
       Shippers.CompanyName As ShipperName,
       `Order Details`.ProductID,
       Products.ProductName, 
       `Order Details`.UnitPrice,
       `Order Details`.Quantity,
       `Order Details`.Discount, 
       (((`Order Details`.UnitPrice*Quantity*(1-Discount))/100)*100) AS ExtendedPrice,
       Orders.Freight 
FROM Customers 
  JOIN Orders ON Customers.CustomerID = Orders.CustomerID  
    JOIN Employees ON Employees.EmployeeID = Orders.EmployeeID    
     JOIN `Order Details` ON Orders.OrderID = `Order Details`.OrderID     
      JOIN Products ON Products.ProductID = `Order Details`.ProductID      
       JOIN Shippers ON Shippers.ShipperID = Orders.ShipVia;

# ---------------------------------------------------------------------- #
# Add View "Orders Qry"                                                  #
# ---------------------------------------------------------------------- #
	   
CREATE VIEW `Orders Qry` AS
SELECT Orders.OrderID,
       Orders.CustomerID,
       Orders.EmployeeID, 
       Orders.OrderDate, 
       Orders.RequiredDate,
       Orders.ShippedDate, 
       Orders.ShipVia, 
       Orders.Freight,
       Orders.ShipName, 
       Orders.ShipAddress, 
       Orders.ShipCity,
       Orders.ShipRegion,
       Orders.ShipPostalCode,
       Orders.ShipCountry,
       Customers.CompanyName,
       Customers.Address,
       Customers.City,
       Customers.Region,
       Customers.PostalCode, 
       Customers.Country
FROM Customers 
     JOIN Orders ON Customers.CustomerID = Orders.CustomerID;     

# ---------------------------------------------------------------------- #
# Add View "Order Subtotals"                                             #
# ---------------------------------------------------------------------- #
	 
CREATE VIEW `Order Subtotals` AS
SELECT `Order Details`.OrderID, 
Sum((`Order Details`.UnitPrice*Quantity*(1-Discount)/100)*100) AS Subtotal
FROM `Order Details`
GROUP BY `Order Details`.OrderID;

# ---------------------------------------------------------------------- #
# Add View "Product Sales for 1997"                                      #
# ---------------------------------------------------------------------- #

CREATE VIEW `Product Sales for 1997` AS
SELECT Categories.CategoryName, 
       Products.ProductName, 
       Sum((`Order Details`.UnitPrice*Quantity*(1-Discount)/100)*100) AS ProductSales
FROM Categories
 JOIN    Products On Categories.CategoryID = Products.CategoryID
    JOIN  `Order Details` on Products.ProductID = `Order Details`.ProductID     
     JOIN  `Orders` on Orders.OrderID = `Order Details`.OrderID 
WHERE Orders.ShippedDate Between '1997-01-01' And '1997-12-31'
GROUP BY Categories.CategoryName, Products.ProductName;

# ---------------------------------------------------------------------- #
# Add View "Products Above Average Price"                                #
# ---------------------------------------------------------------------- #

CREATE VIEW `Products Above Average Price` AS
SELECT Products.ProductName, 
       Products.UnitPrice
FROM Products
WHERE Products.UnitPrice>(SELECT AVG(UnitPrice) From Products);

# ---------------------------------------------------------------------- #
# Add View "Products by Category"                                        #
# ---------------------------------------------------------------------- #

CREATE VIEW `Products by Category` AS
SELECT Categories.CategoryName, 
       Products.ProductName, 
       Products.QuantityPerUnit, 
       Products.UnitsInStock, 
       Products.Discontinued
FROM Categories 
     INNER JOIN Products ON Categories.CategoryID = Products.CategoryID
WHERE Products.Discontinued <> 1;

# ---------------------------------------------------------------------- #
# Add View "Quarterly Orders"                                            #
# ---------------------------------------------------------------------- #

CREATE VIEW `Quarterly Orders` AS
SELECT DISTINCT Customers.CustomerID, 
                Customers.CompanyName, 
                Customers.City, 
                Customers.Country
FROM Customers 
     JOIN Orders ON Customers.CustomerID = Orders.CustomerID
WHERE Orders.OrderDate BETWEEN '1997-01-01' And '1997-12-31';

# ---------------------------------------------------------------------- #
# Add View "Sales Totals by Amount"                                      #
# ---------------------------------------------------------------------- #

CREATE VIEW `Sales Totals by Amount` AS
SELECT `Order Subtotals`.Subtotal AS SaleAmount, 
                  Orders.OrderID, 
               Customers.CompanyName, 
                  Orders.ShippedDate
FROM Customers 
 JOIN Orders ON Customers.CustomerID = Orders.CustomerID
    JOIN `Order Subtotals` ON Orders.OrderID = `Order Subtotals`.OrderID 
WHERE (`Order Subtotals`.Subtotal >2500) 
AND (Orders.ShippedDate BETWEEN '1997-01-01' And '1997-12-31');

# ---------------------------------------------------------------------- #
# Add View "Summary of Sales by Quarter"                                 #
# ---------------------------------------------------------------------- #

CREATE VIEW `Summary of Sales by Quarter` AS
SELECT Orders.ShippedDate, 
       Orders.OrderID, 
       `Order Subtotals`.Subtotal
FROM Orders 
     INNER JOIN `Order Subtotals` ON Orders.OrderID = `Order Subtotals`.OrderID
WHERE Orders.ShippedDate IS NOT NULL;

# ---------------------------------------------------------------------- #
# Add View "Summary of Sales by Year"                                    #
# ---------------------------------------------------------------------- #

CREATE VIEW `Summary of Sales by Year` AS
SELECT      Orders.ShippedDate, 
            Orders.OrderID, 
 `Order Subtotals`.Subtotal
FROM Orders 
     INNER JOIN `Order Subtotals` ON Orders.OrderID = `Order Subtotals`.OrderID
WHERE Orders.ShippedDate IS NOT NULL;

# ---------------------------------------------------------------------- #
# Add View "Category Sales for 1997"                                     #
# ---------------------------------------------------------------------- #

CREATE VIEW `Category Sales for 1997` AS
SELECT     `Product Sales for 1997`.CategoryName, 
       Sum(`Product Sales for 1997`.ProductSales) AS CategorySales
FROM `Product Sales for 1997`
GROUP BY `Product Sales for 1997`.CategoryName;

# ---------------------------------------------------------------------- #
# Add View "Order Details Extended"                                      #
# ---------------------------------------------------------------------- #

CREATE VIEW `Order Details Extended` AS
SELECT `Order Details`.OrderID, 
       `Order Details`.ProductID, 
       Products.ProductName, 
	   `Order Details`.UnitPrice, 
       `Order Details`.Quantity, 
       `Order Details`.Discount, 
      (`Order Details`.UnitPrice*Quantity*(1-Discount)/100)*100 AS ExtendedPrice
FROM Products 
     JOIN `Order Details` ON Products.ProductID = `Order Details`.ProductID;

# ---------------------------------------------------------------------- #
# Add View "Sales by Category"                                           #
# ---------------------------------------------------------------------- #

CREATE VIEW `Sales by Category` AS
SELECT Categories.CategoryID, 
       Categories.CategoryName, 
         Products.ProductName, 
	Sum(`Order Details Extended`.ExtendedPrice) AS ProductSales
FROM  Categories 
    JOIN Products 
      ON Categories.CategoryID = Products.CategoryID
       JOIN `Order Details Extended` 
         ON Products.ProductID = `Order Details Extended`.ProductID                
           JOIN Orders 
             ON Orders.OrderID = `Order Details Extended`.OrderID 
WHERE Orders.OrderDate BETWEEN '1997-01-01' And '1997-12-31'
GROUP BY Categories.CategoryID, Categories.CategoryName, Products.ProductName;

# ---------------------------------------------------------------------- #
# Add Procedure "CustOrderHist"                                          #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `CustOrderHist`;

DELIMITER $$

CREATE PROCEDURE `CustOrderHist`(in AtCustomerID varchar(5))
BEGIN

SELECT ProductName,
    SUM(Quantity) as TOTAL
FROM Products P,
     `Order Details` OD,
     Orders O,
     Customers C
WHERE C.CustomerID = AtCustomerID
  AND C.CustomerID = O.CustomerID
  AND O.OrderID = OD.OrderID
  AND OD.ProductID = P.ProductID
GROUP BY ProductName;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "CustOrdersOrders"                                       #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `CustOrdersOrders`;

DELIMITER $$

CREATE PROCEDURE `CustOrdersOrders`(in AtCustomerID varchar(5))
BEGIN
      SELECT OrderID,
	OrderDate,
	RequiredDate,
	ShippedDate
FROM Orders
WHERE CustomerID = AtCustomerID
ORDER BY OrderID;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "Employee Sales by Country"                              #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `Employee Sales by Country`;

DELIMITER $$

CREATE PROCEDURE `Employee Sales by Country`(in AtBeginning_Date Datetime,in AtEnding_Date Datetime)
BEGIN
  SELECT Employees.Country,
         Employees.LastName,
         Employees.FirstName,
            Orders.ShippedDate,
            Orders.OrderID,
 `Order Subtotals`.Subtotal AS SaleAmount
FROM Employees
 JOIN Orders ON Employees.EmployeeID = Orders.EmployeeID
      JOIN `Order Subtotals` ON Orders.OrderID = `Order Subtotals`.OrderID
WHERE Orders.ShippedDate Between AtBeginning_Date And AtEnding_Date;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "Employee Sales by Year"                                 #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `Sales by Year`;

DELIMITER $$

CREATE PROCEDURE `Sales by Year`(in AtBeginning_Date Datetime,in AtEnding_Date Datetime)
BEGIN

    SELECT Orders.ShippedDate,
	   Orders.OrderID,
	  `Order Subtotals`.Subtotal,
	  ShippedDate AS Year
FROM Orders  JOIN `Order Subtotals` ON Orders.OrderID = `Order Subtotals`.OrderID
WHERE Orders.ShippedDate Between AtBeginning_Date And AtEnding_Date;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "SalesByCategory"                                        #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `SalesByCategory`;
DELIMITER $$

CREATE PROCEDURE `SalesByCategory`()
BEGIN

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "sp_Employees_Insert"                                    #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `sp_Employees_Insert`;

DELIMITER $$

CREATE PROCEDURE `sp_Employees_Insert`(
In AtLastName VARCHAR(20),
In AtFirstName VARCHAR(10),
In AtTitle VARCHAR(30),
In AtTitleOfCourtesy VARCHAR(25),
In AtBirthDate DateTime,
In AtHireDate DateTime,
In AtAddress VARCHAR(60),
In AtCity VARCHAR(15),
In AtRegion VARCHAR(15),
In AtPostalCode VARCHAR(10),
In AtCountry VARCHAR(15),
In AtHomePhone VARCHAR(24),
In AtExtension VARCHAR(4),
In AtPhoto LONGBLOB,
In AtNotes MEDIUMTEXT,
In AtReportsTo INTEGER,
IN AtPhotoPath VARCHAR(255),
OUT AtReturnID INTEGER
)
BEGIN
Insert Into Employees Values(AtLastName,AtFirstName,AtTitle,AtTitleOfCourtesy,AtBirthDate,AtHireDate,AtAddress,AtCity,AtRegion,AtPostalCode,AtCountry,AtHomePhone,AtExtension,AtPhoto,AtNotes,AtReportsTo,AtPhotoPath);

	SELECT AtReturnID = LAST_INSERT_ID();
	
END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "sp_Employees_SelectAll"                                 #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `sp_Employees_SelectAll`;

DELIMITER $$

CREATE PROCEDURE `sp_Employees_SelectAll`()
BEGIN
SELECT * FROM Employees;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "sp_Employees_SelectRow"                                 #
# ---------------------------------------------------------------------- #


DROP PROCEDURE IF EXISTS `sp_Employees_SelectRow`;

DELIMITER $$

CREATE PROCEDURE `sp_Employees_SelectRow`(In AtEmployeeID INTEGER)
BEGIN
SELECT * FROM Employees Where EmployeeID = AtEmployeeID;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add Procedure "sp_Employees_Update"                                    #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `sp_Employees_Update`;

DELIMITER $$

CREATE PROCEDURE `sp_Employees_Update`(
In AtEmployeeID INTEGER,
In AtLastName VARCHAR(20),
In AtFirstName VARCHAR(10),
In AtTitle VARCHAR(30),
In AtTitleOfCourtesy VARCHAR(25),
In AtBirthDate DateTime,
In AtHireDate DateTime,
In AtAddress VARCHAR(60),
In AtCity VARCHAR(15),
In AtRegion VARCHAR(15),
In AtPostalCode VARCHAR(10),
In AtCountry VARCHAR(15),
In AtHomePhone VARCHAR(24),
In AtExtension VARCHAR(4),
In AtPhoto LONGBLOB,
In AtNotes MEDIUMTEXT,
In AtReportsTo INTEGER,
IN AtPhotoPath VARCHAR(255)
)
BEGIN
Update Employees
	Set
		LastName = AtLastName,
		FirstName = AtFirstName,
		Title = AtTitle,
		TitleOfCourtesy = AtTitleOfCourtesy,
		BirthDate = AtBirthDate,
		HireDate = AtHireDate,
		Address = AtAddress,
		City = AtCity,
		Region = AtRegion,
		PostalCode = AtPostalCode,
		Country = AtCountry,
		HomePhone = AtHomePhone,
		Extension = AtExtension,
		Photo = AtPhoto,
		Notes = AtNotes,
		ReportsTo = AtReportsTo,
    PhotoPath = AtPhotoPath
	Where
		EmployeeID = AtEmployeeID;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add FUNCTION "MyRound"                                                 #
# ---------------------------------------------------------------------- #

DROP FUNCTION IF EXISTS `MyRound`;

DELIMITER $$

CREATE FUNCTION `MyRound`(Operand DOUBLE,Places INTEGER) RETURNS DOUBLE
DETERMINISTIC
BEGIN

DECLARE x DOUBLE;
DECLARE i INTEGER;
DECLARE ix DOUBLE;

  SET x = Operand*POW(10,Places);
  SET i=x;
  
  IF (i-x) >= 0.5 THEN                   
    SET ix = 1;                  
  ELSE
    SET ix = 0;                 
  END IF;     

  SET x=i+ix;
  SET x=x/POW(10,Places);

RETURN x;


END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add PROCEDURE "LookByFName"                                            #
# ---------------------------------------------------------------------- #

DROP PROCEDURE IF EXISTS `LookByFName`;

DELIMITER $$

CREATE PROCEDURE `LookByFName`(IN AtFirstLetter CHAR(1))
BEGIN
     SELECT * FROM Employees  Where LEFT(FirstName, 1)=AtFirstLetter;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add FUNCTION "DateOnly"                                                #
# ---------------------------------------------------------------------- #

DELIMITER $$

DROP FUNCTION IF EXISTS `DateOnly` $$

CREATE FUNCTION `DateOnly` (InDateTime datetime) RETURNS VARCHAR(10)
BEGIN

  DECLARE MyOutput varchar(10);
	SET MyOutput = DATE_FORMAT(InDateTime,'%Y-%m-%d');

  RETURN MyOutput;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add PROCEDURE  "sp_employees_cursor" CURSOR SAMPLE                     #
# ---------------------------------------------------------------------- #
DELIMITER $$

DROP PROCEDURE IF EXISTS `sp_employees_cursor` $$
CREATE PROCEDURE `sp_employees_cursor`(IN city_in VARCHAR(15))
BEGIN
  DECLARE name_val VARCHAR(10);
  DECLARE surname_val VARCHAR(10);
  DECLARE photopath_val VARCHAR(255);

  DECLARE no_more_rows BOOLEAN;

  DECLARE fetch_status INT DEFAULT 0;

  DECLARE employees_cur CURSOR FOR SELECT firstname, lastname,photopath FROM employees WHERE city = city_in;

  DECLARE CONTINUE HANDLER FOR NOT FOUND SET no_more_rows = TRUE;

  DROP TABLE IF EXISTS atpeople;
  CREATE TABLE atpeople(
    FirstName VARCHAR(10),
    LastName VARCHAR(20),
    PhotoPath VARCHAR(255)
  );


  OPEN employees_cur;
  select FOUND_ROWS() into fetch_status;


  the_loop: LOOP

    FETCH  employees_cur  INTO   name_val,surname_val,photopath_val;


    IF no_more_rows THEN
       CLOSE employees_cur;
       LEAVE the_loop;
    END IF;


    INSERT INTO atpeople SELECT  name_val,surname_val,photopath_val;

  END LOOP the_loop;

  SELECT * FROM atpeople;
  DROP TABLE atpeople;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add PROCEDURE  "sp_employees_rownum" rownum SAMPLE                     #
# ---------------------------------------------------------------------- #
  
DELIMITER $$

DROP PROCEDURE IF EXISTS `sp_employees_rownum`$$
CREATE PROCEDURE `sp_employees_rownum` ()
BEGIN

SELECT *
FROM
(select @rownum:=@rownum+1  as RowNum,
  p.* from employees p
   ,(SELECT @rownum:=0) R
   order by firstname desc limit 10
) a
WHERE a.RowNum >= 2 AND a.RowNum<= 4;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add PROCEDURE  "sp_employees_rollup" rollup SAMPLE                     #
# ---------------------------------------------------------------------- #
  
DELIMITER $$

DROP PROCEDURE IF EXISTS `sp_employees_rollup`$$
CREATE PROCEDURE `sp_employees_rollup` ()
BEGIN
SELECT Distinct City ,Sum(Salary) Salary_By_City FROM employees
GROUP BY City WITH ROLLUP;

END $$

DELIMITER ;

# ---------------------------------------------------------------------- #
# Add PROCEDURE  "sp_employees_rank" rank SAMPLE                         #
# ---------------------------------------------------------------------- #

DELIMITER $$

DROP PROCEDURE IF EXISTS `northwind`.`sp_employees_rank` $$
CREATE PROCEDURE `northwind`.`sp_employees_rank` ()
BEGIN
select *
     from (select a.Title, a.EmployeeID, a.FirstName, a.Salary,
                  (select 1 + count(*)
                   from Employees b
                   where b.Title = a.Title
                     and b.Salary > a.Salary) RANK
           from Employees as a) as x
     order by x.Title, x.RANK;

END $$

DELIMITER ;